/* ***************************************************************************************
* FILE:          AnimationCloner.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  AnimationCloner is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"

#include "AnimationCloner.h"

#include <Candera/EngineBase/Animation/AnimationPlayer.h>
#include <Candera/System/MetaInfo/InternalMacros.h>

#include "CloneProperties.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/AnimationCloner.cpp.trc.h"
#endif
#include <Candera/System/MetaInfo/PropertyMetaInfoAnimationPropertySetters.h>
#include <Candera/System/MemoryManagement/CanderaHeap.h>

using namespace Candera;
using namespace Candera::Animation;


Candera::Animation::AnimationController::SharedPointer AnimationCloner::CloneAnimationController(const Candera::Animation::AnimationPlayer* templateAnimation,
      ControlTemplateInstancePtr controlTemplateInstance,
      const ControlTemplateMap& controlTemplateMap)
{
   return Clone(templateAnimation, controlTemplateInstance, controlTemplateMap, false);
}


Candera::Animation::AnimationController::SharedPointer AnimationCloner::DuplicateAnimationController(const Candera::Animation::AnimationPlayer* templateAnimation)
{
   DefaultControlTemplateMap dummyMap;
   ControlTemplateInstancePtr dummyCTI;
   return Clone(templateAnimation, dummyCTI, dummyMap, true);
}


Candera::Animation::AnimationController::SharedPointer AnimationCloner::Clone(const Candera::Animation::AnimationPlayer* templateAnimation, ControlTemplateInstancePtr controlTemplateInstance, const ControlTemplateMap& controlTemplateMap, bool acceptOriginalWidget)
{
   Candera::Animation::AnimationController::SharedPointer controllerClone;

   if ((0 == templateAnimation) || templateAnimation->GetController().PointsToNull())
   {
      return controllerClone;
   }

   controllerClone = Candera::Animation::AnimationController::Create();
   if (controllerClone.PointsToNull())
   {
      ETG_TRACE_ERR(("AnimationController could not be allocated!"));
      return controllerClone;
   }

   Animation::AnimationController::SharedPointer templateController = templateAnimation->GetController();
   if (templateController.PointsToNull())
   {
      ETG_TRACE_ERR(("AnimationController template is NULL!"));
      return controllerClone;
   }

   PropertiesClonners::CloneProperties(*templateController, *(controllerClone.GetPointerToSharedInstance()));

   for (SizeType i = 0; i < templateController->GetNumberOfProperties(); i++)
   {
      Animation::AnimationBlendedProperty::SharedPointer animationPropertyTemplate = templateController->GetProperty(i);
      if (!animationPropertyTemplate.PointsToNull())
      {
         Animation::AnimationBlendedProperty::SharedPointer animationPropertyClone = Animation::AnimationBlendedProperty::Create();
         if (!animationPropertyClone.PointsToNull())
         {
            CloneKeyframeSequences(*(animationPropertyTemplate.GetPointerToSharedInstance()), *(animationPropertyClone.GetPointerToSharedInstance()));
            if (CloneAnimationPropertySetter(*(animationPropertyTemplate.GetPointerToSharedInstance()),
                                             *(animationPropertyClone.GetPointerToSharedInstance()),
                                             controlTemplateInstance,
                                             controlTemplateMap, acceptOriginalWidget))
            {
               controllerClone->AddProperty(animationPropertyClone);
            }
         }
      }
   }

   return (controllerClone->GetNumberOfProperties() > 0) ? controllerClone : Candera::Animation::AnimationController::SharedPointer();
}


void AnimationCloner::CloneKeyframeSequences(const Candera::Animation::AnimationBlendedProperty& animPropertyTemplate, Candera::Animation::AnimationBlendedProperty& animPropertyClone)
{
   for (SizeType i = 0; i < animPropertyTemplate.GetNumberOfKeyframeSequences(); i++)
   {
      SharedPointer<AnimationKeyframeSequence> kfs(animPropertyTemplate.GetKeyframeSequence(i));
      SharedPointer<AnimationKeyframeSequence> clonedKfs(AnimationKeyframeSequence::Create());

      CloneKeyFrameSequence(kfs, clonedKfs);

      animPropertyClone.AddKeyframeSequence(clonedKfs);
   }
}


void AnimationCloner::CloneKeyFrameSequence(const SharedPointer<AnimationKeyframeSequence>& kfs, const SharedPointer<AnimationKeyframeSequence>& clonedKfs)
{
   clonedKfs->SetInterpolationStrategy(kfs->GetInterpolationStrategy());
   clonedKfs->SetPeriodic(kfs->IsPeriodic());

   Int32 kfCount(kfs->GetKeyframeCount());
   Int32 numberOfComponents(kfs->GetNumberOfComponents());

   //clone keyframe sequence time and values
   SequenceTimeType* clonedKft(FEATSTD_NEW_ARRAY(SequenceTimeType, kfCount));
   Float* clonedKfv(FEATSTD_NEW_ARRAY(Float, kfCount * numberOfComponents));

   MemoryPlatform::Copy(clonedKft, kfs->GetKeyframeSequenceTimes(), static_cast<SizeType>(kfCount) * sizeof(SequenceTimeType));
   MemoryPlatform::Copy(clonedKfv, kfs->GetKeyframeValues(), static_cast<SizeType>(kfCount * numberOfComponents) * sizeof(Float));

   clonedKfs->SetKeyframes(numberOfComponents, kfCount,
                           clonedKft, Animation::KeyframeSequence::TimeTypeDisposer::Dispose,
                           clonedKfv, Animation::KeyframeSequence::ValuesDisposer::Dispose);
}


bool AnimationCloner::CloneAnimationPropertySetter(const Candera::Animation::AnimationBlendedProperty& animPropertyTemplate,
      Candera::Animation::AnimationBlendedProperty& animPropertyClone,
      ControlTemplateInstancePtr controlTemplateInstance,
      const ControlTemplateMap& controlTemplateMap, bool acceptOriginalWidget)
{
   bool rc = false;
   Animation::AnimationPropertySetter* animPropertySetterTemplate = animPropertyTemplate.GetAnimationPropertySetter().GetPointerToSharedInstance();

   if (0 != animPropertySetterTemplate)
   {
      rc = rc ||
           SetClonedTransformablePropertySetter<Animation::Transformable2DTranslatePropertySetter>(animPropertySetterTemplate, animPropertyClone, controlTemplateMap) ||
           SetClonedTransformablePropertySetter<Animation::Transformable2DTranslateXPropertySetter>(animPropertySetterTemplate, animPropertyClone, controlTemplateMap) ||
           SetClonedTransformablePropertySetter<Animation::Transformable2DTranslateYPropertySetter>(animPropertySetterTemplate, animPropertyClone, controlTemplateMap) ||
           SetClonedTransformablePropertySetter<Animation::Transformable2DRotatePropertySetter>(animPropertySetterTemplate, animPropertyClone, controlTemplateMap) ||
           SetClonedTransformablePropertySetter<Animation::Transformable2DScalePropertySetter>(animPropertySetterTemplate, animPropertyClone, controlTemplateMap) ||
           SetClonedTransformablePropertySetter<Animation::Transformable2DScaleXPropertySetter>(animPropertySetterTemplate, animPropertyClone, controlTemplateMap) ||
           SetClonedTransformablePropertySetter<Animation::Transformable2DScaleYPropertySetter>(animPropertySetterTemplate, animPropertyClone, controlTemplateMap) ||
           SetClonedTransformablePropertySetter<Animation::Transformable2DRelativeTranslatePropertySetter>(animPropertySetterTemplate, animPropertyClone, controlTemplateMap) ||
           SetClonedTransformablePropertySetter<Animation::Transformable2DRelativeTranslateXPropertySetter>(animPropertySetterTemplate, animPropertyClone, controlTemplateMap) ||
           SetClonedTransformablePropertySetter<Animation::Transformable2DRelativeTranslateYPropertySetter>(animPropertySetterTemplate, animPropertyClone, controlTemplateMap) ||
           SetClonedTransformablePropertySetter<Animation::Transformable2DRelativeRotatePropertySetter>(animPropertySetterTemplate, animPropertyClone, controlTemplateMap) ||
           SetClonedTransformablePropertySetter<Animation::Transformable2DRelativeScalePropertySetter>(animPropertySetterTemplate, animPropertyClone, controlTemplateMap) ||
           SetClonedTransformablePropertySetter<Animation::Transformable2DRelativeScaleXPropertySetter>(animPropertySetterTemplate, animPropertyClone, controlTemplateMap) ||
           SetClonedTransformablePropertySetter<Animation::Transformable2DRelativeScaleYPropertySetter>(animPropertySetterTemplate, animPropertyClone, controlTemplateMap) ||
           SetClonedAbstractNodePropertySetter<Animation::LayoutMarginsPropertySetter>(animPropertySetterTemplate, animPropertyClone, controlTemplateMap) ||
           SetClonedAbstractNodePropertySetter<Animation::LayoutMarginsLeftPropertySetter>(animPropertySetterTemplate, animPropertyClone, controlTemplateMap) ||
           SetClonedAbstractNodePropertySetter<Animation::LayoutMarginsTopPropertySetter>(animPropertySetterTemplate, animPropertyClone, controlTemplateMap) ||
           SetClonedAbstractNodePropertySetter<Animation::LayoutMarginsRightPropertySetter>(animPropertySetterTemplate, animPropertyClone, controlTemplateMap) ||
           SetClonedAbstractNodePropertySetter<Animation::LayoutMarginsBottomPropertySetter>(animPropertySetterTemplate, animPropertyClone, controlTemplateMap) ||
           SetClonedNodePropertySetter<Animation::RenderingEnabledNode2DPropertySetter>(animPropertySetterTemplate, animPropertyClone, controlTemplateMap) ||
           SetClonedNodePropertySetter<Animation::AlphaNode2DPropertySetter>(animPropertySetterTemplate, animPropertyClone, controlTemplateMap);

      if ((false == rc) && animPropertySetterTemplate->IsTypeOf(Internal::ObjectPropertySetter<Effect2D>::GetTypeId()))
      {
         Internal::ObjectPropertySetter<Effect2D>* propSetter = Dynamic_Cast<Internal::ObjectPropertySetter<Effect2D> *> (animPropertySetterTemplate);
         if (propSetter != 0)
         {
            Candera::Effect2D* effect = propSetter->GetAnimatedObject();
            Candera::Effect2D* effectClone = controlTemplateMap.ResolveEffectClone(effect);
            if (0 != effect && 0 != effectClone)
            {
               MetaInfo::Effect2DMetaInfo* objectMetaInfo = effect->GetMetaInfo();
               MetaInfo::Effect2DPropertyMetaInfo* propMetaInfo = 0;

               if (0 != objectMetaInfo)
               {
                  propMetaInfo = objectMetaInfo->LookupItem(propSetter->GetPropertyName());
               }

               if (0 != propMetaInfo)
               {
                  Int channel[4] = { 0, 1, 2, 3 };
                  Animation::AnimationPropertySetter::SharedPointer propSetterClone = propMetaInfo->CreateAnimationPropertySetter(effectClone, 4, channel);
                  animPropertyClone.SetAnimationPropertySetter(propSetterClone);
                  rc = true;
               }
            }
         }
      }

      if (false == rc && animPropertySetterTemplate->IsTypeOf(Internal::ObjectPropertySetter<WidgetBase>::GetTypeId()))
      {
         Internal::ObjectPropertySetter<WidgetBase>* propSetter = Dynamic_Cast<Internal::ObjectPropertySetter<WidgetBase>*>(animPropertySetterTemplate);

         if (propSetter != 0)
         {
            WidgetBase* widget = propSetter->GetAnimatedObject();
            WidgetBase* widgetClone = controlTemplateMap.ResolveWidgetClone(widget);
            if (acceptOriginalWidget || (widget != widgetClone))
            {
               if (0 != widget && 0 != widgetClone)
               {
                  if (!controlTemplateInstance.PointsToNull())
                  {
                     controlTemplateInstance->AddWidgetToUpdateAfterScrollAnimations(widgetClone);
                  }
                  MetaInfo::WidgetMetaInfo* objectMetaInfo = widget->GetMetaInfo();
                  MetaInfo::WidgetPropertyMetaInfo* propMetaInfo = 0;
                  if (0 != objectMetaInfo)
                  {
                     propMetaInfo = objectMetaInfo->LookupItem(propSetter->GetPropertyName());
                  }

                  if (0 != propMetaInfo)
                  {
                     Int channel[4] = { 0, 1, 2, 3 };
                     Animation::AnimationPropertySetter::SharedPointer propSetterClone = propMetaInfo->CreateAnimationPropertySetter(widgetClone, 4, channel);

                     animPropertyClone.SetAnimationPropertySetter(propSetterClone);
                     rc = true;
                  }
               }
            }
         }
      }
   }

   return rc;
}
