/* ***************************************************************************************
* FILE:          ColorBarWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TempWidgets2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"

#include "ColorBarWidget2D.h"
#include "CanderaPlatform/Device/Common/Effects/BitmapBrushBlend.h"
#include "Widgets/2D/ImageEffect/ImageEffectWidget2D.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_IMAGE
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ColorBarWidget2D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(ColorBarWidget2D)

using namespace Candera::MemoryManagement;


#if defined(FEATSTD_STRINGBUFFER_APPENDER_ENABLED)
namespace FeatStd {
template<> ::FeatStd::UInt32 StringBufferAppender< ::tColorBarDataPtr >::Append(::FeatStd::StringBuffer& stringBuffer, tColorBarDataPtr const&)
{
   ::FeatStd::UInt32 tcharCount = 0;
   tcharCount += stringBuffer.Append("{ tColorBarDataPtr }");
   return tcharCount;
}


template<> ::FeatStd::UInt32 FeatStd::StringBufferAppender<Candera::ArrayProperty<Candera::Color> >::Append(class FeatStd::StringBuffer& stringBuffer, class Candera::ArrayProperty<Candera::Color> const&)
{
   ::FeatStd::UInt32 tcharCount = 0;
   tcharCount += stringBuffer.Append("{ Colors:  }");

   return tcharCount;
}


template<> ::FeatStd::UInt32 FeatStd::StringBufferAppender<Candera::ArrayProperty<Candera::UInt32> >::Append(class FeatStd::StringBuffer& stringBuffer, class Candera::ArrayProperty<Candera::UInt32> const&)
{
   ::FeatStd::UInt32 tcharCount = 0;
   tcharCount += stringBuffer.Append("{ Lengths:  }");

   return tcharCount;
}


};
#endif


ColorBarWidget2D::ColorBarWidget2D()
{
   _bInvalid = true;
   _u8Buffer = NULL;
   _u32BuffSize = 0;
}


ColorBarWidget2D::~ColorBarWidget2D()
{
   if (_bitmapPtr.PointsToNull())
   {
      if (_u8Buffer != NULL)
      {
         FEATSTD_DELETE_ARRAY(_u8Buffer);
      }
   }
   _u8Buffer = NULL; // Don't need delete it, as the bitmap will delete this buffer when it's disposed.
}


void ColorBarWidget2D::InitWidget()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ColorBarWidget2D: InitWidget [%s]", GetLegacyName()));

   Base::InitWidget();
}


bool ColorBarWidget2D::OnMessage(const Courier::Message& msg)
{
   bool bConsumed = false;

   if (msg.GetId() == ColorBarDataUpdMsg::ID)
   {
      const ColorBarDataUpdMsg* pDataResMsg = Courier::message_cast<const ColorBarDataUpdMsg*>(&msg);
      if ((pDataResMsg != NULL) && (pDataResMsg->GetId() == GetId()))
      {
         if (!pDataResMsg->GetData().PointsToNull())
         {
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ColorBarWidget2D: ColorBarDataResMsg Size=%d. [%20s]",
                                pDataResMsg->GetData()->GetCount(), GetLegacyName()));

            ColorsType Colors;
            LengthsType Lengths;
            const Candera::UInt32 size = pDataResMsg->GetData()->GetCount();
            Colors.Reserve(size);
            Lengths.Reserve(size);
            for (Candera::UInt32 idx = 0; idx < size; ++idx)
            {
               tColorBarSegment& item = pDataResMsg->GetData()->Get(idx);
               Colors.Add(item.color);
               Lengths.Add(item.length);
            }

            SetColors(Colors);
            SetLengths(Lengths);

            // _bInvalid = true;
            // triggerUpdate();
         }
         else
         {
            ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "ColorBarWidget2D: ColorBarDataResMsg Invalid Array Pointer. [%20s]", GetLegacyName()));
         }

         bConsumed = true;
      }
   }
   else
   {
      bConsumed = Base::OnMessage(msg);
   }

   return bConsumed;
}


void ColorBarWidget2D::OnChanged(Candera::UInt32 propertyId)
{
   Base::OnChanged(propertyId);

   if (propertyId == ColorsPropertyId || propertyId == LengthsPropertyId)
   {
      _bInvalid = true;
   }
}


bool ColorBarWidget2D::updateBarImage()
{
   const Candera::UInt32 itemCount = getItemCount();
   if (itemCount == 0)
   {
      ETG_TRACE_USR3_DCL((APP_TRACECLASS_ID(), "ColorBarWidget2D: createBarImage-> Colors/Lengths is empty. [%50s]", GetLegacyName()));
      return false;
   }

   const int length = getLength();
   if (length <= 0)
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "ColorBarWidget2D: createBarImage-> Bar len is less than 0. [%50s]", GetLegacyName()));
      return false;
   }

   const Candera::UInt32 bufferSize = length * 4; // RGBA format
   bool bLenChanged = false;
   if ((_u32BuffSize != 0) && (_u32BuffSize != bufferSize))
   {
      // widget size changes
      bLenChanged = true;
      _u8Buffer = NULL; // the original memory will be delete in Bitmap obj
   }

   if (_u8Buffer == NULL)
   {
      _u8Buffer = CANDERA_NEW_ARRAY(FeatStd::UInt8, bufferSize); // Create the buffer
      _u32BuffSize = bufferSize;
   }

   Candera::UInt16 bmpWidth = 0;
   Candera::UInt16 bmpHeight = 0;

   if (GetOrientation() == Candera::VerticalBar)
   {
      int pos = bufferSize;
      for (Candera::UInt32 item = 0; item < itemCount; ++item)
      {
         const Candera::Color& color = GetColors().Get(item);
         const Candera::UInt8 R = static_cast<Candera::UInt8>(color.GetRed()   * 255.0f); // R
         const Candera::UInt8 G = static_cast<Candera::UInt8>(color.GetGreen() * 255.0f); // G
         const Candera::UInt8 B = static_cast<Candera::UInt8>(color.GetBlue()  * 255.0f); // B
         const Candera::UInt8 A = static_cast<Candera::UInt8>(color.GetAlpha() * 255.0f); // A

         const int itemLen = static_cast<int>(GetLengths().Get(item));
         pos -= (itemLen * 4);
         if (pos < 0)
         {
            break;
         }
         int fillRow = pos;
         for (int row = 0; row < itemLen; ++row)
         {
            _u8Buffer[fillRow++] = R;
            _u8Buffer[fillRow++] = G;
            _u8Buffer[fillRow++] = B;
            _u8Buffer[fillRow++] = A;
         }
      }
      //! Fill the unused zone of bitmap
      if (pos > 0)
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "ColorBarWidget2D: updateBitmap-> VERT pos should not larger then 0. [%50s]", GetLegacyName()));
      }
      bmpWidth = 1;
      bmpHeight = static_cast<Candera::UInt16>(length);
   }
   else if (GetOrientation() == Candera::HorizontalBar)
   {
      Candera::UInt32 pos = 0;
      bool isBufFull = false;
      for (Candera::UInt32 item = 0; (item < itemCount) && (!isBufFull); ++item)
      {
         const Candera::Color& color = GetColors().Get(item);
         const Candera::UInt8 R = static_cast<Candera::UInt8>(color.GetRed()   * 255.0f); // R
         const Candera::UInt8 G = static_cast<Candera::UInt8>(color.GetGreen() * 255.0f); // G
         const Candera::UInt8 B = static_cast<Candera::UInt8>(color.GetBlue()  * 255.0f); // B
         const Candera::UInt8 A = static_cast<Candera::UInt8>(color.GetAlpha() * 255.0f); // A

         Candera::UInt32 itemLen = GetLengths().Get(item);
         for (Candera::UInt32 col = 0; col < itemLen; ++col)
         {
            if (pos + 4 <= bufferSize)
            {
               _u8Buffer[pos++] = R;
               _u8Buffer[pos++] = G;
               _u8Buffer[pos++] = B;
               _u8Buffer[pos++] = A;
            }
            else
            {
               isBufFull = true;
               break;
            }
         }
      }

      //! Fill the unused zone of bitmap
      if (pos < bufferSize)
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "ColorBarWidget2D: updateBitmap-> HORI pos should not larger then buffersize. [%50s]", GetLegacyName()));
      }

      bmpWidth = static_cast<Candera::UInt16>(length);
      bmpHeight = 1;
   }
   else
   {
      ETG_TRACE_USR3_DCL((APP_TRACECLASS_ID(), "ColorBarWidget2D: updateBitmap-> Invalid Orientation [%50s]", GetLegacyName()));
   }

   if (_bitmapPtr.PointsToNull() || bLenChanged)
   {
      _bitmapPtr = Candera::Bitmap::Create(bmpWidth, bmpHeight,
                                           Candera::Bitmap::RgbaUnsignedBytePixelFormat,
                                           Candera::Bitmap::PackAlignment4,
                                           _u8Buffer,
                                           Candera::Bitmap::Disposer::Dispose);
   }

   if (_image2dPtr.PointsToNull())
   {
      _image2dPtr = Candera::BitmapImage2D::Create();
      _image2dPtr->SetBitmap(_bitmapPtr);
      ImageEffectWidget2D::SetImage(*this, _image2dPtr);
   }
   else if (bLenChanged)
   {
      _image2dPtr->Unload();
      _image2dPtr->SetBitmap(_bitmapPtr);
      _image2dPtr->Upload();
   }
   else
   {
      _image2dPtr->Update(0, 0, bmpWidth, bmpHeight, _u8Buffer);
   }

   return true;
}


void ColorBarWidget2D::Update()
{
   Base::Update();

   if (GetNode() != NULL)
   {
      if (_bInvalid)
      {
         _bInvalid = false;
         updateBarImage();
         Invalidate();
      }
   }
   else
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "ColorBarWidget2D: Update-> GetNode is NULL. [%20s]", GetLegacyName()));
   }
}


Candera::UInt32 ColorBarWidget2D::getLength() const
{
   // Get the length property, if it's value is larger then 0, use it, otherwise, get the size of node
   Candera::UInt32 len = 0;
   const Candera::UInt32 itemCount = getItemCount();
   for (Candera::UInt32 item = 0; item < itemCount; ++item)
   {
      len += GetLengths().Get(item);
   }
   return len;
}


Candera::UInt32 ColorBarWidget2D::getItemCount() const
{
   return (GetColors().GetCount() > GetLengths().GetCount()) ? static_cast<Candera::UInt32>(GetLengths().GetCount()) : static_cast<Candera::UInt32>(GetColors().GetCount());
}


bool ColorBarWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   bool cloned = false;
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const ColorBarWidget2D* original = CLONEABLE_WIDGET_CAST<const ColorBarWidget2D*>(originalWidget);
      if (original == NULL)
      {
         return false;
      }

      SetOrientation(original->GetOrientation());
      SetId(original->GetId());
      SetColors(original->GetColors());
      SetLengths(original->GetLengths());
      // Copy bitmap
      _u32BuffSize = original->_u32BuffSize;
      if (_u32BuffSize != 0 && (original->_u8Buffer != NULL) && !original->_bitmapPtr.PointsToNull() && !original->_image2dPtr.PointsToNull())
      {
         _u8Buffer = CANDERA_NEW_ARRAY(FeatStd::UInt8, _u32BuffSize);  // Create new Buffer directly, the original memory will be deleted by Image2D

         memcpy((void*)_u8Buffer, (void*)original->_u8Buffer, _u32BuffSize);

         _bitmapPtr = Candera::Bitmap::Create(
                         original->_bitmapPtr->GetWidth(),
                         original->_bitmapPtr->GetHeight(),
                         Candera::Bitmap::RgbaUnsignedBytePixelFormat,
                         Candera::Bitmap::PackAlignment4,
                         _u8Buffer,
                         Candera::Bitmap::Disposer::Dispose);

         if (_image2dPtr.PointsToNull())
         {
            _image2dPtr = Candera::BitmapImage2D::Create();
            _image2dPtr->SetBitmap(_bitmapPtr);
         }
         else
         {
            _image2dPtr->Unload();
            _image2dPtr->SetBitmap(_bitmapPtr);
            _image2dPtr->Upload();
         }
      }

      cloned = true;
   }
   return cloned;
}
