/* ***************************************************************************************
* FILE:          ButtonImageWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ButtonImageWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "CanderaPlatform/Device/Common/Effects/BitmapBrushBlend.h"
#include "Widgets/2D/Button/ButtonWidget2D.h"
#include <Widgets/2D/Button/ButtonAppearance2D.h>
#include "Widgets/2D/ImageEffect/ImageEffectWidget2D.h"
#include "ButtonImageWidget2D.h"
#include <Focus/FManager.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_APPEARANCE
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ButtonImageWidget2D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(ButtonImageWidget2D);

using namespace hmibase::widget::appearance;

ButtonImageWidget2D::ButtonImageWidget2D(): Base()
{
}


ButtonImageWidget2D::~ButtonImageWidget2D()
{
}


static Candera::MemoryManagement::SharedPointer<Candera::Image2D> getButtonStateBitmap(const ButtonImageWidget2D& buttonImage, const AppearanceState& state)
{
   switch (state.asEnum())
   {
      case AppearanceState::enNormal:
         return buttonImage.GetNormalBitmap();
      case AppearanceState::enPressed:
         return buttonImage.GetPressedBitmap();
      case AppearanceState::enDisabled:
         return buttonImage.GetDisabledBitmap();
      case AppearanceState::enPressedDisabled:
         return buttonImage.GetPressedDisabledBitmap();

      case AppearanceState::enActiveNormal:
         return buttonImage.GetActiveNormalBitmap();
      case AppearanceState::enActivePressed:
         return buttonImage.GetActivePressedBitmap();
      case AppearanceState::enActiveDisabled:
         return buttonImage.GetActiveDisabledBitmap();
      case AppearanceState::enActivePressedDisabled:
         return buttonImage.GetActivePressedDisabledBitmap();

      case AppearanceState::enFocusedNormal:
         return buttonImage.GetFocusedNormalBitmap();
      case AppearanceState::enFocusedPressed:
         return buttonImage.GetFocusedPressedBitmap();
      case AppearanceState::enFocusedDisabled:
         return buttonImage.GetFocusedDisabledBitmap();
      case AppearanceState::enFocusedPressedDisabled:
         return buttonImage.GetFocusedPressedDisabledBitmap();

      case AppearanceState::enFocusedActiveNormal:
         return buttonImage.GetFocusedActiveNormalBitmap();
      case AppearanceState::enFocusedActivePressed:
         return buttonImage.GetFocusedActivePressedBitmap();
      case AppearanceState::enFocusedActiveDisabled:
         return buttonImage.GetFocusedActiveDisabledBitmap();
      case AppearanceState::enFocusedActivePressedDisabled:
         return buttonImage.GetFocusedActivePressedDisabledBitmap();

      default:
         break;
   }

   return Candera::MemoryManagement::SharedPointer<Candera::Image2D>();
}


static AppearanceState getButtonStateFallback(AppearanceState state)
{
   if (state.isFocused())
   {
      state.setFocused(false);
   }
   else if (state.isActive())
   {
      state.setActive(false);
   }
   else if (state.isPressed())
   {
      state.setPressed(false);
   }
   else if (!state.isEnabled())
   {
      state.setEnabled(true);
   }
   else
   {
      //empty
   }
   return state;
}


Candera::MemoryManagement::SharedPointer<Candera::Image2D> ButtonImageWidget2D::GetButtonStateBitmap() const
{
   if (GetButtonWidget() != NULL)
   {
      AppearanceState state(GetButtonState());

      Candera::MemoryManagement::SharedPointer<Candera::Image2D> bitmap = getButtonStateBitmap(*this, state);
      while (GetFallbackStrategyEnabled() && bitmap.PointsToNull())
      {
         AppearanceState fallbackState = getButtonStateFallback(state);
         if (fallbackState == state)
         {
            break;
         }

         state = fallbackState;
         bitmap = getButtonStateBitmap(*this, state);
      };
      return bitmap;
   }

   return Candera::MemoryManagement::SharedPointer<Candera::Image2D>();
}


bool ButtonImageWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   bool cloned(false);
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const ButtonImageWidget2D* original = CLONEABLE_WIDGET_CAST<const ButtonImageWidget2D*>(originalWidget);
      if (original == NULL)
      {
         return false;
      }

      SetNormalBitmap(original->GetNormalBitmap());
      SetPressedBitmap(original->GetPressedBitmap());
      SetDisabledBitmap(original->GetDisabledBitmap());
      SetPressedDisabledBitmap(original->GetPressedDisabledBitmap());

      SetActiveNormalBitmap(original->GetActiveNormalBitmap());
      SetActivePressedBitmap(original->GetActivePressedBitmap());
      SetActiveDisabledBitmap(original->GetActiveDisabledBitmap());
      SetActivePressedDisabledBitmap(original->GetActivePressedDisabledBitmap());

      SetFocusedNormalBitmap(original->GetFocusedNormalBitmap());
      SetFocusedPressedBitmap(original->GetFocusedPressedBitmap());
      SetFocusedDisabledBitmap(original->GetFocusedDisabledBitmap());
      SetFocusedPressedDisabledBitmap(original->GetFocusedPressedDisabledBitmap());

      SetFocusedActiveNormalBitmap(original->GetFocusedActiveNormalBitmap());
      SetFocusedActivePressedBitmap(original->GetFocusedActivePressedBitmap());
      SetFocusedActiveDisabledBitmap(original->GetFocusedActiveDisabledBitmap());
      SetFocusedActivePressedDisabledBitmap(original->GetFocusedActivePressedDisabledBitmap());

      hmibase::widget::utils::EffectUtils::copyColor(original->GetNode(), GetNode());

      cloned = true;
   }
   return cloned;
}


void ButtonImageWidget2D::OnChanged(::FeatStd::UInt32 propertyId)
{
   Base::OnChanged(propertyId);

   switch (propertyId)
   {
      case FallbackStrategyEnabledPropertyId:
      case NormalBitmapPropertyId:
      case PressedBitmapPropertyId:
      case DisabledBitmapPropertyId:
      case PressedDisabledBitmapPropertyId:
      case ActiveNormalBitmapPropertyId:
      case ActivePressedBitmapPropertyId:
      case ActiveDisabledBitmapPropertyId:
      case ActivePressedDisabledBitmapPropertyId:
      case FocusedNormalBitmapPropertyId:
      case FocusedPressedBitmapPropertyId:
      case FocusedDisabledBitmapPropertyId:
      case FocusedPressedDisabledBitmapPropertyId:
      case FocusedActiveNormalBitmapPropertyId:
      case FocusedActivePressedBitmapPropertyId:
      case FocusedActiveDisabledBitmapPropertyId:
      case FocusedActivePressedDisabledBitmapPropertyId:
         SetInvalid(true);
         break;

      default:
         break;
   }
}


void ButtonImageWidget2D::Update()
{
   Base::Update();

   if ((GetNode() != NULL) && (IsInvalid()
                               //we update the image also if the button is focused and there is an animation manager registered
                               || ((Focus::FManager::getInstance().getAnimationManager() != NULL) && (GetButtonWidget() != NULL) && GetButtonWidget()->IsFocused())))
   {
      Candera::MemoryManagement::SharedPointer<Candera::Image2D> newImage = GetButtonStateBitmap();
      ImageEffectWidget2D::SetImage(*this, newImage);

      SetInvalid(false);
   }
}


using namespace hmibase::widget::appearance;
hmibase::widget::appearance::AppearanceTemplate* ButtonImageWidget2D::CreateAppearanceTemplate()
{
   AppearanceImageData::SharedPointer buttonImageData = AppearanceImageData::create(GetNormalBitmap(), GetPressedBitmap(), GetDisabledBitmap(), GetActiveNormalBitmap(), GetActivePressedBitmap(), GetActiveDisabledBitmap(),
         GetFocusedNormalBitmap(), GetFocusedPressedBitmap(), GetFocusedDisabledBitmap(), GetFocusedActiveNormalBitmap(), GetFocusedActivePressedBitmap(), GetFocusedActiveDisabledBitmap());
   return FEATSTD_NEW(ImageAppearanceTemplate)(buttonImageData);
}
