/* ***************************************************************************************
* FILE:          ButtonGroupController2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ButtonGroupController2D.cpp is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "widget2D_std_if.h"
#include "ButtonGroupController2D.h"
#include "ButtonGroupWidget2D.h"

#include <Widgets/2D/Button/ButtonWidget2D.h>
#include <Widgets/utils/WidgetFunctors.h>
#include <Widgets/utils/WidgetTraverser.h>

#include <Trace/ToString.h>
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_BUTTON
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ButtonGroupController2D.cpp.trc.h"
#endif


using ::Candera::Char;//required for CANDERA_RTTI_DEFINITION
CANDERA_RTTI_DEFINITION(ButtonGroupAnimationReqEvent)
CANDERA_RTTI_DEFINITION(ButtonGroupSelectReqEvent)
CANDERA_RTTI_DEFINITION(ButtonGroupController2DData)
CANDERA_RTTI_DEFINITION(ButtonGroupController2D)
CANDERA_RTTI_DEFINITION(DefaultButtonGroupController2DData)
CANDERA_RTTI_DEFINITION(DefaultButtonGroupController2D)


/*****************************************************************************/
bool ButtonGroupController2D::OnMessage(DelegateWidget& widget, const Courier::Message& msg)
{
   bool consumed = false;

   if (!widget.IsItemTemplate())
   {
      switch (msg.GetId())
      {
         case ButtonGroupReqMsg::ID:
         {
            const ButtonGroupReqMsg* buttonGroupReqMsg = Courier::message_cast<const ButtonGroupReqMsg*>(&msg);
            if (buttonGroupReqMsg != NULL)
            {
               consumed = OnButtonGroupReqMsg(widget, *buttonGroupReqMsg);
            }
            break;
         }

         default:
            break;
      }
   }
   return consumed;
}


/*****************************************************************************/
bool ButtonGroupController2D::OnEvent(DelegateWidget& widget, const hmibase::widget::WidgetControllerEvent& e)
{
   bool consumed = false;

   if (!widget.IsItemTemplate())
   {
      ButtonGroupWidget2D* buttonGroup = Candera::Dynamic_Cast<ButtonGroupWidget2D*>(&widget);
      const ButtonGroupSelectReqEvent* selectReq = Candera::Dynamic_Cast<const ButtonGroupSelectReqEvent*>(&e);
      if ((buttonGroup != NULL) && (selectReq != NULL))
      {
         consumed = OnButtonGroupSelectReqEvent(*buttonGroup, *selectReq);
      }
   }
   return consumed;
}


/*****************************************************************************/
class AncestorButtonGroupFinderCallback : public WidgetCheckCallback
{
   public:
      AncestorButtonGroupFinderCallback(Courier::Identifier buttonIdentifier) : _buttonIdentifier(buttonIdentifier), _button(NULL), _group(NULL)
      {
      }

      ~AncestorButtonGroupFinderCallback()
      {
         _button = NULL;
         _group = NULL;
      }

      virtual bool CheckWidget(Candera::Widget2D* widget)
      {
         if ((widget == NULL) || (widget->GetNode() == NULL))
         {
            //continue searching through all the widgets
            return false;
         }

         ButtonGroupWidget2D* group = Candera::Dynamic_Cast<ButtonGroupWidget2D*>(widget);
         if (group != NULL)
         {
            if (_button == NULL)
            {
               //keep the group for later search
               _groups.push_back(group);
            }
            else if (hmibase::widget::utils::CompareUtils::areAncestorAndDescendent(group->GetNode(), _button->GetNode()))
            {
               //stop the search
               _group = group;
               _groups.clear();
               return true;
            }
            else
            {
               //nothing to do
            }

            //continue searching through all the widgets
            return false;
         }

         ButtonWidget2D* button = Candera::Dynamic_Cast<ButtonWidget2D*>(widget);
         if ((button != NULL) && (_buttonIdentifier == Courier::Identifier(button->GetLegacyName())))
         {
            _button = button;

            std::vector<ButtonGroupWidget2D*>::iterator it = std::find_if(_groups.begin(), _groups.end(), hmibase::widget::utils::functors::IsAncestorOf<ButtonGroupWidget2D, Candera::Node2D>(_button->GetNode()));
            if (it != _groups.end())
            {
               //stop the search
               _group = *it;
               _groups.clear();
               return true;
            }

            _groups.clear();
         }

         //continue searching through all the widgets
         return false;
      }

      ButtonGroupWidget2D* getGroup() const
      {
         return _group;
      }

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(AncestorButtonGroupFinderCallback);

      Courier::Identifier _buttonIdentifier;
      std::vector<ButtonGroupWidget2D*> _groups;
      ButtonWidget2D* _button;
      ButtonGroupWidget2D* _group;
};


/*****************************************************************************/
class ButtonGroupContentFinderCallback : public WidgetCheckCallback
{
   public:
      ButtonGroupContentFinderCallback(bool useAllButtons, Candera::Node2D* groupNode, std::vector<ButtonWidget2D*>& buttons)
         : _useAllButtons(useAllButtons), _groupNode(groupNode), _buttons(buttons)
      {
      }

      ~ButtonGroupContentFinderCallback()
      {
         _groupNode = NULL;
      }

      virtual bool CheckWidget(Candera::Widget2D* widget)
      {
         ButtonWidget2D* button = Candera::Dynamic_Cast<ButtonWidget2D*>(widget);
         if ((button != NULL) && (button->GetNode() != NULL)
               && (_useAllButtons || button->GetNode()->IsEffectiveRenderingEnabled())
               && hmibase::widget::utils::CompareUtils::areAncestorAndDescendent(_groupNode, button->GetNode()))
         {
            _buttons.push_back(button);
         }

         //continue searching through all the widgets to find the entire content
         return false;
      }

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(ButtonGroupContentFinderCallback);

      bool _useAllButtons;
      Candera::Node2D* _groupNode;
      std::vector<ButtonWidget2D*>& _buttons;
};


/*****************************************************************************/
DefaultButtonGroupController2D::DefaultButtonGroupController2D() : _useAllButtons(true), _activeButtonRenderOrderRank(0)
{
}


/*****************************************************************************/
void DefaultButtonGroupController2D::SetUseAllButtons(bool value)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SetUseAllButtons value=%u controller=%p", value, this));
   _useAllButtons = value;
}


/*****************************************************************************/
void DefaultButtonGroupController2D::SetActiveButtonRenderOrderRank(int value)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SetActiveButtonRenderOrderRank value=%u controller=%p", value, this));
   _activeButtonRenderOrderRank = value;
}


/*****************************************************************************/
bool DefaultButtonGroupController2D::OnButtonGroupReqMsg(DelegateWidget& widget, const ButtonGroupReqMsg& msg)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnButtonGroupReqMsg action=%u buttonIndex=%d buttonId=%15s animate=%u groupId=%15s %s",
                       msg.GetAction(),
                       msg.GetButtonIndex(),
                       HMIBASE_TO_STRING(msg.GetButtonIdentifier()),
                       msg.GetAnimate(),
                       HMIBASE_TO_STRING(msg.GetGroup()),
                       HMIBASE_TO_STRING_VW(&widget)));

   ButtonGroupWidget2D* buttonGroup = NULL;

   //not our view
   if ((widget.GetParentView() == NULL) || ((msg.GetView() != Courier::ViewId()) && (msg.GetView() != widget.GetParentView()->GetId())))
   {
      //nothing to do
   }
   //group id is specified => check if it matches current widget
   else if (msg.GetGroup() != Courier::Identifier())
   {
      if (msg.GetGroup() == Courier::Identifier(widget.GetLegacyName()))
      {
         buttonGroup = Candera::Dynamic_Cast<ButtonGroupWidget2D*>(&widget);
      }
   }
   //button id is specified => find the button first, then the group being its ancestor
   else if (msg.GetButtonIdentifier() != Courier::Identifier())
   {
      AncestorButtonGroupFinderCallback finderCallback(msg.GetButtonIdentifier());
      hmibase::widget::utils::MessageUtils::distribute(hmibase::widget::utils::MessageUtils::getSceneContext(Candera::Dynamic_Cast<Candera::Widget2D*>(&widget)), WidgetCheckReqMsg(&finderCallback));
      buttonGroup = finderCallback.getGroup();
   }
   //can't process the message without an identifier
   else
   {
      //nothing to do
   }

   bool consumed = false;
   if (buttonGroup != NULL)
   {
      ButtonGroupSelectReqEvent reqEvent(msg.GetAction(), msg.GetButtonIdentifier(), msg.GetButtonIndex(), msg.GetAnimate());
      consumed = buttonGroup->OnEvent(reqEvent);
   }

   return consumed;
}


/*****************************************************************************/
bool DefaultButtonGroupController2D::OnButtonGroupSelectReqEvent(ButtonGroupWidget2D& buttonGroup, const ButtonGroupSelectReqEvent& e)
{
   //find buttons which have the specified button group as ancestor (the relation is checked for the associated nodes)
   std::vector<ButtonWidget2D*> buttons;
   if (!findButtons(buttonGroup, buttons))
   {
      return false;
   }

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnButtonGroupSelectReqEvent action=%u index=%d id=%15s animate=%u buttonsSize=%u groupAnimationEnabled=%u group=%s",
                       e.GetAction(),
                       e.GetButtonIndex(),
                       HMIBASE_TO_STRING(e.GetButtonIdentifier()),
                       e.GetAnimate(),
                       buttons.size(),
                       buttonGroup.GetAnimationEnabled(),
                       HMIBASE_TO_STRING_VW(&buttonGroup)));

   ButtonWidget2D* newButton = NULL;
   int newActiveIndex = -1;
   switch (e.GetAction())
   {
      case enSelectIndex:
      {
         if ((e.GetButtonIndex() >= 0) && (e.GetButtonIndex() < static_cast<int>(buttons.size())))
         {
            newActiveIndex = e.GetButtonIndex();
            newButton = buttons.at(e.GetButtonIndex());
         }
      }
      break;

      case enSelectIdentifier:
      {
         std::vector<ButtonWidget2D*>::const_iterator it = std::find_if(buttons.begin(), buttons.end(), hmibase::widget::utils::functors::MatchesId<ButtonWidget2D>(e.GetButtonIdentifier()));
         if (it != buttons.end())
         {
            newActiveIndex = static_cast<int>(it - buttons.begin());
            newButton = *it;
         }
      }
      break;

      case enSelectNextOfIdentifier:
      {
         std::vector<ButtonWidget2D*>::const_iterator it = std::find_if(buttons.begin(), buttons.end(), hmibase::widget::utils::functors::MatchesId<ButtonWidget2D>(e.GetButtonIdentifier()));
         if (it != buttons.end())
         {
            ++it;
            if (it != buttons.end())
            {
               newActiveIndex = static_cast<int>(it - buttons.begin());
               newButton = *it;
            }
         }
      }
      break;

      case enSelectPreviousOfIdentifier:
      {
         std::vector<ButtonWidget2D*>::const_iterator it = std::find_if(buttons.begin(), buttons.end(), hmibase::widget::utils::functors::MatchesId<ButtonWidget2D>(e.GetButtonIdentifier()));
         if ((it != buttons.end()) && (it != buttons.begin()))
         {
            --it;
            newActiveIndex = static_cast<int>(it - buttons.begin());
            newButton = *it;
         }
      }
      break;

      default:
         break;
   }

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnButtonGroupSelectReqEvent newActiveIndex=%d newActiveButton=%s",
                       newActiveIndex, HMIBASE_TO_STRING_VW(newButton)));

   buttonGroup.SetActiveIndex(newActiveIndex);
   DefaultButtonGroupController2DData* buttonGroupData = Candera::Dynamic_Cast<DefaultButtonGroupController2DData*>(buttonGroup.GetControllerData());
   if ((buttonGroupData != NULL) && !buttonGroupData->Group.PointsToNull())
   {
      buttonGroupData->Group->ActiveIndex = newActiveIndex;
      buttonGroupData->Group->ActiveName = (newButton != NULL) ? newButton->GetLegacyName() : "";
   }

   updateButtons(buttonGroup, buttons, newButton, e.GetAnimate());

   return true;
}


/*****************************************************************************/
void DefaultButtonGroupController2D::Update(DelegateWidget& widget)
{
   if (!widget.IsItemTemplate())
   {
      ButtonGroupWidget2D* buttonGroup = Candera::Dynamic_Cast<ButtonGroupWidget2D*>(&widget);
      DefaultButtonGroupController2DData* buttonGroupData = Candera::Dynamic_Cast<DefaultButtonGroupController2DData*>(widget.GetControllerData());
      if ((buttonGroup != NULL) && (buttonGroup->GetNode() != NULL)
            && (buttonGroupData != NULL) && !buttonGroupData->Group.PointsToNull()
            && (buttonGroup->GetActiveIndex() != buttonGroupData->Group->ActiveIndex))
      {
         ButtonGroupSelectReqEvent reqEvent(enSelectIndex, Courier::Identifier(), buttonGroup->GetActiveIndex(), false);
         OnButtonGroupSelectReqEvent(*buttonGroup, reqEvent);
      }
   }
}


/*****************************************************************************/
bool DefaultButtonGroupController2D::findButtons(ButtonGroupWidget2D& buttonGroup, std::vector<ButtonWidget2D*>& buttons)
{
   if (buttonGroup.GetNode() != NULL)
   {
      ButtonGroupContentFinderCallback finderCallback(GetUseAllButtons(), buttonGroup.GetNode(), buttons);
      //hmibase::widget::utils::MessageUtils::distribute(hmibase::widget::utils::MessageUtils::getSceneContext(&buttonGroup), WidgetCheckReqMsg(&finderCallback));
      hmibase::widget::utils::MessageUtils::distributeToDescendants(hmibase::widget::utils::MessageUtils::getSceneContext(&buttonGroup), *(buttonGroup.GetNode()), WidgetCheckReqMsg(&finderCallback));
      return true;
   }
   return false;
}


/*****************************************************************************/
void DefaultButtonGroupController2D::updateButtons(ButtonGroupWidget2D& buttonGroup, std::vector<ButtonWidget2D*>& buttons, ButtonWidget2D* newButton, bool animate)
{
   ButtonWidget2D* oldButton = NULL;
   for (std::vector<ButtonWidget2D*>::const_iterator it = buttons.begin(); it != buttons.end(); ++it)
   {
      ButtonWidget2D* button = *it;
      if ((button != NULL) && (button->IsActive()) && (button != newButton))
      {
         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "UpdateButtons deactivate button %s", HMIBASE_TO_STRING_VW(button)));

         updateButton(buttonGroup, *button, false);
         oldButton = button;
      }
   }

   if (newButton != NULL)
   {
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "UpdateButtons activate button %s", HMIBASE_TO_STRING_VW(newButton)));

      updateButton(buttonGroup, *newButton, true);
   }

   if ((oldButton != NULL) && (newButton != NULL) && animate && buttonGroup.GetAnimationEnabled())
   {
      sendAnimationReqEvent(buttonGroup, oldButton, newButton);
   }
}


/*****************************************************************************/
void DefaultButtonGroupController2D::updateButton(ButtonGroupWidget2D& /*buttonGroup*/, ButtonWidget2D& button, bool active)
{
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "UpdateButton active=%u activeButtonRenderOrderRank=%d button=%s",
                       active, GetActiveButtonRenderOrderRank(), HMIBASE_TO_STRING_VW(&button)));

   button.SetOwnedByButtonGroup(true);
   button.SetActive(active);
   if ((GetActiveButtonRenderOrderRank() != 0) && (button.GetNode() != NULL))
   {
      button.GetNode()->SetRenderOrderRank(active ? GetActiveButtonRenderOrderRank() : 0);
   }
}


/*****************************************************************************/
void DefaultButtonGroupController2D::sendAnimationReqEvent(ButtonGroupWidget2D& buttonGroup, ButtonWidget2D* oldButton, ButtonWidget2D* newButton)
{
   if (newButton != NULL)
   {
      newButton->OnEvent(ButtonGroupAnimationReqEvent(&buttonGroup, oldButton, newButton, buttonGroup.GetAnimationDuration()));
   }
}
