/* ***************************************************************************************
* FILE:          ToggleButton2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ToggleButton2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "widget2D_std_if.h"
#include "ToggleButton2D.h"

#include <Widgets/2D/WidgetFinder2D.h>
#include <Widgets/2D/Button/ButtonWidget2D.h>
#include <Widgets/2D/ButtonGroup/ButtonGroupWidget2D.h>
#include <View/CGI/CgiExtensions/AppViewHandler.h>

#include <Trace/ToString.h>
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_BUTTON
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ToggleButton2D.cpp.trc.h"
#endif


using ::Candera::Char;//required for CANDERA_RTTI_DEFINITION
CANDERA_RTTI_DEFINITION(ToggleButtonAdornerAnimation)

using namespace hmibase::widget::adorner;


/*****************************************************************************/
ToggleButtonAdornerAnimation::ToggleButtonAdornerAnimation(AppViewHandler& appViewHandler) : Base(appViewHandler)
{
   //2 keyframes, 1 channel (default)
   _leftNodeAnimatedProperty.init(2, 1, Candera::Animation::StepInterpolationStrategy::Create());
   _leftNodeAnimatedProperty.setSequenceTime(0, 0);

   //2 keyframes, 1 channel (default)
   _rightNodeAnimatedProperty.init(2, 1, Candera::Animation::StepInterpolationStrategy::Create());
   _rightNodeAnimatedProperty.setSequenceTime(0, 0);
}


/*****************************************************************************/
bool ToggleButtonAdornerAnimation::attachToAdorner(Adorner::SharedPointer adorner)
{
   if (!adorner.PointsToNull() && (adorner->getNode() != NULL))
   {
      ToggleButtonAdornerData* toggleData = adorner->Data.get<ToggleButtonAdornerData>();
      if (toggleData != NULL)
      {
         //attach animated property for left node
         if (toggleData->LeftNode != NULL)
         {
            Candera::Animation::AnimationPropertySetter::SharedPointer propertySetter(::hmibase::view::AnimationPropertySetterUtils::createNodePropertySetter<Candera::Animation::RenderingEnabledNode2DPropertySetter>(toggleData->LeftNode));
            if (!propertySetter.PointsToNull())
            {
               _leftNodeAnimatedProperty.setPropertySetter(propertySetter);
               _leftNodeAnimatedProperty.addToPlayer(_animationWrapper.getPlayer());
            }
         }

         //attach animated property for right node
         if (toggleData->RightNode != NULL)
         {
            Candera::Animation::AnimationPropertySetter::SharedPointer propertySetter(::hmibase::view::AnimationPropertySetterUtils::createNodePropertySetter<Candera::Animation::RenderingEnabledNode2DPropertySetter>(toggleData->RightNode));
            if (!propertySetter.PointsToNull())
            {
               _rightNodeAnimatedProperty.setPropertySetter(propertySetter);
               _rightNodeAnimatedProperty.addToPlayer(_animationWrapper.getPlayer());
            }
         }
      }
   }

   return Base::attachToAdorner(adorner);
}


/*****************************************************************************/
void ToggleButtonAdornerAnimation::detachFromAdorner()
{
   Base::detachFromAdorner();

   //detach animated property for left node
   _leftNodeAnimatedProperty.setPropertySetter(Candera::Animation::AnimationPropertySetter::SharedPointer());
   _leftNodeAnimatedProperty.removeFromPlayer(_animationWrapper.getPlayer());

   //detach animated property for right node
   _rightNodeAnimatedProperty.setPropertySetter(Candera::Animation::AnimationPropertySetter::SharedPointer());
   _rightNodeAnimatedProperty.removeFromPlayer(_animationWrapper.getPlayer());
}


/*****************************************************************************/
bool ToggleButtonAdornerAnimation::start()
{
   if (!_adorner.PointsToNull() && (_adorner->getNode() != NULL))
   {
      Candera::Vector2 beginPosition(getBeginValues());
      Candera::Vector2 endPosition(getEndValues());

      bool isValid = false;
      ToggleButtonAdornerData* toggleData = _adorner->Data.get<ToggleButtonAdornerData>();
      ToggleButtonAdornerManager::enToggleSide beginSide = ToggleButtonAdornerManager::getSideForPosition(_adorner, beginPosition);
      ToggleButtonAdornerManager::enToggleSide endSide = ToggleButtonAdornerManager::getSideForPosition(_adorner, endPosition);

      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "start beginPos=%d beginSide=%u endPos=%d endSide=%u",
                          static_cast<int>(beginPosition.GetX()), beginSide, static_cast<int>(endPosition.GetX()), endSide));

      if ((toggleData != NULL) && (beginSide != ToggleButtonAdornerManager::UndefSide) && (endSide != ToggleButtonAdornerManager::UndefSide))
      {
         isValid = true;
         switch (beginSide)
         {
            case ToggleButtonAdornerManager::LeftSide:
               _leftNodeAnimatedProperty.setKeyframeValue(0, 0, 1.0f);//left starts as visible
               _rightNodeAnimatedProperty.setKeyframeValue(0, 0, 0.0f);//right starts as hidden
               break;

            case ToggleButtonAdornerManager::RightSide:
               _leftNodeAnimatedProperty.setKeyframeValue(0, 0, 0.0f);//left starts as hidden
               _rightNodeAnimatedProperty.setKeyframeValue(0, 0, 1.0f);//right starts as visible
               break;

            default:
               isValid = false;
               break;
         }

         switch (endSide)
         {
            case ToggleButtonAdornerManager::LeftSide:
               _leftNodeAnimatedProperty.setKeyframeValue(1, 0, 1.0f);//left ends as visible
               _rightNodeAnimatedProperty.setKeyframeValue(1, 0, 0.0f);//right ends as hidden
               break;

            case ToggleButtonAdornerManager::RightSide:
               _leftNodeAnimatedProperty.setKeyframeValue(1, 0, 0.0f);//left ends as hidden
               _rightNodeAnimatedProperty.setKeyframeValue(1, 0, 1.0f);//right ends as visible
               break;

            default:
               isValid = false;
               break;
         }

         //different sides => calculate changing sequence time
         if (beginSide != endSide)
         {
            Candera::Vector2 absoluteChangingPositionForFullRange(toggleData->GroupBounds.GetPosition() + (toggleData->GroupBounds.GetSize() - toggleData->AdornerSize) / 2);

            FeatStd::Float changingDistance = Candera::Math::Absolute(absoluteChangingPositionForFullRange.GetX() - beginPosition.GetX());
            FeatStd::Float totalDistance = Candera::Math::Absolute(endPosition.GetX() - beginPosition.GetX());
            Candera::Animation::SequenceTimeType changingSeqTime = static_cast<Candera::Animation::SequenceTimeType>(static_cast<FeatStd::Float>(getDuration()) * changingDistance / totalDistance);

            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "start changingDist(crt=%d absAndForFullRange=%d) totalDist=%d changingSeqTime=%d",
                                static_cast<int>(changingDistance),
                                static_cast<int>(absoluteChangingPositionForFullRange.GetX()),
                                static_cast<int>(totalDistance),
                                changingSeqTime));

            _leftNodeAnimatedProperty.setSequenceTime(1, changingSeqTime);
            _rightNodeAnimatedProperty.setSequenceTime(1, changingSeqTime);
         }
         //same side => animation with duration 0
         else
         {
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "start No side change detected"));

            _leftNodeAnimatedProperty.setSequenceTime(1, 0);
            _rightNodeAnimatedProperty.setSequenceTime(1, 0);
         }
      }

      //invalid sides=>hide nodes
      if (!isValid)
      {
         _leftNodeAnimatedProperty.setKeyframeValue(0, 0, 0.0f);
         _leftNodeAnimatedProperty.setKeyframeValue(1, 0, 0.0f);
         _leftNodeAnimatedProperty.setSequenceTime(1, 0);

         _rightNodeAnimatedProperty.setKeyframeValue(0, 0, 0.0f);
         _rightNodeAnimatedProperty.setKeyframeValue(1, 0, 0.0f);
         _rightNodeAnimatedProperty.setSequenceTime(1, 0);
      }
   }
   return Base::start();
}


/*****************************************************************************/
ToggleButtonAdornerManager::enToggleSide ToggleButtonAdornerManager::getSideForPosition(Adorner::SharedPointer adorner, const Candera::Vector2& position)
{
   if (!adorner.PointsToNull() && (adorner->getNode() != NULL) && (adorner->getNode()->GetParent() != NULL))
   {
      ToggleButtonAdornerData* toggleData = adorner->Data.get<ToggleButtonAdornerData>();
      if ((toggleData != NULL) && (toggleData->GroupBounds.GetWidth() > toggleData->AdornerSize.GetX()))
      {
         Candera::Vector2 adornerPosition(adorner->getNode()->GetParent()->GetWorldPosition() + position);

         FeatStd::Float leftMargin = Candera::Math::Absolute(adornerPosition.GetX() - toggleData->GroupBounds.GetLeft());
         FeatStd::Float rightMargin = Candera::Math::Absolute(toggleData->GroupBounds.GetLeft() + toggleData->GroupBounds.GetWidth() - adornerPosition.GetX() - toggleData->AdornerSize.GetX());

         return (leftMargin < rightMargin) ? LeftSide : RightSide;
      }
   }
   return UndefSide;
}


/*****************************************************************************/
ToggleButtonAdornerManager::enToggleSide ToggleButtonAdornerManager::getCurrentSide(Adorner::SharedPointer adorner)
{
   if (!adorner.PointsToNull() && (adorner->getNode() != NULL))
   {
      return getSideForPosition(adorner, adorner->getNode()->GetPosition());
   }
   return UndefSide;
}


/*****************************************************************************/
Adorner::SharedPointer ToggleButtonAdornerManager::createAdorner(Candera::Widget2D& widget, Candera::Node2D* adornerNode, Candera::Node2D* adornerContainer)
{
   Adorner::SharedPointer adorner(Base::createAdorner(widget, adornerNode, adornerContainer));

   ButtonWidget2D* button = Candera::Dynamic_Cast<ButtonWidget2D*>(&widget);

   if (!adorner.PointsToNull() && (adorner->getNode() != NULL) && (button != NULL))
   {
      Candera::Node2D* leftNode = adorner->getNode()->GetDescendant(getLeftNodeName().c_str());
      Candera::Node2D* rightNode = adorner->getNode()->GetDescendant(getRightNodeName().c_str());

      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "createAdorner leftNode(%30s,%p) rightNode(%30s,%p)",
                          getLeftNodeName().c_str(), leftNode, getRightNodeName().c_str(), rightNode));

      if ((leftNode != NULL) || (rightNode != NULL))
      {
         ButtonGroupWidget2D* group = WidgetFinder::FindAncestorWidget<ButtonGroupWidget2D>(button);   //lint !e740
         ToggleButtonAdornerData* toggleData = adorner->Data.getOrCreate<ToggleButtonAdornerData>();
         if ((group != NULL) && (group->GetNode() != NULL) && (toggleData != NULL))
         {
            group->GetNode()->GetWorldAxisAlignedBoundingRectangle(toggleData->GroupBounds);

            Candera::Rectangle adornerBounds;
            adorner->getNode()->GetWorldAxisAlignedBoundingRectangle(adornerBounds);
            toggleData->AdornerSize = adornerBounds.GetSize();

            enToggleSide adornerSide = getCurrentSide(adorner);
            if (leftNode != NULL)
            {
               leftNode->SetRenderingEnabled(adornerSide == LeftSide);
            }
            if (rightNode != NULL)
            {
               rightNode->SetRenderingEnabled(adornerSide == RightSide);
            }

            toggleData->LeftNode = leftNode;
            toggleData->RightNode = rightNode;

            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "createAdorner side=%u adornerSize=%12s group=%s",
                                adornerSide,
                                HMIBASE_TO_STRING(adornerBounds.GetSize()),
                                HMIBASE_TO_STRING(toggleData->GroupBounds)));
         }
      }
      else
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "createAdorner No left node (%30s) and no right node (%30s) found %s!",
                            getLeftNodeName().c_str(), getRightNodeName().c_str(), HMIBASE_TO_STRING_VW(&widget)));
      }
   }
   return adorner;
}


/*****************************************************************************/
void ToggleButtonAdornerManager::moveAdorner(Adorner::SharedPointer adorner, const Candera::Vector2& delta)
{
   Base::moveAdorner(adorner, delta);

   if (!adorner.PointsToNull() && (adorner->getNode() != NULL))
   {
      ToggleButtonAdornerData* toggleData = adorner->Data.get<ToggleButtonAdornerData>();
      if (toggleData != NULL)
      {
         enToggleSide adornerSide = getCurrentSide(adorner);
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "moveAdorner side=%u", adornerSide));

         if (toggleData->LeftNode != NULL)
         {
            toggleData->LeftNode->SetRenderingEnabled(adornerSide == LeftSide);
         }
         if (toggleData->RightNode != NULL)
         {
            toggleData->RightNode->SetRenderingEnabled(adornerSide == RightSide);
         }
      }
   }
}


/*****************************************************************************/
AdornerAnimation* ToggleButtonAdornerManager::newAnimation(Candera::Widget2D& widget)
{
   if (widget.GetParentView() != NULL)
   {
      AppViewHandler* appViewHandler = dynamic_cast<AppViewHandler*>(widget.GetParentView()->GetViewHandler());
      if (appViewHandler != NULL)
      {
         return CANDERA_NEW(ToggleButtonAdornerAnimation)(*appViewHandler);
      }
   }

   return NULL;
}


/*****************************************************************************/
ToggleButtonAdornerManager::ToggleButtonAdornerManager() : _leftNodeName("_LeftNode"), _rightNodeName("_RightNode")
{
}


/*****************************************************************************/
const std::string& ToggleButtonAdornerManager::getLeftNodeName() const
{
   return _leftNodeName;
}


/*****************************************************************************/
void ToggleButtonAdornerManager::setLeftNodeName(const std::string& name)
{
   _leftNodeName = name;
}


/*****************************************************************************/
const std::string& ToggleButtonAdornerManager::getRightNodeName() const
{
   return _rightNodeName;
}


/*****************************************************************************/
void ToggleButtonAdornerManager::setRightNodeName(const std::string& name)
{
   _rightNodeName = name;
}
