/* ***************************************************************************************
 * FILE:          ButtonWidget2D.h
 * SW-COMPONENT:  HMI-BASE
 *  DESCRIPTION:  ButtonWidget2D is part of HMI-Base Widget Library
 *    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
 *
 * The reproduction, distribution and utilization of this file as well as the
 * communication of its contents to others without express authorization is
 * prohibited. Offenders will be held liable for the payment of damages.
 * All rights reserved in the event of the grant of a patent, utility model or design.
 *
 *************************************************************************************** */
#pragma once

#include <View/CGI/Widget/WidgetController.h>
#include <Widgets/2D/Button/generated/ButtonWidget2DBase.h>
#include <Widgets/2D/Appearance/AppearanceState.h>

namespace hmibase {
namespace widget {
namespace flags {
enum Enum_Button2D
{
   OwnedByButtonGroup = _LastValue_Touchable2D,
   TapStarted,

   //last value
   _LastValue_Button2D
};


}
namespace appearance {
class Appearance;
class AppearanceTemplateContainer;
class ButtonAppearanceHelper;
}


}
}


class ButtonExtensionWidget2D;

/////////////////////////////////////////////////////////////////////////
class ButtonWidget2D : public ButtonWidget2DBase
{
      friend class ButtonController2D;
      friend class hmibase::widget::appearance::ButtonAppearanceHelper;

   public:
      ButtonWidget2D();
      virtual ~ButtonWidget2D();

      CGI_WIDGET_RTTI_DECLARATION(ButtonWidget2D, ButtonWidget2DBase);

      /////////////////////////////////////////////////////////////////////////
      // overrides from Courier::FrameworkWidget & Candera::Widget
      virtual void OnFocus() override;
      virtual void OnLostFocus() override;
      virtual void OnParentViewRenderingEnabled(bool viewLoaded) override;
      virtual bool OnMessage(const Courier::Message& msg) override;

      // overrides from BaseWidget2D & Touchable2D
      virtual bool OnTouch(const Candera::Camera2D& camera, const Candera::Vector2& touchedPoint) override;
      virtual hmibase::input::gesture::GestureListener::Gestures GetGestureList() const override;

      // overrides from ControlTemplateCloneableWidget
      virtual bool CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap) override;

      /////////////////////////////////////////////////////////////////////////
      // returns the state which should be shown (when an adorner is visible or when a button animation is running it may be required to suppress some state flags like IsFocused or IsActive)
      virtual hmibase::widget::appearance::AppearanceState GetAppearanceState();

      /////////////////////////////////////////////////////////////////////////
      // presses the button from another component. Sends the reaction message too.
      void Press();
      // releases the button from another component. Sends the reaction message too. It has no effect if the button is already released.
      void Release();
      // aborts the pressed state of the button from another component. No reaction message is sent. It has no effect if the button is already released.
      void AbortPress();

      void postReactionMessage(enReaction reaction);

      /////////////////////////////////////////////////////////////////////////
      // this button belongs to a button group
      DefWidgetFlag(OwnedByButtonGroup, 0)
      DefWidgetFlag(TapStarted, 0)

      CdaWidget2DDef(ButtonWidget2D, ButtonWidget2DBase)
      CdaProperties()

      // we define this property here because we don't want to have a dedicated field member for it
      CdaBindableProperty(IsActive, bool, IsActive, SetActive)
      CdaDescription("Indicates if the button is active. For toggle/radio button active means checked/selected. For normal button active means that the option/action is active (for example track is playing, station is tuned, etc).")
      CdaCategory("Button")
      CdaBindablePropertyEnd()

      CdaPropertiesEnd()
      CdaWidgetDefEnd()

      struct ButtonTimerConfig
      {
         ButtonTimerConfig(Candera::UInt16 shortPress, Candera::UInt16 longPress, Candera::UInt16 primaryTimerInitial, Candera::UInt16 primaryTimerRepeat, Candera::UInt16 secondaryTimerInitial, Candera::UInt16 secondaryTimerRepeat)
         {
            ShortPressDelay = shortPress;
            LongPressDelay = longPress;
            PrimaryTimerInitialDelay = primaryTimerInitial;
            PrimaryTimerRepeatDelay = primaryTimerRepeat;
            SecondaryTimerInitialDelay = secondaryTimerInitial;
            SecondaryTimerRepeatDelay = secondaryTimerRepeat;
         }

         Candera::UInt16 ShortPressDelay;            //  700ms
         Candera::UInt16 LongPressDelay;             // 2000ms
         Candera::UInt16 PrimaryTimerInitialDelay;   //  200ms
         Candera::UInt16 PrimaryTimerRepeatDelay;    //  200ms
         Candera::UInt16 SecondaryTimerInitialDelay; // 1000ms
         Candera::UInt16 SecondaryTimerRepeatDelay;  //  100ms
      };

      static void setButtonTimerConfig(const ButtonTimerConfig& timerConfig)
      {
         _timerConfig = timerConfig;
      }

      class ButtonTimers
      {
         public:
            ButtonTimers();
            ~ButtonTimers();

            /////////////////////////////////////////////////////////////////////////
            void start(Candera::enTimerConfiguration config, const ButtonTimerConfig& delays, const FeatStd::Char* name);
            void stop(bool forceStop = false);

            /////////////////////////////////////////////////////////////////////////
            void stopPrimary();
            bool matchesPrimary(const Util::Timer& timer) const;
            Util::Timer::Status getPrimaryStatus() const;

            /////////////////////////////////////////////////////////////////////////
            void stopSecondary();
            bool matchesSecondary(const Util::Timer& timer) const;
            Util::Timer::Status getSecondaryStatus() const;

         private:
            static void createTimers();

            bool _aquired;

            // _timer is used for short/long press and primary repeat timer
            static Util::Timer* _primaryTimer;

            // _secondaryTimer is used for secondary repeat timer
            static Util::Timer* _secondaryTimer;
      };

   protected:
      // overrides from BaseWidget2D & Touchable2D
      virtual bool OnTouchMessage(const Courier::TouchMsg& msg) override;
      virtual bool OnTapGesture(const hmibase::input::gesture::GestureEvent& gestureData) override;
      virtual bool OnDragGesture(const hmibase::input::gesture::GestureEvent& gestureData) override;
      virtual bool OnSwipeGesture(const hmibase::input::gesture::GestureEvent& gestureData) override;

      virtual bool composerPropVisibleDisabledTouching() const override
      {
         return true;
      }
      virtual bool composerPropVisibleFilterPressHold() const override
      {
         return false;   // button has its own timers
      }
      virtual bool composerPropVisibleFilterPressRepeat() const override
      {
         return false;   // button has its own timers
      }

      virtual void OnChanged(FeatStd::UInt32 propertyId);

      // called by ButtonAppearanceHelper
      hmibase::widget::appearance::AppearanceTemplateContainer* GetOrCreateAppearanceTemplateContainer();

   private:
      void startTimers();
      void stopTimers();
      void onReaction(enReaction reaction);

      bool OnEnterKeyMessage(const EnterKeyStatusChangedUpdMsg& msg);
      bool OnListStatusMessage(const ListStatusUpdMsg& msg);
      bool OnTimerExpiredMessage(const TimerExpiredMsg& msg);

      bool OnTouchDown(const Candera::Vector2& point);
      bool OnTouchUp(const Candera::Vector2& point);
      bool OnTouchAbort();

      ButtonTimers _timers;

      // stores the index of the last timeout which expired for _timer
      int _lastTimeoutIndex;
      static const int INVALID_TIMEOUT_INDEX = -1;
      static const int SHORT_PRESS_TIMEOUT_INDEX = 0;
      static const int LONG_PRESS_TIMEOUT_INDEX = 1;

      // static button timer configuration
      static ButtonTimerConfig _timerConfig;

      // template appearance for this button used to create the appearance of clones
      hmibase::widget::appearance::AppearanceTemplateContainer* _appearanceTemplateContainer;
};


typedef ButtonWidget2D BaseButtonWidget2D;


/////////////////////////////////////////////////////////////////////////
class ButtonController2DData : public hmibase::widget::WidgetControllerData
{
      FEATSTD_TYPEDEF_BASE(hmibase::widget::WidgetControllerData);

   public:
      CANDERA_RTTI_DECLARATION(ButtonController2DData);

      ButtonController2DData() {}
};


/////////////////////////////////////////////////////////////////////////
class ButtonController2D : public hmibase::widget::WidgetController
{
      FEATSTD_TYPEDEF_BASE(hmibase::widget::WidgetController);

   public:
      CANDERA_RTTI_DECLARATION(ButtonController2D);

      typedef hmibase::widget::appearance::AppearanceState AppearanceState;
      virtual AppearanceState getAppearanceState(ButtonWidget2D& buttonWidget);
      AppearanceState getOriginalAppearanceState(hmibase::widget::Widget& buttonWidget);

      virtual bool onReaction(ButtonWidget2D&, enReaction)
      {
         return false;
      }

      WIDGET_CONTROLLER_ON_GESTURE()
      virtual bool OnTapGesture(DelegateWidget&, const hmibase::input::gesture::GestureEvent&)
      {
         return false;
      }
      virtual bool OnDragGesture(DelegateWidget&, const hmibase::input::gesture::GestureEvent&)
      {
         return false;
      }
      virtual bool OnSwipeGesture(DelegateWidget&, const hmibase::input::gesture::GestureEvent&)
      {
         return false;
      }
      virtual bool OnPinchSpreadGesture(DelegateWidget&, const hmibase::input::gesture::GestureEvent&)
      {
         return false;
      }
      virtual bool OnRotateGesture(DelegateWidget&, const hmibase::input::gesture::GestureEvent&)
      {
         return false;
      }
};


typedef ButtonController2D ButtonWidgetController;
