/* ***************************************************************************************
 * FILE:          ButtonWidget2D.cpp
 * SW-COMPONENT:  HMI-BASE
 *  DESCRIPTION:  ButtonWidget2D is part of HMI-Base Widget Library
 *    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
 *
 * The reproduction, distribution and utilization of this file as well as the
 * communication of its contents to others without express authorization is
 * prohibited. Offenders will be held liable for the payment of damages.
 * All rights reserved in the event of the grant of a patent, utility model or design.
 *
 *************************************************************************************** */

#include "widget2D_std_if.h"
#include "ButtonWidget2D.h"
#include "ButtonTouchHandler.h"
#include "ButtonAppearance2D.h"
#include "ButtonExtensionWidget2D.h"
#include <Widgets/2D/WidgetGestureConfig.h>
#include <Widgets/2D/Gesture/GestureWidget2D.h>
#include <Widgets/2D/Focus/FocusUtils2D.h>
#include <Widgets/2D/ControlTemplate/ControlTemplateBinding.h>
#include <Focus/FManager.h>

#include <Trace/ToString.h>
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_BUTTON
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ButtonWidget2D.cpp.trc.h"
#endif

CGI_WIDGET_RTTI_DEFINITION(ButtonWidget2D)
using ::Candera::Char;
CANDERA_RTTI_DEFINITION(ButtonController2DData)
CANDERA_RTTI_DEFINITION(ButtonController2D)


ButtonWidget2D::ButtonTimerConfig ButtonWidget2D::_timerConfig(700, 2000, 200, 200, 1000, 100);

///////////////////////////////////////////////////////////////////////////////
ButtonWidget2D::ButtonWidget2D() : Base(),
   _lastTimeoutIndex(INVALID_TIMEOUT_INDEX),
   _appearanceTemplateContainer(NULL)
{
   //by default buttons are touchable and focusable
   SetTouchable(true);
   SetTap(true);
   SetDoubleTap(false);
   SetPressHold(false); //long press is handled using button's own timers
   SetPressRepeat(false); //repeat is handled using button's own timers
   SetFocusableStatus(true);
}


///////////////////////////////////////////////////////////////////////////////
ButtonWidget2D::~ButtonWidget2D()
{
   stopTimers();

   if (_appearanceTemplateContainer != NULL)
   {
      FEATSTD_DELETE(_appearanceTemplateContainer);
      _appearanceTemplateContainer = NULL;
   }
}


///////////////////////////////////////////////////////////////////////////////
hmibase::widget::appearance::AppearanceTemplateContainer* ButtonWidget2D::GetOrCreateAppearanceTemplateContainer()
{
   if ((_appearanceTemplateContainer == NULL) && (GetAppearanceId() == 0))
   {
      _appearanceTemplateContainer = FEATSTD_NEW(hmibase::widget::appearance::AppearanceTemplateContainer);
   }
   return _appearanceTemplateContainer;
}


///////////////////////////////////////////////////////////////////////////////
hmibase::widget::appearance::AppearanceState ButtonWidget2D::GetAppearanceState()
{
   ButtonController2D* controller = Candera::Dynamic_Cast<ButtonController2D*>(GetController());
   if (controller != NULL)
   {
      return controller->getAppearanceState(*this);
   }

   AppearanceState state(IsFocused(), IsActive(), IsPressed(), IsEnabled());
   //don't show the focused image while the focus animation is running
   if (state.isFocused()
         && (Focus::FManager::getInstance().getAnimationManager() != NULL)
         && Focus::FManager::getInstance().getAnimationManager()->isAnimationRunning(this))
   {
      state.setFocused(false);
   }
   return state;
}


///////////////////////////////////////////////////////////////////////////////
bool ButtonWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   bool cloned(false);
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const ButtonWidget2D* original = CLONEABLE_WIDGET_CAST<const ButtonWidget2D*>(originalWidget);
      if (original == NULL)
      {
         return false;
      }

      SetPostButtonReactionMsg(original->GetPostButtonReactionMsg());
      SetTimerConfiguration(original->GetTimerConfiguration());
      SetTouchableArea(original->GetTouchableArea());
      SetTouchHandler(original->GetTouchHandler());

      // avoid getting the active state from the template if this button is part of a button group
      if (!IsOwnedByButtonGroup())
      {
         if (ControlTemplateBinding::IsSelectedBindable(*this))
         {
            Candera::Int32 value = ControlTemplateBinding::GetSelectedValue(*this);
            SetActive(value != 0);
         }
         else
         {
            SetActive(original->IsActive());
         }
      }

      //if (ControlTemplateBinding::IsItemsBindable(*this))
      //{
      //   FeatStd::Int appId = ControlTemplateBinding::GetItemsBindingIndex(*this);
      //   SetAppearanceId((appId > 0) ? static_cast<AppearanceIdType>(appId) : 0);
      //}

      if ((original->_appearanceTemplateContainer != NULL) && (GetAppearanceId() == 0) && (GetAppearance() == NULL))
      {
         SetAppearance(original->_appearanceTemplateContainer->createAppearance(*this));
      }
      cloned = true;
   }
   return cloned;
}


///////////////////////////////////////////////////////////////////////////////
void ButtonWidget2D::OnChanged(FeatStd::UInt32 propertyId)
{
   Base::OnChanged(propertyId);

   switch (propertyId)
   {
      case ActivePropertyId:
      {
         Invalidate();

         // property is bindable
         if (GetParentView() != NULL)
         {
            PropertyModified("IsActive");
         }
         break;
      }

      case EnabledPropertyId:
      {
         if (!IsEnabled())
         {
            stopTimers();
         }

         Invalidate();
         break;
      }

      case FocusedPropertyId:
      {
         Invalidate();

         // losing focus causes the button to abort its pressed state if it is not touched
         if (!IsFocused() && IsPressed() && !IsTapStarted())
         {
            AbortPress();
         }
         break;
      }

      case PressedPropertyId:
      {
         Invalidate();
         break;
      }

      case VisiblePropertyId:
      {
         if (!IsVisible())
         {
            stopTimers();
         }
         break;
      }

      default:
         break;
   }
}


///////////////////////////////////////////////////////////////////////////////
bool ButtonWidget2D::OnMessage(const Message& msg)
{
   if (Base::OnMessage(msg))
   {
      return true;
   }

   bool consumed = false;
   switch (msg.GetId())
   {
      case EnterKeyStatusChangedUpdMsg::ID:
      {
         const EnterKeyStatusChangedUpdMsg* enterKeyMsg = Courier::message_cast<const EnterKeyStatusChangedUpdMsg*>(&msg);
         if (enterKeyMsg != NULL)
         {
            consumed = OnEnterKeyMessage(*enterKeyMsg);
         }
      }
      break;

#ifndef VARIANT_S_FTR_ENABLE_BUTTON_GESTURES
      case hmibase::input::TouchAbort::ID:
      {
         consumed = OnTouchAbort();
      }
      break;
#endif

      case ListStatusUpdMsg::ID:
      {
         const ListStatusUpdMsg* listStatusUpdMsg = Courier::message_cast<const ListStatusUpdMsg*>(&msg);
         if (listStatusUpdMsg != NULL)
         {
            consumed = OnListStatusMessage(*listStatusUpdMsg);
         }
      }
      break;

      case TimerExpiredMsg::ID:
      {
         const TimerExpiredMsg* timerExpiredMsg = Courier::message_cast<const TimerExpiredMsg*>(&msg);
         if (timerExpiredMsg != NULL)
         {
            consumed = OnTimerExpiredMessage(*timerExpiredMsg);
         }
      }
      break;

      default:
         break;
   }

   return consumed;
}


///////////////////////////////////////////////////////////////////////////////
bool ButtonWidget2D::OnEnterKeyMessage(const EnterKeyStatusChangedUpdMsg& msg)
{
   if (IsFocused())
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnEnterKeyMessage key=%u %s", msg.GetState(), HMIBASE_TO_STRING_VW(this)));

      switch (msg.GetState())
      {
         case hmibase::HARDKEYSTATE_DOWN:
         {
            Press();
         }
         break;

         case hmibase::HARDKEYSTATE_UP:
         case hmibase::HARDKEYSTATE_LONGUP://long timer is handled by the button, we just need the up message
         {
            if (IsPressed())
            {
               Release();
            }
         }
         break;

         default:
            break;
      }

      //the focused button will consume the message
      return true;
   }
   return false;
}


///////////////////////////////////////////////////////////////////////////////
bool ButtonWidget2D::OnListStatusMessage(const ListStatusUpdMsg& msg)
{
   // if any list enters into swiping state it will consume all touch messages
   // =>we have to clear touched/pressed flags if they are set
   switch (msg.GetStatus())
   {
      case ListScrolling:
      case ListSwiping:
      {
         if (IsTapStarted())
         {
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnListStatusMessage status=%u %s",
                                msg.GetStatus(), HMIBASE_TO_STRING_VW(this)));

            AbortPress();
            SetTapStarted(false);
         }
      }
      break;

      default:
         break;
   }

   //don't consume this broadcast message so that others can also react on it
   return false;
}


///////////////////////////////////////////////////////////////////////////////
bool ButtonWidget2D::OnTimerExpiredMessage(const TimerExpiredMsg& msg)
{
   bool consumed = false;
   Util::Timer* timer = msg.GetTimer();
   if ((timer != NULL) && IsPressed())
   {
      //////////////////////////////////////////////////////////////////////
      //_Timer is used for short/long press and primary repeat timer
      if (_timers.matchesPrimary(*timer))
      {
         _lastTimeoutIndex = static_cast<int>(msg.GetTimoutCnt());
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnTimerExpiredMessage primary timer=%u %s",
                             msg.GetTimoutCnt(), HMIBASE_TO_STRING_VW(this)));

         switch (GetTimerConfiguration())
         {
            case Candera::enShortAndLongTimer:
            {
               //first timeout is for short press
               if (_lastTimeoutIndex == SHORT_PRESS_TIMEOUT_INDEX)
               {
                  onReaction(enShortPress);
               }
               //second timeout is for long press
               else if (_lastTimeoutIndex == LONG_PRESS_TIMEOUT_INDEX)
               {
                  //we don't need to have the timer running anymore
                  _timers.stopPrimary();

                  onReaction(enLongPress);
               }
               else
               {
                  _timers.stopPrimary();
               }
            }
            break;

            case Candera::enPrimarySecondaryRepeatTimer:
            {
               //post the repeat msg only if the timer is still running
               if (_timers.getPrimaryStatus() == Util::Timer::Running)
               {
                  onReaction(enRepeatPressPrimary);
               }
            }
            break;

            default:
               //nothing to do
               break;
         }
         consumed = true;
      }
      //////////////////////////////////////////////////////////////////////
      //_SecondaryTimer is used for secondary repeat timer
      else if (_timers.matchesSecondary(*timer))
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnTimerExpiredMessage secondary timer=%u %s",
                             msg.GetTimoutCnt(), HMIBASE_TO_STRING_VW(this)));

         //stop the primary repeat timer
         _timers.stopPrimary();

         //post the repeat msg only if the timer is still running
         if (_timers.getSecondaryStatus() == Util::Timer::Running)
         {
            onReaction(enRepeatPressSecondary);
         }
         consumed = true;
      }
      //////////////////////////////////////////////////////////////////////
      else
      {
         //nothing to do, not our timer
      }
   }
   return consumed;
}


///////////////////////////////////////////////////////////////////////////////
void ButtonWidget2D::OnParentViewRenderingEnabled(bool viewLoaded)
{
   Base::OnParentViewRenderingEnabled(viewLoaded);

   //abort press on view hide
   if (!viewLoaded && IsPressed())
   {
      AbortPress();
   }
}


///////////////////////////////////////////////////////////////////////////////
bool ButtonWidget2D::OnTouchMessage(const Courier::TouchMsg& msg)
{
   //handles old (scope 2) drag drop
   Base::OnTouchMessage(msg);

   bool consumed = false;
   switch (msg.GetState())
   {
      case Courier::TouchMsgState::Down:
         consumed = OnTouchDown(Candera::Vector2(static_cast<float>(msg.GetXPos()), static_cast<float>(msg.GetYPos())));
         break;

      case Courier::TouchMsgState::Move:
         if (IsTapStarted())
         {
            consumed = true;
            //if touch move is received for a coordinate outside the button, the press is aborted
            if (!IsInsideBoundingRect(Courier::TouchInfo(msg.GetXPos(), msg.GetYPos(), msg.GetTimeStamp(), msg.GetPointerId(), msg.GetSourceId())))
            {
               OnTouchAbort();
            }
         }
         break;

      case Courier::TouchMsgState::Up:
         if (IsTapStarted())
         {
            consumed = true;
            //if touch move is received for a coordinate outside the button, the press is aborted
            if (!IsInsideBoundingRect(Courier::TouchInfo(msg.GetXPos(), msg.GetYPos(), msg.GetTimeStamp(), msg.GetPointerId(), msg.GetSourceId())))
            {
               OnTouchAbort();
            }
            else
            {
               OnTouchUp(Candera::Vector2(static_cast<float>(msg.GetXPos()), static_cast<float>(msg.GetYPos())));
            }
         }
         break;

      default:
         break;
   }
   return consumed;
}


///////////////////////////////////////////////////////////////////////////////
bool ButtonWidget2D::OnTapGesture(const hmibase::input::gesture::GestureEvent& gestureData)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnTapGesture gesture=%100s %s",
                       HMIBASE_TO_STRING(gestureData),
                       HMIBASE_TO_STRING_VW(this)));

   bool consumed = false;
   switch (gestureData._gestureState)
   {
      case hmibase::input::gesture::GestureEvent::ET_START:
         consumed = OnTouchDown(Candera::Vector2(static_cast<float>(gestureData._pt1.x), static_cast<float>(gestureData._pt1.y)));
         break;

      case hmibase::input::gesture::GestureEvent::ET_END:
         consumed = OnTouchUp(Candera::Vector2(static_cast<float>(gestureData._pt1.x), static_cast<float>(gestureData._pt1.y)));
         break;

      case hmibase::input::gesture::GestureEvent::ET_ABORT:
         consumed = OnTouchAbort();
         break;

      case hmibase::input::gesture::GestureEvent::ET_DT_START:
         if (IsEnabled() && (GetNode() != NULL) && GetNode()->IsEffectiveRenderingEnabled())
         {
            onReaction(enDoubleTap);
            consumed = true;
         }
         break;

      case hmibase::input::gesture::GestureEvent::ET_MOVE:
         //if touch move is received for a coordinate outside the button, the press is aborted
         if (!IsInsideBoundingRect(getCurrentSessionTouchInfo()))
         {
            OnTouchAbort();
         }
         break;

      default:
         break;
   }

   return consumed;
}


///////////////////////////////////////////////////////////////////////////////
bool ButtonWidget2D::OnDragGesture(const hmibase::input::gesture::GestureEvent& gestureData)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnDragGesture gesture=%100s %s",
                       HMIBASE_TO_STRING(gestureData),
                       HMIBASE_TO_STRING_VW(this)));

   return hmibase::widget::gesture::DefaultGestureController2D::getInstance().postDragGestureMessage(*this, gestureData);
}


///////////////////////////////////////////////////////////////////////////////
bool ButtonWidget2D::OnSwipeGesture(const hmibase::input::gesture::GestureEvent& gestureData)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnSwipeGesture gesture=%100s %s",
                       HMIBASE_TO_STRING(gestureData),
                       HMIBASE_TO_STRING_VW(this)));

   return hmibase::widget::gesture::DefaultGestureController2D::getInstance().postSwipeGestureMessage(*this, gestureData);
}


///////////////////////////////////////////////////////////////////////////////
bool ButtonWidget2D::OnTouchDown(const Candera::Vector2& point)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnTouchDown point=%12s %s",
                       HMIBASE_TO_STRING(point),
                       HMIBASE_TO_STRING_VW(this)));

   Press();
   SetTapStarted(true);
   return true;
}


///////////////////////////////////////////////////////////////////////////////
bool ButtonWidget2D::OnTouchUp(const Candera::Vector2& point)
{
   bool consumed = false;
   if (IsTapStarted())
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnTouchUp point=%12s %s",
                          HMIBASE_TO_STRING(point),
                          HMIBASE_TO_STRING_VW(this)));

      Release();
      SetTapStarted(false);
      consumed = true;
   }
   else
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnTouchDown ignored point=%12s %s",
                          HMIBASE_TO_STRING(point),
                          HMIBASE_TO_STRING_VW(this)));
   }
   return consumed;
}


bool ButtonWidget2D::OnTouchAbort()
{
   if (IsTapStarted())
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnTouchAbort %s", HMIBASE_TO_STRING_VW(this)));

      AbortPress();
      SetTapStarted(false);
   }

   //don't consume this broadcast message so that others can also react on it
   return false;
}


///////////////////////////////////////////////////////////////////////////////
// Checks for intersection of touch coordinates with Widgets Nodes only when widgets is in enable state else touch messages will not be processed.
// Returns true if touched coordinates intersect with Widget's Node bounding rectangle, false otherwise.
bool ButtonWidget2D::OnTouch(const Candera::Camera2D& camera, const Candera::Vector2& touchedPoint)
{
   bool nodeTouched = false;

   if (IsVisible() && (GetNode() != NULL) && (GetNode()->IsEffectiveRenderingEnabled()))
   {
      ButtonTouchHandler* handler = NULL;
      const Candera::Rectangle& touchableArea = GetTouchableArea();
      if (GetTouchHandler() > 0)
      {
         handler = ButtonTouchHandlerRegistry::getItem(static_cast<ButtonTouchHandlerRegistry::IdType>(GetTouchHandler()));
      }

      if (handler != NULL)
      {
         nodeTouched = handler->contains(*this, camera, touchedPoint);
         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "OnTouch intersects=%u handler=%d %s",
                             nodeTouched, GetTouchHandler(), HMIBASE_TO_STRING_VW(this)));
      }
      else if ((touchableArea.GetWidth() > 0) && (touchableArea.GetHeight() > 0))
      {
         const Candera::Vector2 pointInWorldSpace = Candera::Math2D::TransformViewportToScene(camera, Candera::Math2D::TransformRenderTargetToViewport(camera, touchedPoint));

         Candera::Matrix3x2 matrix(GetNode()->GetWorldTransform());
         matrix.Inverse();
         const Candera::Vector2 pointInNodeSpace = matrix.Multiply(pointInWorldSpace);

         nodeTouched = touchableArea.Contains(pointInNodeSpace);

         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "OnTouch intersects=%u area=%24s point=%12s %s",
                             nodeTouched, HMIBASE_TO_STRING(touchableArea),
                             HMIBASE_TO_STRING(pointInNodeSpace),
                             HMIBASE_TO_STRING_VW(this)));
      }
      //allow the controller to also check the touch coordinates
      else if (GetController() != NULL)
      {
         nodeTouched = Base::OnTouch(camera, touchedPoint);
      }
      else
      {
         nodeTouched = IsPickIntersectingNode(camera, GetNode(), touchedPoint);
         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "OnTouch intersects=%u camera=%50s point=%12s %s",
                             nodeTouched, HMIBASE_TO_STRING_N2D(&camera),
                             HMIBASE_TO_STRING(touchedPoint),
                             HMIBASE_TO_STRING_VW(this)));
      }
   }

   return nodeTouched;
}


///////////////////////////////////////////////////////////////////////////////
void ButtonWidget2D::OnLostFocus()
{
   Base::OnLostFocus();

   SetFocused(false);
}


///////////////////////////////////////////////////////////////////////////////
void ButtonWidget2D::OnFocus()
{
   Base::OnFocus();

   SetFocused(true);
}


///////////////////////////////////////////////////////////////////////////////
void ButtonWidget2D::Press()
{
   if ((GetNode() != NULL) && GetNode()->IsEffectiveRenderingEnabled())
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "Press %s", HMIBASE_TO_STRING_VW(this)));

      if (IsEnabled())
      {
         SetPressed(true);
         onReaction(enPress);
         startTimers();
      }
      else if (IsDisabledTouching())
      {
         SetPressed(true);
         onReaction(enDisabledPress);
      }
      else
      {
         //nothing to do
      }
   }
}


///////////////////////////////////////////////////////////////////////////////
void ButtonWidget2D::Release()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "Release %s", HMIBASE_TO_STRING_VW(this)));

   //we stop the timers but don't delete them so we can reuse them later
   stopTimers();

   SetPressed(false);
   if ((GetNode() != NULL) && GetNode()->IsEffectiveRenderingEnabled())
   {
      if (IsEnabled())
      {
         switch (GetTimerConfiguration())
         {
            case Candera::enShortAndLongTimer:
            {
               //we are after short press
               if (_lastTimeoutIndex == SHORT_PRESS_TIMEOUT_INDEX)
               {
                  onReaction(enShortPressRelease);
               }
               //we are after long press
               else if (_lastTimeoutIndex == LONG_PRESS_TIMEOUT_INDEX)
               {
                  onReaction(enLongPressRelease);
               }
               //press and release
               else
               {
                  onReaction(enRelease);
               }
               break;
            }

            case Candera::enPrimarySecondaryRepeatTimer:
            {
               //if no repeat timer expired we send normal release
               if (_lastTimeoutIndex == INVALID_TIMEOUT_INDEX)
               {
                  onReaction(enRelease);
               }
               break;
            }

            default:
            {
               onReaction(enRelease);
               break;
            }
         }
      }
      else if (IsDisabledTouching())
      {
         onReaction(enDisabledRelease);
      }
      else
      {
         //nothing to do
      }
   }
}


///////////////////////////////////////////////////////////////////////////////
// Abort reaction is sent on rollback
void ButtonWidget2D::AbortPress()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "AbortPress %s", HMIBASE_TO_STRING_VW(this)));

   stopTimers();

   SetPressed(false);
   onReaction(enAbortPress);
}


///////////////////////////////////////////////////////////////////////////////
void ButtonWidget2D::onReaction(enReaction reaction)
{
   bool consumed = false;
   ButtonController2D* controller = Candera::Dynamic_Cast<ButtonController2D*>(GetController());
   if (controller != NULL)
   {
      consumed = controller->onReaction(*this, reaction);
   }

   if (!consumed)
   {
      postReactionMessage(reaction);
   }
   else
   {
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "OnReaction consumed by controller %s", HMIBASE_TO_STRING_VW(this)));
   }
}


///////////////////////////////////////////////////////////////////////////////
void ButtonWidget2D::postReactionMessage(enReaction reaction)
{
   Courier::View* parentView = GetParentView();
   if (GetPostButtonReactionMsg() && (parentView != NULL))
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PostReactionMessage reaction=%d userData=%u surfaceId=%u %s",
                          ETG_CENUM(enReaction, reaction), GetUserData(),
                          getCurrentSessionTouchInfo().mSourceId, HMIBASE_TO_STRING_VW(this)));

      ButtonReactionMsg* msg = COURIER_MESSAGE_NEW(ButtonReactionMsg)(parentView->GetId(), Courier::Identifier(GetLegacyName()), GetUserData(), reaction, getCurrentSessionTouchInfo().mSourceId);
      if (msg != NULL)
      {
         bool result = msg->Post();
         if (result)
         {
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ButtonWidget2D: [%50s] in view [%50s] successfully posted ButtonReactionMsg with Button Reaction - [%d] from Surface - [%u]",
                                GetLegacyName(), parentView->GetId().CStr(), reaction, getCurrentSessionTouchInfo().mSourceId));
         }
      }
   }
}


///////////////////////////////////////////////////////////////////////////////
void ButtonWidget2D::startTimers()
{
   _lastTimeoutIndex = INVALID_TIMEOUT_INDEX;
   _timers.start(GetTimerConfiguration(), _timerConfig, GetLegacyName());
}


///////////////////////////////////////////////////////////////////////////////
void ButtonWidget2D::stopTimers()
{
   _timers.stop();
}


///////////////////////////////////////////////////////////////////////////////
hmibase::input::gesture::GestureListener::Gestures ButtonWidget2D::GetGestureList() const
{
//#define VARIANT_S_FTR_ENABLE_BUTTON_GESTURES

#ifdef VARIANT_S_FTR_ENABLE_BUTTON_GESTURES
   return Base::GetGestureList();
#else
   hmibase::input::gesture::GestureListener::Gestures gestures;
   hmibase::input::gesture::GestureListener::GestureConfig config;
   config.legacyTouchParameter.dummy = 0;
   gestures.push_back(Gesture(GestureEvent::GT_LegacyTouch, config));
   return gestures;
#endif
}


///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
Util::Timer* ButtonWidget2D::ButtonTimers::_primaryTimer = NULL;
Util::Timer* ButtonWidget2D::ButtonTimers::_secondaryTimer = NULL;


///////////////////////////////////////////////////////////////////////////////
ButtonWidget2D::ButtonTimers::ButtonTimers() : _aquired(false)
{
}


///////////////////////////////////////////////////////////////////////////////
ButtonWidget2D::ButtonTimers::~ButtonTimers()
{
   stop();
}


void ButtonWidget2D::ButtonTimers::createTimers()
{
   if (_primaryTimer == NULL)
   {
      _primaryTimer = CANDERA_NEW(Util::Timer);
   }
   if (_secondaryTimer == NULL)
   {
      _secondaryTimer = CANDERA_NEW(Util::Timer);
   }
}


///////////////////////////////////////////////////////////////////////////////
void ButtonWidget2D::ButtonTimers::start(Candera::enTimerConfiguration config, const ButtonTimerConfig& delays, const FeatStd::Char* name)
{
   stop();

   createTimers();

   _aquired = true;
   switch (config)
   {
      case Candera::enShortAndLongTimer:
      {
         if (_primaryTimer != NULL)
         {
            //configure timer and start it
            _primaryTimer->setName("Button", name);
            _primaryTimer->setTimeout(SHORT_PRESS_TIMEOUT_INDEX, delays.ShortPressDelay);
            _primaryTimer->setTimeout(LONG_PRESS_TIMEOUT_INDEX, delays.LongPressDelay);
            _primaryTimer->start();
         }
      }
      break;

      case Candera::enPrimarySecondaryRepeatTimer:
      {
         if (_primaryTimer != NULL)
         {
            //configure timer and start it
            _primaryTimer->setName("Button-Primary", name);
            _primaryTimer->setTimeoutWithRepeat(delays.PrimaryTimerInitialDelay, delays.PrimaryTimerRepeatDelay);
            _primaryTimer->start();
         }

         if (_secondaryTimer != NULL)
         {
            //configure secondary timer and start it
            _secondaryTimer->setName("Button-Secondary", name);
            _secondaryTimer->setTimeoutWithRepeat(delays.SecondaryTimerInitialDelay, delays.SecondaryTimerRepeatDelay);
            _secondaryTimer->start();
         }
      }
      break;

      default:
         break;
   }
}


/////////////////////////////////////////////////////////////////////////
void ButtonWidget2D::ButtonTimers::stop(bool forceStop)
{
   if (_aquired || forceStop)
   {
      if (_primaryTimer != NULL)
      {
         _primaryTimer->stop();
      }
      if (_secondaryTimer != NULL)
      {
         _secondaryTimer->stop();
      }
   }
   _aquired = false;
}


/////////////////////////////////////////////////////////////////////////
void ButtonWidget2D::ButtonTimers::stopPrimary()
{
   if (_aquired && (_primaryTimer != NULL))
   {
      _primaryTimer->stop();
   }
}


/////////////////////////////////////////////////////////////////////////
bool ButtonWidget2D::ButtonTimers::matchesPrimary(const Util::Timer& timer) const
{
   return _aquired && (_primaryTimer != NULL) && (&timer == _primaryTimer);
}


/////////////////////////////////////////////////////////////////////////
Util::Timer::Status ButtonWidget2D::ButtonTimers::getPrimaryStatus() const
{
   return (_primaryTimer != NULL) ? _primaryTimer->getStatus() : Util::Timer::Stopped;
}


/////////////////////////////////////////////////////////////////////////
void ButtonWidget2D::ButtonTimers::stopSecondary()
{
   if (_aquired && (_secondaryTimer != NULL))
   {
      _secondaryTimer->stop();
   }
}


/////////////////////////////////////////////////////////////////////////
bool ButtonWidget2D::ButtonTimers::matchesSecondary(const Util::Timer& timer) const
{
   return _aquired && (_secondaryTimer != NULL) && (&timer == _secondaryTimer);
}


/////////////////////////////////////////////////////////////////////////
Util::Timer::Status ButtonWidget2D::ButtonTimers::getSecondaryStatus() const
{
   return (_secondaryTimer != NULL) ? _secondaryTimer->getStatus() : Util::Timer::Stopped;
}


/////////////////////////////////////////////////////////////////////////
hmibase::widget::appearance::AppearanceState ButtonController2D::getAppearanceState(ButtonWidget2D& buttonWidget)
{
   return getOriginalAppearanceState(buttonWidget);
}


/////////////////////////////////////////////////////////////////////////
hmibase::widget::appearance::AppearanceState ButtonController2D::getOriginalAppearanceState(hmibase::widget::Widget& buttonWidget)
{
   return AppearanceState(buttonWidget.IsFocused(), buttonWidget.IsActive(), buttonWidget.IsPressed(), buttonWidget.IsEnabled());
}
