/* ***************************************************************************************
* FILE:          ButtonSlideController2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ButtonSlideController2D.cpp is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "widget2D_std_if.h"
#include "ButtonSlideController2D.h"

#include <Widgets/2D/WidgetFinder2D.h>
#include <Widgets/2D/Button/ButtonWidget2D.h>
#include <Widgets/2D/ButtonGroup/ButtonGroupWidget2D.h>
#include <Widgets/2D/Marker/MarkerWidget2D.h>
#include <Widgets/utils/WidgetFunctors.h>
#include <Widgets/utils/WidgetTraverser.h>
#include <View/CGI/CgiExtensions/AnimationWrapper.h>

#include <Trace/ToString.h>
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_BUTTON
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ButtonSlideController2D.cpp.trc.h"
#endif


using ::Candera::Char;//required for CANDERA_RTTI_DEFINITION
CANDERA_RTTI_DEFINITION(ButtonSlideController2DData)
CANDERA_RTTI_DEFINITION(ButtonSlideController2D)

using namespace hmibase::widget::adorner;

/*****************************************************************************/
ButtonSlideController2D::ButtonSlideController2D(AdornerManager& adornerManager, const AdornerMarkerFilter& adornerMarkerFilter)
   : _adornerManager(adornerManager), _adornerMarkerFilter(adornerMarkerFilter), _adornerAnimation(NULL), _adornerHideTimer(NULL), _adornerHideTimerTimeout(250)
{
}


/*****************************************************************************/
ButtonSlideController2D::~ButtonSlideController2D()
{
   if (_adornerHideTimer != NULL)
   {
      stopAdornerHideTimer();
      CANDERA_DELETE(_adornerHideTimer);
      _adornerHideTimer = NULL;
   }

   if (_adornerAnimation != NULL)
   {
      _adornerManager.destroyAnimation(_adornerAnimation);
      _adornerAnimation = NULL;
   }
}


/*****************************************************************************/
void ButtonSlideController2D::setAdornerHideTimerTimeout(unsigned int timeout)
{
   _adornerHideTimerTimeout = timeout;
}


/*****************************************************************************/
void ButtonSlideController2D::startAdornerHideTimer()
{
   if (_adornerHideTimer == NULL)
   {
      _adornerHideTimer = CANDERA_NEW(Util::Timer);
   }
   if (_adornerHideTimer != NULL)
   {
      _adornerHideTimer->setName("ButtonSlideAdornerHideTimer", 0);
      _adornerHideTimer->setTimeout(0, _adornerHideTimerTimeout);

      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "StartAdornerHideTimer timeout=%u", _adornerHideTimerTimeout));
      _adornerHideTimer->start();
   }
}


/*****************************************************************************/
void ButtonSlideController2D::stopAdornerHideTimer()
{
   if (_adornerHideTimer != NULL)
   {
      if (_adornerHideTimer->running())
      {
         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "StopAdornerHideTimer"));
      }
      _adornerHideTimer->stop();
   }
}


/*****************************************************************************/
bool ButtonSlideController2D::OnEvent(DelegateWidget& delegateWidget, const hmibase::widget::WidgetControllerEvent& e)
{
   const ButtonGroupAnimationReqEvent* animationReq = Candera::Dynamic_Cast<const ButtonGroupAnimationReqEvent*>(&e);
   if (animationReq != NULL)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnEvent(ButtonGroupAnimationReqEvent) %s", HMIBASE_TO_STRING_VW(&delegateWidget)));

      ButtonWidget2D* widget = Candera::Dynamic_Cast<ButtonWidget2D*>(&delegateWidget);
      ButtonSlideController2DData* data = Candera::Dynamic_Cast<ButtonSlideController2DData*>(delegateWidget.GetControllerData());

      if ((widget != NULL) && (data != NULL))
      {
         return onButtonGroupAnimationReqEvent(*widget, *data, *animationReq);
      }
   }

   return false;
}


/*****************************************************************************/
bool ButtonSlideController2D::OnMessage(DelegateWidget& widget, const Courier::Message& msg)
{
   if (Base::OnMessage(widget, msg))
   {
      return true;
   }

   bool consumed = false;
   switch (msg.GetId())
   {
      case TimerExpiredMsg::ID:
      {
         const TimerExpiredMsg* timerExpiredMsg = Courier::message_cast<const TimerExpiredMsg*>(&msg);
         if (timerExpiredMsg != NULL)
         {
            consumed = onTimerExpiredMessage(widget, *timerExpiredMsg);
         }
      }
      break;

      default:
         break;
   }

   return consumed;
}


/*****************************************************************************/
bool ButtonSlideController2D::onTimerExpiredMessage(DelegateWidget& widget, const TimerExpiredMsg& msg)
{
   if (msg.GetTimer() == _adornerHideTimer)
   {
      onAdornerHideTimerExpired(widget);
      return true;
   }
   return false;
}


/*****************************************************************************/
void ButtonSlideController2D::onAdornerHideTimerExpired(DelegateWidget& widget)
{
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "OnAdornerHideTimerExpired buttonGroupData=%p %s",
                       _adornerGroupToHide.GetPointerToSharedInstance(), HMIBASE_TO_STRING_VW(&widget)));

   stopAdornerHideTimer();
   hideOldAdorner();
}


/*****************************************************************************/
void ButtonSlideController2D::scheduleAdornerHide(ButtonWidget2D& button, ButtonSlideController2DData& buttonData, bool startTimer)
{
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "ScheduleAdornerHide buttonGroupData=%p startTimer=%u timeout=%u %s",
                       buttonData.GroupData.GetPointerToSharedInstance(), startTimer,
                       _adornerHideTimerTimeout, HMIBASE_TO_STRING_VW(&button)));

   _adornerGroupToHide = buttonData.GroupData;

   //timeout is zero, perform immediate hide
   if (_adornerHideTimerTimeout == 0)
   {
      hideOldAdorner();
   }
   //start the timer
   else if (startTimer)
   {
      startAdornerHideTimer();
   }
   else
   {
      //nothing to do
   }
}


/*****************************************************************************/
static bool groupHasAnyButtonTouched(ButtonGroupWidget2D& buttonGroup)
{
   std::vector<ButtonWidget2D*> buttons;
   if (buttonGroup.findButtons(buttons))
   {
      for (std::vector<ButtonWidget2D*>::iterator it = buttons.begin(); it != buttons.end(); ++it)
      {
         ButtonWidget2D* button = (*it);
         if (button != NULL)
         {
            ButtonSlideController2DData* buttonData = Candera::Dynamic_Cast<ButtonSlideController2DData*>(button->GetControllerData());
            if ((buttonData != NULL) && buttonData->ActiveGestures.any())
            {
               return true;
            }
         }
      }
   }

   return false;
}


/*****************************************************************************/
bool ButtonSlideController2D::onButtonGroupAnimationReqEvent(ButtonWidget2D& widget, ButtonSlideController2DData& buttonData, const ButtonGroupAnimationReqEvent& animationReq)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnButtonGroupAnimationReqEvent buttonGroupData=%p %s",
                       buttonData.GroupData.GetPointerToSharedInstance(), HMIBASE_TO_STRING_VW(&widget)));

   if (animationReq.getEndWidget() == &widget)
   {
      ButtonGroupWidget2D* buttonGroup = animationReq.getButtonGroup();
      ButtonWidget2D* startButton = animationReq.getStartWidget();
      ButtonWidget2D* endButton = animationReq.getEndWidget();

      if ((buttonGroup != NULL) && (startButton != NULL) && (endButton != NULL))
      {
         if (!groupHasAnyButtonTouched(*buttonGroup))
         {
            startAnimation(*startButton, *endButton, animationReq.getAnimationDuration());
         }
         else
         {
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnButtonGroupAnimationReqEvent buttonGroup=%s already has a touched button!", HMIBASE_TO_STRING_VW(buttonGroup)));
         }
      }
      return true;
   }
   else
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnButtonGroupAnimationReqEvent no matching endWidget=%s!",
                          HMIBASE_TO_STRING_VW(animationReq.getEndWidget())));
   }

   return false;
}


/*****************************************************************************/
ButtonGroupWidget2D* ButtonSlideController2D::findGroup(ButtonWidget2D& button, ButtonSlideController2DData&)
{
   ButtonGroupWidget2D* group = WidgetFinder::FindAncestorWidget<ButtonGroupWidget2D>(&button);   //lint !e740

   return group;
}


/*****************************************************************************/
void ButtonSlideController2D::updateButtonData(ButtonWidget2D& button, ButtonSlideController2DData& buttonData)
{
   if (!buttonData.IsGroupSearched)
   {
      //prevent further updates
      buttonData.IsGroupSearched = true;

      ButtonGroupWidget2D* group = findGroup(button, buttonData);
      if (group != NULL)
      {
         buttonData.GroupId = Courier::Identifier(group->GetLegacyName());
         DefaultButtonGroupController2DData* groupData = Candera::Dynamic_Cast<DefaultButtonGroupController2DData*>(group->GetControllerData());
         if (groupData != NULL)
         {
            buttonData.GroupData = groupData->Group;
         }
         else
         {
            buttonData.GroupId = Courier::Identifier();
            buttonData.GroupData.Release();
         }
      }
      else
      {
         buttonData.GroupId = Courier::Identifier();
         buttonData.GroupData.Release();
      }

      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "UpdateButtonData buttonGroupData=%p %s",
                          buttonData.GroupData.GetPointerToSharedInstance(), HMIBASE_TO_STRING_VW(&button)));
   }
}


/*****************************************************************************/
Adorner::SharedPointer ButtonSlideController2D::createAdorner(ButtonWidget2D& button, ButtonSlideController2DData& buttonData)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "CreateAdorner buttonGroupData=%p %s",
                       buttonData.GroupData.GetPointerToSharedInstance(), HMIBASE_TO_STRING_VW(&button)));

   DefaultButtonGroupData::SharedPointer groupData = buttonData.GroupData;
   if (groupData.PointsToNull())
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "CreateAdorner no button group data %s!", HMIBASE_TO_STRING_VW(&button)));
      return Adorner::SharedPointer();
   }

   ButtonGroupWidget2D* group = findGroup(button, buttonData);
   Candera::Node2D* adornerNode = NULL;

   //if an adorner marker filter is specified use it to find the adorner node
   if (_adornerMarkerFilter != AdornerMarkerFilter())
   {
      MarkerWidget2D* adornerMarker = NULL;

      //first search for the adorner marker as a descendant of the button group
      if ((group != NULL) && (group->GetNode() != NULL))
      {
         adornerMarker = MarkerWidget2D::findDescendantMarker(hmibase::widget::utils::MessageUtils::getSceneContext(&button), *(group->GetNode()), _adornerMarkerFilter);
      }
      //if the button group has no descendant adorner marker search for it in the entire scene
      if (adornerMarker == NULL)
      {
         adornerMarker = MarkerWidget2D::findMarker(hmibase::widget::utils::MessageUtils::getSceneContext(&button), _adornerMarkerFilter);
      }
      if (adornerMarker != NULL)
      {
         adornerNode = adornerMarker->GetNode();
      }
   }

   Candera::Node2D* adornerContainer = NULL;
   if (group != NULL)
   {
      adornerContainer = group->GetAdornerContainerNode();
      if ((adornerContainer == NULL) && (group->GetNode() != NULL))
      {
         adornerContainer = group->GetNode()->GetChild("_AdornerContainerNode");
      }
   }
   groupData->AdornerHandle = _adornerManager.createAdorner(button, adornerNode, adornerContainer);
   return groupData->AdornerHandle;
}


/*****************************************************************************/
void ButtonSlideController2D::showAdorner(ButtonWidget2D& button, ButtonSlideController2DData& buttonData)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ShowAdorner buttonGroupData=%p %s",
                       buttonData.GroupData.GetPointerToSharedInstance(), HMIBASE_TO_STRING_VW(&button)));

   DefaultButtonGroupData::SharedPointer groupData = buttonData.GroupData;
   if (!groupData.PointsToNull())
   {
      //hide the old adorner if it is for a different group
      stopAdornerHideTimer();
      if (!_adornerGroupToHide.PointsToNull())
      {
         if (_adornerGroupToHide.GetPointerToSharedInstance() != groupData.GetPointerToSharedInstance())
         {
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ShowAdorner hide old adorner because it is for a different group."));
            hideOldAdorner();
         }
         _adornerGroupToHide.Release();
      }

      bool isPreviousPositionValid = false;
      Candera::Vector2 previousPosition;
      //if adorner is already created reuse its position
      if (!groupData->AdornerHandle.PointsToNull())
      {
         if ((groupData->AdornerHandle->getNode() != NULL) && (groupData->AdornerHandle->isVisible()))
         {
            previousPosition = groupData->AdornerHandle->getNode()->GetWorldPosition();
            isPreviousPositionValid = true;
         }
         //hide adorner
         hideAdorner(button, buttonData);
      }

      //create new adorner
      createAdorner(button, buttonData);
      if (isPreviousPositionValid && !groupData->AdornerHandle.PointsToNull() && (groupData->AdornerHandle->getNode() != NULL))
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ShowAdorner move new adorner to previous position %s", HMIBASE_TO_STRING(previousPosition)));

         Candera::Vector2 deltaPos(previousPosition - groupData->AdornerHandle->getNode()->GetWorldPosition());
         _adornerManager.moveAdorner(groupData->AdornerHandle, deltaPos);
      }

      //show the adorner
      _adornerManager.showAdorner(groupData->AdornerHandle);
   }
   else
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "ShowAdorner no button group data %s!", HMIBASE_TO_STRING_VW(&button)));
   }
}


/*****************************************************************************/
void ButtonSlideController2D::hideAdorner(ButtonWidget2D& button, ButtonSlideController2DData& buttonData)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "HideAdorner buttonGroupData=%p %s",
                       buttonData.GroupData.GetPointerToSharedInstance(), HMIBASE_TO_STRING_VW(&button)));

   DefaultButtonGroupData::SharedPointer groupData = buttonData.GroupData;
   if (!groupData.PointsToNull())
   {
      _adornerManager.hideAdorner(groupData->AdornerHandle);
      _adornerManager.destroyAdorner(groupData->AdornerHandle);
      groupData->AdornerHandle.Release();
   }
}


/*****************************************************************************/
void ButtonSlideController2D::hideOldAdorner()
{
   if (!_adornerGroupToHide.PointsToNull())
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "HideOldAdorner buttonGroupData=%p",
                          _adornerGroupToHide.GetPointerToSharedInstance()));

      if (!_adornerGroupToHide->AdornerHandle.PointsToNull())
      {
         bool buttonAvailable = false;
         ButtonWidget2D* button = Candera::Dynamic_Cast<ButtonWidget2D*>(_adornerGroupToHide->AdornerHandle->getSourceWidget());
         if (button != NULL)
         {
            ButtonSlideController2DData* buttonData = Candera::Dynamic_Cast<ButtonSlideController2DData*>(button->GetControllerData());
            if ((buttonData != NULL) && (!buttonData->GroupData.PointsToNull()))
            {
               buttonAvailable = true;
               hideAdorner(*button, *buttonData);
            }
         }
         if (!buttonAvailable)
         {
            ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "HideOldAdorner widget not available anymore for adorner!"));
            _adornerGroupToHide->AdornerHandle->hide();
         }
      }

      _adornerGroupToHide.Release();
   }
}


/*****************************************************************************/
void ButtonSlideController2D::CheckGestureConfig(const DelegateWidget& delegateWidget, hmibase::widget::WidgetGestureConfig& config)
{
   const ButtonWidget2D* widget = Candera::Dynamic_Cast<const ButtonWidget2D*>(&delegateWidget);
   if ((widget != NULL) && !widget->IsActive() && config.getDrag().Enabled)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "CheckGestureConfig disable Drag for inactive button %s",
                          HMIBASE_TO_STRING_VW(widget)));

      config.set(WidgetGestureConfig::Drag(false));
   }
}


/*****************************************************************************/
bool ButtonSlideController2D::OnGesture(DelegateWidget& delegateWidget, const hmibase::input::gesture::GestureEvent& gestureData)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnGesture gesture=%100s %s",
                       HMIBASE_TO_STRING(gestureData), HMIBASE_TO_STRING_VW(&delegateWidget)));

   bool consumed = false;
   ButtonWidget2D* widget = Candera::Dynamic_Cast<ButtonWidget2D*>(&delegateWidget);
   ButtonSlideController2DData* data = Candera::Dynamic_Cast<ButtonSlideController2DData*>(delegateWidget.GetControllerData());
   if ((widget != NULL) && (data != NULL))
   {
      switch (gestureData._gestureState)
      {
         case hmibase::input::gesture::GestureEvent::ET_START:
            onGestureBeforeStart(*widget, *data, gestureData);
            consumed = Base::OnGesture(delegateWidget, gestureData);
            onGestureAfterStart(*widget, *data, gestureData);
            break;
         case hmibase::input::gesture::GestureEvent::ET_END:
            onGestureAfterEnd(*widget, *data, gestureData);
            consumed = Base::OnGesture(delegateWidget, gestureData);
            onGestureAfterEnd(*widget, *data, gestureData);
            break;
         case hmibase::input::gesture::GestureEvent::ET_ABORT:
            onGestureAfterAbort(*widget, *data, gestureData);
            consumed = Base::OnGesture(delegateWidget, gestureData);
            onGestureAfterAbort(*widget, *data, gestureData);
            break;
         default:
            consumed = Base::OnGesture(delegateWidget, gestureData);
            break;
      }
   }

   return consumed;
}


/*****************************************************************************/
void ButtonSlideController2D::onGestureBeforeStart(ButtonWidget2D& /*widget*/, ButtonSlideController2DData& data, const hmibase::input::gesture::GestureEvent& gestureData)
{
   data.ActiveGestures.set(static_cast<size_t>(gestureData._gestureType));
}


/*****************************************************************************/
void ButtonSlideController2D::onGestureAfterStart(ButtonWidget2D& /*widget*/, ButtonSlideController2DData& /*data*/, const hmibase::input::gesture::GestureEvent& /*gestureData*/)
{
}


/*****************************************************************************/
void ButtonSlideController2D::onGestureBeforeEnd(ButtonWidget2D& /*widget*/, ButtonSlideController2DData& /*data*/, const hmibase::input::gesture::GestureEvent& /*gestureData*/)
{
}


/*****************************************************************************/
void ButtonSlideController2D::onGestureAfterEnd(ButtonWidget2D& /*widget*/, ButtonSlideController2DData& data, const hmibase::input::gesture::GestureEvent& gestureData)
{
   data.ActiveGestures.reset(static_cast<size_t>(gestureData._gestureType));
}


/*****************************************************************************/
void ButtonSlideController2D::onGestureBeforeAbort(ButtonWidget2D& /*widget*/, ButtonSlideController2DData& /*data*/, const hmibase::input::gesture::GestureEvent& /*gestureData*/)
{
}


/*****************************************************************************/
void ButtonSlideController2D::onGestureAfterAbort(ButtonWidget2D& /*widget*/, ButtonSlideController2DData& data, const hmibase::input::gesture::GestureEvent& gestureData)
{
   data.ActiveGestures.reset(static_cast<size_t>(gestureData._gestureType));
}


/*****************************************************************************/
bool ButtonSlideController2D::onDragGestureStart(ButtonWidget2D& button, ButtonSlideController2DData& buttonData, const hmibase::input::gesture::GestureEvent& /*gestureData*/)
{
   Courier::TouchInfo startTouchInfo = GetCurrentSessionStartTouchInfo(button);

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnDragGestureStart startTouchCoord=%s", HMIBASE_TO_STRING(startTouchInfo)));

   buttonData.DestinationId = Courier::Identifier();

   DefaultButtonGroupData::SharedPointer groupData = buttonData.GroupData;
   if (!groupData.PointsToNull())
   {
      showAdorner(button, buttonData);
      if (!groupData->AdornerHandle.PointsToNull() && (groupData->AdornerHandle->getNode() != NULL))
      {
         AdornerDragStartInfo coordinates;
         coordinates.touchCoordinates = Candera::Vector2(static_cast<FeatStd::Float>(startTouchInfo.mX), static_cast<FeatStd::Float>(startTouchInfo.mY));
         coordinates.adornerCoordinates = groupData->AdornerHandle->getNode()->GetWorldPosition();
         groupData->AdornerHandle->Data.set(coordinates);
      }
      postMessage(button, buttonData, hmibase::widget::gesture::enGestureEvent::Start);
   }

   return true;
}


/*****************************************************************************/
bool ButtonSlideController2D::moveAdornerOnDragMove(ButtonGroupWidget2D& group, ButtonWidget2D&, ButtonSlideController2DData&,
      hmibase::widget::adorner::Adorner::SharedPointer adorner, const hmibase::input::gesture::GestureEvent& gestureData)
{
   Candera::Vector2 touchCoord1(static_cast<FeatStd::Float>(gestureData._pt1.x), static_cast<FeatStd::Float>(gestureData._pt1.y));

   if ((group.GetNode() != NULL) && !adorner.PointsToNull() && (adorner->getNode() != NULL) && (adorner->getNode()->GetParent() != NULL))
   {
      AdornerDragStartInfo* startCoordinates = adorner->Data.get<AdornerDragStartInfo>();
      if (startCoordinates != NULL)
      {
         //get group absolute bounds
         Candera::Rectangle groupBounds;
         group.GetNode()->GetWorldAxisAlignedBoundingRectangle(groupBounds);

         //get adorner absolute bounds
         Candera::Rectangle adornerBounds;
         adorner->getNode()->GetWorldAxisAlignedBoundingRectangle(adornerBounds);

         //get adorner parent absolute position
         Candera::Vector2 adornerParentPosition(adorner->getNode()->GetParent()->GetWorldPosition());

         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "MoveAdornerOnDragMove groupBounds=%24s adornerBounds=%s",
                             HMIBASE_TO_STRING(groupBounds),
                             HMIBASE_TO_STRING(adornerBounds)));

         //calculate new position based on touch move delta
         Candera::Vector2 deltaTouchCoord(touchCoord1 - startCoordinates->touchCoordinates);
         Candera::Vector2 deltaPosition;

         //move adorner only if it fits inside the group (horizontally)
         if (adornerBounds.GetWidth() <= groupBounds.GetWidth())
         {
            adornerBounds.SetLeft(startCoordinates->adornerCoordinates.GetX() + deltaTouchCoord.GetX());

            //clamp left edge
            if (adornerBounds.GetLeft() < groupBounds.GetLeft())
            {
               adornerBounds.SetLeft(groupBounds.GetLeft());
            }

            //clamp right edge
            if (adornerBounds.GetLeft() + adornerBounds.GetWidth() > groupBounds.GetLeft() + groupBounds.GetWidth())
            {
               adornerBounds.SetLeft(groupBounds.GetLeft() + groupBounds.GetWidth() - adornerBounds.GetWidth());
            }

            deltaPosition.SetX(adornerBounds.GetLeft() - adornerParentPosition.GetX() - adorner->getNode()->GetPosition().GetX());
         }
         else
         {
            ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "MoveAdornerOnDragMove Can't move adorner horizontally because its width [%d] is greater than button group width [%d]",
                               static_cast<int>(adornerBounds.GetWidth()),
                               static_cast<int>(groupBounds.GetWidth())));
         }

         //move adorner only if it fits inside the group (vertically)
         if (adornerBounds.GetHeight() <= groupBounds.GetHeight())
         {
            adornerBounds.SetTop(startCoordinates->adornerCoordinates.GetY() + deltaTouchCoord.GetY());

            //clamp top edge
            if (adornerBounds.GetTop() < groupBounds.GetTop())
            {
               adornerBounds.SetTop(groupBounds.GetTop());
            }
            //clamp bottom edge
            if (adornerBounds.GetTop() + adornerBounds.GetHeight() > groupBounds.GetTop() + groupBounds.GetHeight())
            {
               adornerBounds.SetTop(groupBounds.GetTop() + groupBounds.GetHeight() - adornerBounds.GetHeight());
            }

            deltaPosition.SetY(adornerBounds.GetTop() - adornerParentPosition.GetY() - adorner->getNode()->GetPosition().GetY());
         }
         else
         {
            ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "MoveAdornerOnDragMove Can't move adorner vertically  because its height [%d] is greater than button group height [%d]",
                               static_cast<int>(adornerBounds.GetHeight()),
                               static_cast<int>(groupBounds.GetHeight())));
         }

         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "MoveAdornerOnDragMove startTouchCoord=%12s deltaPosition=%12s newAdornerBounds=%s",
                             HMIBASE_TO_STRING(startCoordinates->touchCoordinates),
                             HMIBASE_TO_STRING(deltaPosition),
                             HMIBASE_TO_STRING(adornerBounds)));

         _adornerManager.moveAdorner(adorner, deltaPosition);
         return true;
      }
   }
   return false;
}


/*****************************************************************************/
bool ButtonSlideController2D::onDragGestureMove(ButtonWidget2D& button, ButtonSlideController2DData& buttonData, const hmibase::input::gesture::GestureEvent& gestureData)
{
   Candera::Vector2 touchCoord1(static_cast<FeatStd::Float>(gestureData._pt1.x), static_cast<FeatStd::Float>(gestureData._pt1.y));

   ButtonWidget2D* destination = NULL;

   DefaultButtonGroupData::SharedPointer groupData = buttonData.GroupData;
   if (!groupData.PointsToNull() && !groupData->AdornerHandle.PointsToNull())
   {
      hmibase::widget::adorner::Adorner::SharedPointer adorner = groupData->AdornerHandle;
      ButtonGroupWidget2D* group = WidgetFinder::FindAncestorWidget<ButtonGroupWidget2D>(&button);   //lint !e740

      if ((group != NULL) && (adorner->getNode() != NULL))
      {
         moveAdornerOnDragMove(*group, button, buttonData, adorner, gestureData);

         Candera::Rectangle adornerBounds;
         adorner->getNode()->GetWorldAxisAlignedBoundingRectangle(adornerBounds);

         //find buttons contained in the group
         std::vector<ButtonWidget2D*> buttons;
         ButtonGroupController2D* groupController = Candera::Dynamic_Cast<ButtonGroupController2D*>(group->GetController());
         if (groupController != NULL)
         {
            groupController->findButtons(*group, buttons);
         }

         //find in the buttons the one which has the greatest intersection surface with the adorner
         Candera::Float destinationIntersectionSurface = 0.0f;
         for (std::vector<ButtonWidget2D*>::iterator it = buttons.begin(); it != buttons.end(); ++it)
         {
            ButtonWidget2D* candidateDestination = *it;
            if ((candidateDestination != NULL) && (candidateDestination->GetNode() != NULL))
            {
               //get candidate rectangle
               Candera::Rectangle candidateDestinationRectangle;
               candidateDestination->GetNode()->GetWorldAxisAlignedBoundingRectangle(candidateDestinationRectangle);

               candidateDestinationRectangle.Intersect(adornerBounds);
               if ((candidateDestinationRectangle.GetWidth() > 0.0f) && (candidateDestinationRectangle.GetHeight() > 0.0f))
               {
                  Candera::Float candidateDestinationIntersectionSurface = candidateDestinationRectangle.GetWidth() * candidateDestinationRectangle.GetHeight();
                  if (candidateDestinationIntersectionSurface > destinationIntersectionSurface)
                  {
                     destinationIntersectionSurface = candidateDestinationIntersectionSurface;
                     destination = candidateDestination;
                  }
               }
            }
         }
      }
   }

   if (destination != NULL)
   {
      buttonData.DestinationId = Courier::Identifier(destination->GetLegacyName());
      postMessage(button, buttonData, hmibase::widget::gesture::enGestureEvent::Move);
      //buttonGroup->OnEvent(ButtonGroupSelectReqEvent(enSelectIdentifier, Courier::Identifier(dragDestination->GetLegacyName())));
   }
   else
   {
      buttonData.DestinationId = Courier::Identifier();
   }

   return true;
}


/*****************************************************************************/
bool ButtonSlideController2D::onDragGestureEnd(ButtonWidget2D& button, ButtonSlideController2DData& buttonData, const hmibase::input::gesture::GestureEvent& /*gestureData*/)
{
   scheduleAdornerHide(button, buttonData);
   postMessage(button, buttonData, hmibase::widget::gesture::enGestureEvent::End);

   buttonData.DestinationId = Courier::Identifier();

   return true;
}


/*****************************************************************************/
bool ButtonSlideController2D::onDragGestureAbort(ButtonWidget2D& button, ButtonSlideController2DData& buttonData, const hmibase::input::gesture::GestureEvent& /*gestureData*/)
{
   scheduleAdornerHide(button, buttonData);
   postMessage(button, buttonData, hmibase::widget::gesture::enGestureEvent::Abort);

   buttonData.DestinationId = Courier::Identifier();

   return true;
}


/*****************************************************************************/
bool ButtonSlideController2D::OnDragGesture(DelegateWidget& widget, const hmibase::input::gesture::GestureEvent& gestureData)
{
   ButtonWidget2D* button = Candera::Dynamic_Cast<ButtonWidget2D*>(&widget);
   ButtonSlideController2DData* buttonData = Candera::Dynamic_Cast<ButtonSlideController2DData*>(widget.GetControllerData());
   if ((button == NULL) || (buttonData == NULL))
   {
      return false;
   }

   //force group search at the start of a drag gesture to ensure the correct group
   if (gestureData._gestureState == hmibase::input::gesture::GestureEvent::ET_START)
   {
      buttonData->IsGroupSearched = false;
      buttonData->GroupData.Release();
   }

   updateButtonData(*button, *buttonData);
   DefaultButtonGroupData::SharedPointer groupData = buttonData->GroupData;
   if (groupData.PointsToNull())
   {
      return false;
   }

   bool consumed = false;
   switch (gestureData._gestureState)
   {
      case hmibase::input::gesture::GestureEvent::ET_START:
         consumed = onDragGestureStart(*button, *buttonData, gestureData);
         break;

      case hmibase::input::gesture::GestureEvent::ET_MOVE:
         consumed = onDragGestureMove(*button, *buttonData, gestureData);
         break;

      case hmibase::input::gesture::GestureEvent::ET_END:
         consumed = onDragGestureEnd(*button, *buttonData, gestureData);
         break;

      case hmibase::input::gesture::GestureEvent::ET_ABORT:
         consumed = onDragGestureAbort(*button, *buttonData, gestureData);
         break;

      default:
         break;
   }

   return consumed;
}


/*****************************************************************************/
hmibase::widget::appearance::AppearanceState ButtonSlideController2D::getAppearanceState(ButtonWidget2D& button)
{
   AppearanceState state = getOriginalAppearanceState(button);
   if (state.isActive())
   {
      ButtonSlideController2DData* buttonData = Candera::Dynamic_Cast<ButtonSlideController2DData*>(button.GetControllerData());
      if (buttonData != NULL)
      {
         updateButtonData(button, *buttonData);
         DefaultButtonGroupData::SharedPointer group = buttonData->GroupData;
         if (!group.PointsToNull() && !group->AdornerHandle.PointsToNull() && group->AdornerHandle->isVisible())
         {
            state.setActive(false);
         }
      }
   }

   return state;
}


/*****************************************************************************/
bool ButtonSlideController2D::postMessage(ButtonWidget2D& button, ButtonSlideController2DData& buttonData, hmibase::widget::gesture::enGestureEvent::Enum evt)
{
   Courier::View* view = button.GetParentView();
   if (view != NULL)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PostMessage evt=[%d] group=%15s destination=%15s %s",
                          ETG_CENUM(hmibase::widget::gesture::enGestureEvent::Enum, evt),
                          HMIBASE_TO_STRING(buttonData.GroupId),
                          HMIBASE_TO_STRING(buttonData.DestinationId),
                          HMIBASE_TO_STRING_VW(&button)));

      ButtonSlideMsg* msg = COURIER_MESSAGE_NEW(ButtonSlideMsg)(view->GetId(), Courier::Identifier(button.GetLegacyName()), button.GetUserData(),
                            buttonData.DestinationId, buttonData.GroupId, evt);

      if (msg != NULL)
      {
         return msg->Post();
      }
   }
   return false;
}


/*****************************************************************************/
void ButtonSlideController2D::startAnimation(ButtonWidget2D& startWidget, ButtonWidget2D& endWidget, unsigned int animationDuration)
{
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "StartAnimation animationDuration=%u defaultAnimationDuration=%u startWidget=%50s endWidget=%s",
                       animationDuration,
                       _adornerManager.getDefaultAnimationDuration(),
                       HMIBASE_TO_STRING_W(&startWidget),
                       HMIBASE_TO_STRING_VW(&endWidget)));

   if ((startWidget.GetNode() == NULL) || (endWidget.GetNode() == NULL))
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "StartAnimation widget node is null for startWidget(%50s) or endWidget(%s)!",
                         HMIBASE_TO_STRING_W(&startWidget), HMIBASE_TO_STRING_VW(&endWidget)));
      return;
   }

   //reuse the adorner animation, so create it only if required
   if (_adornerAnimation == NULL)
   {
      _adornerAnimation = _adornerManager.createAnimation(startWidget);
      if (_adornerAnimation == NULL)
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "StartAnimation failed to create adorner for widget(%s)!", HMIBASE_TO_STRING_VW(&startWidget)));
         return;
      }
      _adornerAnimation->addListener(*this);
   }

   _adornerManager.stopAnimation(*_adornerAnimation);

   ButtonSlideController2DData* buttonData = Candera::Dynamic_Cast<ButtonSlideController2DData*>(startWidget.GetControllerData());
   if (buttonData == NULL)
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "StartAnimation invalid controller data for widget(%s)!", HMIBASE_TO_STRING_VW(&startWidget)));
      return;
   }

   DefaultButtonGroupData::SharedPointer groupData = buttonData->GroupData;
   if (groupData.PointsToNull())
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "StartAnimation no button group data for widget(%s)!", HMIBASE_TO_STRING_VW(&startWidget)));
      return;
   }

   Candera::Vector2 startPos;
   //if adorner is already created reuse its position
   if (!groupData->AdornerHandle.PointsToNull())
   {
      startPos = groupData->AdornerHandle->getNode()->GetWorldPosition();
      hideAdorner(startWidget, *buttonData);
   }
   //otherwise take position from the start widget's node
   else
   {
      startPos = startWidget.GetNode()->GetWorldPosition();
   }

   showAdorner(startWidget, *buttonData);
   if ((!groupData->AdornerHandle.PointsToNull()) && (groupData->AdornerHandle->getNode() != NULL) && (groupData->AdornerHandle->getNode()->GetParent() != NULL))
   {
      Candera::Vector2 adornerParentPos(groupData->AdornerHandle->getNode()->GetParent()->GetWorldPosition());

      _adornerManager.setAnimationDuration(*_adornerAnimation, (animationDuration > 0) ? animationDuration : _adornerManager.getDefaultAnimationDuration());

      _adornerManager.attachAnimationToAdorner(*_adornerAnimation, groupData->AdornerHandle);
      _adornerManager.setAnimationBeginValues(*_adornerAnimation, startPos - adornerParentPos);

      Candera::Vector2 endPosition = endWidget.GetNode()->GetWorldPosition();
      _adornerManager.setAnimationEndValues(*_adornerAnimation, endPosition - adornerParentPos);

      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "StartAnimation startPosition=%12s endPosition=%s",
                          HMIBASE_TO_STRING(startPos),
                          HMIBASE_TO_STRING(endPosition)));

      _adornerManager.startAnimation(*_adornerAnimation);
   }
}


/*****************************************************************************/
void ButtonSlideController2D::onAnimationEnded(hmibase::widget::adorner::AdornerAnimation* animation)
{
   if ((_adornerAnimation != NULL) && (_adornerAnimation == animation))
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnAnimationEnded"));

      if (!_adornerAnimation->getAdorner().PointsToNull())
      {
         bool buttonAvailable = false;
         ButtonWidget2D* button = Candera::Dynamic_Cast<ButtonWidget2D*>(_adornerAnimation->getAdorner()->getSourceWidget());
         if (button != NULL)
         {
            ButtonSlideController2DData* buttonData = Candera::Dynamic_Cast<ButtonSlideController2DData*>(button->GetControllerData());
            if ((buttonData != NULL) && (!buttonData->GroupData.PointsToNull()))
            {
               buttonAvailable = true;
               scheduleAdornerHide(*button, *buttonData);
            }
         }
         if (!buttonAvailable)
         {
            ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "OnAnimationEnded widget not available anymore for adorner!"));
            _adornerAnimation->getAdorner()->hide();
         }
      }

      _adornerManager.detachAnimationFromAdorner(*_adornerAnimation);
   }
}
