/* ***************************************************************************************
* FILE:          ButtonBubbleController2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ButtonBubbleController2D.cpp is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "widget2D_std_if.h"
#include "ButtonBubbleController2D.h"

#include <CanderaPlatform/Device/Common/Effects/TextBrush.h>

#include <Widgets/2D/WidgetFinder2D.h>
#include <Widgets/2D/Button/ButtonWidget2D.h>
#include <Widgets/2D/ControlTemplate/TTraverserBase.h>
#include <Widgets/2D/Marker/MarkerWidget2D.h>
#include <Widgets/utils/WidgetFunctors.h>
#include <Widgets/utils/WidgetTraverser.h>

#include <Trace/ToString.h>
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_BUTTON
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ButtonBubbleController2D.cpp.trc.h"
#endif


using ::Candera::Char;//required for CANDERA_RTTI_DEFINITION
CANDERA_RTTI_DEFINITION(ButtonBubbleController2D)

using namespace hmibase::widget::adorner;

/*****************************************************************************/
ButtonBubbleController2D::ButtonBubbleController2D(AdornerManager& adornerManager, const AdornerMarkerFilter& adornerMarkerFilter)
   : _adornerManager(adornerManager), _adornerMarkerFilter(adornerMarkerFilter), _adornerHideTimer(NULL), _adornerHideTimerTimeout(0), _bubbleEnabled(true), _bubbleShowReaction(enPress)
{
}


/*****************************************************************************/
ButtonBubbleController2D::~ButtonBubbleController2D()
{
   if (_adornerHideTimer != NULL)
   {
      stopAdornerHideTimer();
      CANDERA_DELETE(_adornerHideTimer);
      _adornerHideTimer = NULL;
   }
}


/*****************************************************************************/
void ButtonBubbleController2D::setBubbleEnabled(bool enabled)
{
   _bubbleEnabled = enabled;
}


/*****************************************************************************/
void ButtonBubbleController2D::setBubbleShowReaction(enReaction reaction)
{
   _bubbleShowReaction = reaction;
}


/*****************************************************************************/
void ButtonBubbleController2D::setAdornerHideTimerTimeout(unsigned int timeout)
{
   _adornerHideTimerTimeout = timeout;
}


/*****************************************************************************/
void ButtonBubbleController2D::startAdornerHideTimer()
{
   if (_adornerHideTimerTimeout > 0)
   {
      if (_adornerHideTimer == NULL)
      {
         _adornerHideTimer = CANDERA_NEW(Util::Timer);
      }
      if (_adornerHideTimer != NULL)
      {
         _adornerHideTimer->setName("ButtonBubbleAdornerHideTimer", 0);

         _adornerHideTimer->setTimeout(0, _adornerHideTimerTimeout);

         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "StartAdornerHideTimer timeout=%u", _adornerHideTimerTimeout));
         _adornerHideTimer->start();
      }
   }
}


/*****************************************************************************/
void ButtonBubbleController2D::stopAdornerHideTimer()
{
   if (_adornerHideTimer != NULL)
   {
      if (_adornerHideTimer->running())
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "StopAdornerHideTimer"));
      }
      _adornerHideTimer->stop();
   }
}


/*****************************************************************************/
bool ButtonBubbleController2D::OnMessage(DelegateWidget& widget, const Courier::Message& msg)
{
   if (Base::OnMessage(widget, msg))
   {
      return true;
   }

   bool consumed = false;
   switch (msg.GetId())
   {
      case TimerExpiredMsg::ID:
      {
         const TimerExpiredMsg* timerExpiredMsg = Courier::message_cast<const TimerExpiredMsg*>(&msg);
         if ((timerExpiredMsg != NULL) && (timerExpiredMsg->GetTimer() == _adornerHideTimer))
         {
            onAdornerHideTimerExpired(widget);
            consumed = true;
         }
      }
      break;

      default:
         break;
   }

   return consumed;
}


/*****************************************************************************/
void ButtonBubbleController2D::onAdornerHideTimerExpired(DelegateWidget& widget)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnAdornerHideTimerExpired %s", HMIBASE_TO_STRING_VW(&widget)));

   stopAdornerHideTimer();
   hideAdorner();
}


/*****************************************************************************/
bool ButtonBubbleController2D::createAdorner(ButtonWidget2D& button)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "CreateAdorner %s", HMIBASE_TO_STRING_VW(&button)));

   Candera::Node2D* adornerNode = NULL;

   //if an adorner marker filter is specified use it to find the adorner node
   if (_adornerMarkerFilter != AdornerMarkerFilter())
   {
      MarkerWidget2D* adornerMarker = MarkerWidget2D::findMarker(hmibase::widget::utils::MessageUtils::getSceneContext(&button), _adornerMarkerFilter);
      if (adornerMarker != NULL)
      {
         adornerNode = adornerMarker->GetNode();
      }
   }

   _adorner = _adornerManager.createAdorner(button, adornerNode);

   return !_adorner.PointsToNull();
}


/*****************************************************************************/
void ButtonBubbleController2D::showAdorner(ButtonWidget2D& button)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ShowAdorner %s", HMIBASE_TO_STRING_VW(&button)));

   if (!_adorner.PointsToNull())
   {
      hideAdorner();
   }

   //create new adorner
   createAdorner(button);
   //show the adorner
   _adornerManager.showAdorner(_adorner);
}


/*****************************************************************************/
void ButtonBubbleController2D::hideAdorner()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "HideAdorner"));

   _adornerManager.hideAdorner(_adorner);
   _adornerManager.destroyAdorner(_adorner);
   _adorner.Release();
}


/*****************************************************************************/
bool ButtonBubbleController2D::onReaction(ButtonWidget2D& button, enReaction reaction)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnReaction reaction=%u expectedReaction=%u bubbleEnabled=%u %s",
                       reaction, getBubbleShowReaction(), isBubbleEnabled(), HMIBASE_TO_STRING_VW(&button)));

   stopAdornerHideTimer();

   if (isBubbleEnabled() && (reaction == getBubbleShowReaction()))
   {
      showAdorner(button);
      startAdornerHideTimer();
   }
   else if (!_adorner.PointsToNull())
   {
      hideAdorner();
   }
   else
   {
      //nothing to do
   }

   //return false to allow default handling in the button widget
   return false;
}


/*****************************************************************************/
class NodeTextBrushTraverser : public TTraverserBase<Candera::Node2D>
{
      typedef TTraverserBase<Candera::Node2D> Base;

   public:
      NodeTextBrushTraverser() : Base(), _readMode(true)
      {
      }

      bool isReadMode() const
      {
         return _readMode;
      }
      void setReadMode(bool val)
      {
         _readMode = val;
      }
      const std::string getText() const
      {
         return _text;
      }
      void setText(const std::string& text)
      {
         _text = text;
      }

   protected:
      /*****************************************************************************/
      virtual TraverserBase::TraverserAction ProcessNode(Candera::Node2D& node)
      {
         Candera::RenderNode* renderNode = Candera::Dynamic_Cast<Candera::RenderNode*>(&node);
         if (renderNode != NULL)
         {
            Candera::Effect2D* effect = renderNode->GetEffect(0);
            if (effect != NULL)
            {
               Candera::TextBrush* textBrush = Candera::Dynamic_Cast<Candera::TextBrush*>(effect->GetBrushEffect2D());
               if (textBrush != NULL)
               {
                  //read mode => get text from the brush
                  if (_readMode)
                  {
                     if (textBrush->Text().Get() != NULL)
                     {
                        _text = textBrush->Text().Get();
                     }
                     else
                     {
                        _text.clear();
                     }

                     ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "NodeTextBrushTraverser get text '%20s' from node %s",
                                         _text.c_str(), HMIBASE_TO_STRING_PN2D(&node)));
                  }
                  //write mode => set text on the brush
                  else
                  {
                     if (_text.empty())
                     {
                        textBrush->Text().Set("", NULL);
                     }
                     else
                     {
                        FeatStd::TChar* buffer = FEATSTD_NEW_ARRAY(FeatStd::TChar, FeatStd::Internal::String::Length(_text.c_str()) + 1);
                        if (buffer != NULL)
                        {
                           FeatStd::Internal::String::Copy(buffer, _text.c_str());
                           textBrush->Text().Set(buffer, FeatStd::MemoryManagement::ArrayDisposer<const FeatStd::TChar*>::Dispose);
                        }
                     }

                     ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "NodeTextBrushTraverser set text '%20s' on node %s",
                                         _text.c_str(), HMIBASE_TO_STRING_PN2D(&node)));
                  }
                  return TraverserBase::StopTraversing;
               }
            }
         }
         return TraverserBase::ProceedTraversing;
      }

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(NodeTextBrushTraverser);
      std::string _text;
      bool _readMode;
};


/*****************************************************************************/
Candera::Node2D* ButtonBubbleAdornerManager::prepareAdornerNodeContent(Candera::Widget2D& widget, Candera::Node2D* adornerNodeTemplate)
{
   Candera::Node2D* node = Base::prepareAdornerNodeContent(widget, adornerNodeTemplate);

   if ((node != NULL) && (widget.GetNode() != NULL))
   {
      NodeTextBrushTraverser traverser;

      //get text from the original widget
      traverser.setReadMode(true);
      traverser.Traverse(*(widget.GetNode()));

      //set text on the adorner node
      traverser.setReadMode(false);
      traverser.Traverse(*node);
   }

   return node;
}


/*****************************************************************************/
Candera::Rectangle ButtonBubbleAdornerManager::calculateAdornerNodeBounds(Candera::Widget2D& widget, AdornerNode* adornerNode, Candera::Node2D* adornerNodeContent, Candera::Node2D* /*adornerContainer*/)
{
   Candera::Rectangle adornerNodeBounds;
   if ((widget.GetNode() != NULL) && (adornerNodeContent != NULL) && (adornerNode != NULL) && (adornerNode->GetParent() != NULL))
   {
      Candera::Rectangle widgetRect;
      widget.GetNode()->GetWorldAxisAlignedBoundingRectangle(widgetRect);

      Candera::Rectangle adornerRect;
      adornerNodeContent->GetEffectiveBoundingRectangle(adornerRect);

      Candera::Vector2 adornerParentPosition(adornerNode->GetParent()->GetWorldPosition());

      //align adorner center to widget node center
      float xPos = widgetRect.GetLeft() + widgetRect.GetWidth() / 2 - adornerRect.GetWidth() / 2 - adornerParentPosition.GetX();

      //align adorner bottom to widget node top
      float yPos = widgetRect.GetTop() - adornerRect.GetHeight() - adornerParentPosition.GetY();

      adornerNodeBounds.SetPosition(xPos, yPos);

      ETG_TRACE_USR1(("PrepareAdornerNode pos=[%d,%d]", static_cast<int>(xPos), static_cast<int>(yPos)));
   }

   return adornerNodeBounds;
}
