/* ***************************************************************************************
* FILE:          ButtonAnimatedReleaseController2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ButtonAnimatedReleaseController2D.cpp is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "widget2D_std_if.h"
#include "ButtonAnimatedReleaseController2D.h"

#include <Courier/Visualization/ViewScene2D.h>

#include <Widgets/2D/ButtonAnimation/ButtonAnimationWidget2D.h>
#include <Widgets/utils/WidgetTraverser.h>

#include <Trace/ToString.h>
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_BUTTON
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ButtonAnimatedReleaseController2D.cpp.trc.h"
#endif


using ::Candera::Char;//required for CANDERA_RTTI_DEFINITION
CANDERA_RTTI_DEFINITION(ButtonAnimatedReleaseController2D)


/*****************************************************************************/
ButtonAnimatedReleaseController2D& ButtonAnimatedReleaseController2D::getInstance()
{
   static ButtonAnimatedReleaseController2D _instance;
   return _instance;
}


/*****************************************************************************/
ButtonAnimatedReleaseController2D::ButtonAnimatedReleaseController2D() : _controllerEnabled(true), _buttonReleased(false)
{
}


/*****************************************************************************/
ButtonAnimatedReleaseController2D::~ButtonAnimatedReleaseController2D()
{
}


/*****************************************************************************/
bool ButtonAnimatedReleaseController2D::OnMessage(DelegateWidget& widget, const Courier::Message& msg)
{
   if (Base::OnMessage(widget, msg))
   {
      return true;
   }

   bool consumed = false;
   switch (msg.GetId())
   {
      case AnimationWidgetIndMsg::ID:
      {
         ButtonWidget2D* button = Candera::Dynamic_Cast<ButtonWidget2D*>(&widget);
         const AnimationWidgetIndMsg* animationIndMsg = Courier::message_cast<const AnimationWidgetIndMsg*>(&msg);
         if ((button != NULL) && (animationIndMsg != NULL))
         {
            consumed = onAnimationFinishedMsg(*button, *animationIndMsg);
         }
      }
      break;

      default:
         break;
   }

   return consumed;
}


/*****************************************************************************/
class ButtonAnimationFinder : private WidgetCheckCallback
{
   public:
      static ButtonAnimationWidget2D* Find(Courier::ViewScene2D& view, Candera::Node2D& buttonNode, Courier::Identifier buttonAnimationId)
      {
         ButtonAnimationFinder callback(buttonAnimationId);

         //post a widget check request message to all descendant widgets
         hmibase::widget::utils::MessageUtils::distributeToDescendants(view.GetScene2DContext(), buttonNode, WidgetCheckReqMsg(&callback));

         return callback._buttonAnimation;
      }

   private:
      explicit ButtonAnimationFinder(const Courier::Identifier& widgetId) : _buttonAnimationId(widgetId), _buttonAnimation(NULL)
      {
      }

      virtual ~ButtonAnimationFinder()
      {
         _buttonAnimation = NULL;
      }

      virtual bool CheckWidget(Candera::Widget2D* widget)
      {
         //check if id of the widget matches our animation widget id
         if ((widget != NULL) && (_buttonAnimationId == Courier::Identifier(widget->GetLegacyName())))
         {
            _buttonAnimation = Candera::Dynamic_Cast<ButtonAnimationWidget2D*>(widget);

            //checking of widgets stops when the message is consumed (this method returns true)
            return (_buttonAnimation != NULL);
         }
         return false;
      }

      Courier::Identifier _buttonAnimationId;
      ButtonAnimationWidget2D* _buttonAnimation;
};


/*****************************************************************************/
ButtonAnimationWidget2D* ButtonAnimatedReleaseController2D::findButtonAnimation(Courier::ViewScene2D& view, Candera::Node2D& buttonNode, Courier::Identifier buttonAnimationId)
{
   return ButtonAnimationFinder::Find(view, buttonNode, buttonAnimationId);
}


/*****************************************************************************/
bool ButtonAnimatedReleaseController2D::onAnimationFinishedMsg(ButtonWidget2D& button, const AnimationWidgetIndMsg& animationIndMsg)
{
   bool consumed = false;

   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "onAnimationFinishedMsg buttonReleased=%u animationId=%15s %s",
                       _buttonReleased,
                       HMIBASE_TO_STRING(animationIndMsg.GetSender()),
                       HMIBASE_TO_STRING_VW(&button)));

   Courier::View* view = button.GetParentView();
   if (_buttonReleased && (button.GetNode() != NULL) && (view != NULL) && (view->ToViewScene2D() != NULL) && (view->GetId() == animationIndMsg.GetView()))
   {
      //find the animation widget which posted the animation indication (finished) message by checking the descendants of the button's node
      ButtonAnimationWidget2D* buttonAnimation = ButtonAnimationFinder::Find(*(view->ToViewScene2D()), *(button.GetNode()), animationIndMsg.GetSender());

      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "OnAnimationFinishedMsg buttonAnimation=%50s %s",
                          HMIBASE_TO_STRING_W(buttonAnimation), HMIBASE_TO_STRING_VW(&button)));

      //if the button animation matches this button and if it is not pressed
      if ((buttonAnimation != NULL) && (buttonAnimation->GetButtonWidget() == &button))
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnAnimationFinishedMsg post reaction %s", HMIBASE_TO_STRING_VW(&button)));

         //if (!button.IsPressed())
         {
            button.postReactionMessage(enRelease);
         }

         //consume the message and reset the released flag
         _buttonReleased = false;
         consumed = true;
      }
   }
   return consumed;
}


/*****************************************************************************/
bool ButtonAnimatedReleaseController2D::onReaction(ButtonWidget2D& button, enReaction reaction)
{
   //consume the release reaction so that the reaction message is posted only after the animation is finished
   if (isEnabled() && (reaction == enRelease))
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnReaction delay reaction %s", HMIBASE_TO_STRING_VW(&button)));
      _buttonReleased = true;

      // return true (consume the reaction)
      return true;
   }

   //return false to allow default handling in the button widget
   return false;
}
