/* ***************************************************************************************
* FILE:          BusyIndicatorWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  BusyIndicatorWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "Widgets/2D/BaseWidget2D.h"
#include "BusyIndicatorWidget2D.h"

using namespace Candera;
using Courier::message_cast;

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_ANIMATION
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/BusyIndicatorWidget2D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(BusyIndicatorWidget2D)

static const FeatStd::Int c_repeatCount = 1;

/****************************************************************************
CONSTRUCTOR
****************************************************************************/
BusyIndicatorWidget2D::BusyIndicatorWidget2D():
   m_bMinimumAnimationDuration_TimerExpired(true),
   m_u8DotArrayIndex(1),
   m_u8IncCirNodeCount(2),
   m_IncCirAnimationValues(NULL),
   m_pCameraNode(NULL),
   m_pAnimationController(NULL),
   m_pAlphaPS(NULL),
   m_pScalePS(NULL),
   m_IncCirAnimationKfs(NULL),
   m_pMainNode(NULL),
   m_pNode(NULL),
   m_pAnimationTimeDispatcher(NULL),
   m_TextWidgetPtr(NULL),
   m_BusyIndicator_Timer(NULL)

   /*------------------------------------------------------------------------
     Text Releated variables initialization
   ------------------------------------------------------------------------*/
{
   stDot.Add(" ");
   stDot.Add(".");
   stDot.Add("..");
   stDot.Add("...");
   memset(m_BusyIndicatorAnimationKfs, 0, sizeof(m_BusyIndicatorAnimationKfs));
   memset(m_BusyIndicatorAnimationValues, 0, sizeof(m_BusyIndicatorAnimationValues));
}


/****************************************************************************
DESTRUCTOR
****************************************************************************/
BusyIndicatorWidget2D::~BusyIndicatorWidget2D()
{
   if (m_pAnimationTimeDispatcher != 0)
   {
      m_pAnimationTimeDispatcher->RemovePlayer(m_pAnimPlayer);
   }

   if (m_pAnimationController != 0)
   {
      const ::FeatStd::Int32 count = static_cast<FeatStd::Int32>(m_pAnimationController->GetNumberOfProperties());

      for (::FeatStd::Int32 i = 0; i < count; i++)
      {
         m_pAnimationController->RemoveProperty(i);
      }
   }

   if (!m_pAnimPlayer.PointsToNull())
   {
      m_pAnimPlayer.Release();
   }

   m_animScaleProperty.Clear();
   m_animAlphaProperty.Clear();
   FEATSTD_SAFE_DELETE_ARRAY(m_IncCirAnimationKfs);
   FEATSTD_SAFE_DELETE_ARRAY(m_IncCirAnimationValues);
   FEATSTD_SAFE_DELETE_ARRAY(m_pAlphaPS);
   FEATSTD_SAFE_DELETE_ARRAY(m_pScalePS);
   FEATSTD_SAFE_DELETE(m_BusyIndicator_Timer);
   m_pCameraNode = NULL;
   m_pMainNode = NULL;
   m_pNode = NULL;
   m_pAnimationTimeDispatcher = NULL;
   m_TextWidgetPtr = NULL;
}


/****************************************************************************
*   Function    : Init
*   Description : Initializes the widget so that all referred resource can be
*              resolved
*   Parameters  : Candera::AssetProvider*
*   Return      : void
****************************************************************************/
void BusyIndicatorWidget2D::InitWidget()
{
   Base::InitWidget();
   if (NULL != GetNode())
   {
      // Text initialize
      m_TextWidgetPtr = ::Candera::Dynamic_Cast<TextWidget2D*>(GetTextWidget());
      m_sText = GetText();
      m_pMainNode = GetNode();

      if (NULL != m_pMainNode)
      {
         m_pNode = m_pMainNode->GetFirstChild();
      }
      InitAnimations();
   }
}


/****************************************************************************
*   Function    : StopBusyWaitAnimation
*   Description : Stop the busy wait animation
*   Parameters  : void
*   Return      : void
****************************************************************************/
void BusyIndicatorWidget2D::StopBusyIndicatorAnimation()
{
   if (!m_pAnimPlayer.PointsToNull())
   {
      if (m_pAnimPlayer->IsEnabled())
      {
         m_pAnimPlayer->Stop();
      }

      // Disable the Rendering of the Nodes
      if (NULL != m_pMainNode)
      {
         m_pMainNode->SetRenderingEnabled(false);
      }

      if (m_TextWidgetPtr != NULL)
      {
         m_TextWidgetPtr->GetNode()->SetRenderingEnabled(false);
      }
   }
}


/****************************************************************************
*   Function    : StartBusyWaitAnimation
*   Description : Start the busy wait animation
*   Parameters  : void
*   Return      : void
****************************************************************************/
void BusyIndicatorWidget2D::StartBusyIndicatorAnimation()
{
   if (!m_pAnimPlayer.PointsToNull())
   {
      if (m_pAnimPlayer->IsEnabled())
      {
         m_pAnimPlayer->Stop();
      }

      // Enable the Rendering of the Nodes
      if (NULL != m_pMainNode)
      {
         m_pMainNode->SetRenderingEnabled(true);
      }

      if (m_TextWidgetPtr != NULL)
      {
         m_TextWidgetPtr->GetNode()->SetRenderingEnabled(true);
      }

      m_pAnimPlayer->Start();
   }
}


/****************************************************************************
*   Function    : Update
*   Description :
*   Parameters  :
*   Return      :
****************************************************************************/
void BusyIndicatorWidget2D::Update()
{
   Base::Update();

   if (!m_pAnimPlayer.PointsToNull())
   {
      if (m_pAnimPlayer->IsEnabled())
      {
         Invalidate();
      }
   }
}


/****************************************************************************
*   Function    :
*   Description :
*   Parameters  :
*   Return      :
****************************************************************************/
void BusyIndicatorWidget2D::InitAnimations()
{
   if (NULL != m_pNode)
   {
      if (GetBusyWaitIndicatorType() == Candera::IncrementallyCircularBusyIndicator)
      {
         if (InitIncCirNodeInfo())
         {
            InitIncCirBusyIndicatorAnimation();
         }
      }
      else
      {
         InitCirularOrHorizontalBusyIndicatorAnimation();
      }
   }

   if (GetMinimumAnimationDuration() > 0)
   {
      m_BusyIndicator_Timer = FEATSTD_NEW(::Util::Timer); // Create timer only if Min Duration is greater than 0.

      if (m_BusyIndicator_Timer != NULL)
      {
         m_BusyIndicator_Timer->setTimeout(0, GetMinimumAnimationDuration());
      }
   }
}


/****************************************************************************
*   Function    : InitIncCirBusyIndicatorAnimation
*   Description : Initialize Incremental Circular Busy indicator
*   Parameters  :
*   Return      :
****************************************************************************/
void BusyIndicatorWidget2D::InitIncCirBusyIndicatorAnimation()
{
   CreateIncCirKeyFrames();
   CreateAnimationKeyFrameSequence();
   CreateAnimationProperties();
   CreateAnimationController();
   CreateAnimationPlayer();
}


/****************************************************************************
*   Function    : InitCirularOrHorizontalBusyIndicatorAnimation
*   Description : Initialize Cirular and Horizontal type of busy indicator
*   Parameters  :
*   Return      : void
****************************************************************************/
void BusyIndicatorWidget2D::InitCirularOrHorizontalBusyIndicatorAnimation()
{
   if (GetBusyWaitIndicatorType() == CircularBusyIndicator)
   {
      CreateCircularBusyKeyFrames();
   }
   else if (GetBusyWaitIndicatorType() == HorizontalBusyIndicator)
   {
      // Get Camera pointer
      if (NULL != m_pNode)
      {
         Candera::Node2D* nextSiblingNode = m_pNode->GetNextSibling();

         if (NULL != nextSiblingNode)
         {
            if (nextSiblingNode->IsTypeOf(Candera::Camera2D::GetTypeId()))
            {
               m_pCameraNode = ::Candera::Dynamic_Cast<Candera::Camera2D*>(m_pNode->GetNextSibling());
               CreateHorizontalBusyKeyFrames();
            }
            else
            {
               ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "BusyIndicatorWidget2D : The First child of the Node associated to the Widget for Horizontal BusyIndicator Type is NOT a camera"));
            }
         }
      }
   }

   CreateAnimationKeyFrameSequence();
   CreateAnimationProperties();
   CreateAnimationController();
   CreateAnimationPlayer();
}


/****************************************************************************
*   Function    : CreateIncCirKeyFrames
*   Description :
*   Parameters  : void
*   Return      : void
****************************************************************************/
void BusyIndicatorWidget2D::CreateIncCirKeyFrames()
{
   m_pAlphaPS = FEATSTD_NEW_ARRAY(Candera::Animation::AlphaNode2DPropertySetter, m_u8IncCirNodeCount);

   if (GetEnableScaling())
   {
      m_pScalePS = FEATSTD_NEW_ARRAY(Animation::Scale2DPropertySetter, m_u8IncCirNodeCount);
   }

   const ::FeatStd::UInt32 uiAnimationValueSize = m_u8IncCirNodeCount * m_u8IncCirNodeCount;
   const ::FeatStd::UInt32 fAnimationKfUnit     = (GetAnimationDuration() / (m_u8IncCirNodeCount - 1));
   const ::FeatStd::Float  fAnimationValueUnit  = (1 / (::Candera::Float)m_u8IncCirNodeCount);

   if (NULL != m_IncCirAnimationValues)
   {
      FEATSTD_DELETE_ARRAY(m_IncCirAnimationValues);
   }

   m_IncCirAnimationValues = FEATSTD_NEW_ARRAY(::Candera::Float, uiAnimationValueSize);

   if (NULL != m_IncCirAnimationKfs)
   {
      FEATSTD_DELETE_ARRAY(m_IncCirAnimationKfs);
   }

   m_IncCirAnimationKfs = FEATSTD_NEW_ARRAY(Candera::Animation::SequenceTimeType, m_u8IncCirNodeCount);

   // Adjust the pivot offset
   if (GetEnableScaling())
   {
      AdjustNodePivotOffset();
   }

   // Calculate the Time Sequence
   for (FeatStd::UInt8 i = 0; i < m_u8IncCirNodeCount; i++)
   {
      m_IncCirAnimationKfs[i] = ::FeatStd::Int32(i * fAnimationKfUnit);
   }

   // Calculate the Alpha and Scale Animation values
   FeatStd::UInt8 u_8AnimationValueIndex = 0;

   for (; u_8AnimationValueIndex < m_u8IncCirNodeCount; u_8AnimationValueIndex++)
   {
      if (m_IncCirAnimationValues != NULL)
      {
         m_IncCirAnimationValues[u_8AnimationValueIndex] = (static_cast<Candera::Float>(u_8AnimationValueIndex) + 1.0F) * fAnimationValueUnit;
      }
   }

   FeatStd::UInt8 multiplyFactor = 2;

   while (u_8AnimationValueIndex < uiAnimationValueSize)
   {
      for (FeatStd::UInt8 i = 0; i < m_u8IncCirNodeCount; i++, u_8AnimationValueIndex++, multiplyFactor++)
      {
         if (multiplyFactor > m_u8IncCirNodeCount)
         {
            multiplyFactor = multiplyFactor % UInt8(m_u8IncCirNodeCount);
         }

         if (m_IncCirAnimationValues != NULL)
         {
            m_IncCirAnimationValues[u_8AnimationValueIndex] = fAnimationValueUnit * multiplyFactor;
         }
      }

      multiplyFactor++;
   }

   // ![Calculate the Alpha and Scale Animation values]
}


/****************************************************************************
*   Function    :
*   Description :
*   Parameters  :
*   Return      :
****************************************************************************/
void BusyIndicatorWidget2D::CreateCircularBusyKeyFrames()
{
   m_BusyIndicatorAnimationKfs[0] = 0;
   m_BusyIndicatorAnimationKfs[1] = ::FeatStd::Int32(GetAnimationDuration());
   m_BusyIndicatorAnimationValues[0] = 0.0F;
   m_BusyIndicatorAnimationValues[1] = 360.0F;
   // get the node width and height
   Candera::Rectangle nodeBoundingRect;

   if (NULL != m_pNode)
   {
      m_pNode->GetEffectiveBoundingRectangle(nodeBoundingRect);
      Candera::Vector2 nodePos = m_pNode->GetPosition();
      ::Candera::Float flHalfNodeWidth = (nodeBoundingRect.GetWidth()) / 2;
      ::Candera::Float flHalfNodeHeight = (nodeBoundingRect.GetHeight()) / 2;
      // set the pivot offset for the node
      m_pivotOffset.SetX(-flHalfNodeWidth);
      m_pivotOffset.SetY(-flHalfNodeHeight);
      nodePos.SetX(nodePos.GetX() + flHalfNodeWidth);
      nodePos.SetY(nodePos.GetY() + flHalfNodeHeight);
      m_pNode->SetPivotOffset(m_pivotOffset);
      m_pNode->SetPosition(nodePos);
   }
}


/****************************************************************************
*   Function    :
*   Description :
*   Parameters  :
*   Return      :
****************************************************************************/
void BusyIndicatorWidget2D::CreateHorizontalBusyKeyFrames()
{
   // Get the bitmap nodeWidth
   Candera::Rectangle nodeBoundingRect;

   if ((NULL != m_pNode) && (NULL != m_pCameraNode))
   {
      m_pNode->GetEffectiveBoundingRectangle(nodeBoundingRect);
      ::Candera::Float flNodeWidth = nodeBoundingRect.GetWidth();
      Candera::Vector2 flCameraInitialPos = m_pNode->GetPosition();
      //m_pCameraNode->SetPosition(flCameraInitialPos.GetX(), flCameraInitialPos.GetY());
      ::Candera::Float flCameraViewport = m_pCameraNode->GetViewport().GetWidth();
      ::Candera::Float flCameraFinalPos = flCameraInitialPos.GetX() + flNodeWidth - flCameraViewport;
      m_BusyIndicatorAnimationKfs[0] = 0;
      m_BusyIndicatorAnimationKfs[1] = ::FeatStd::Int32(GetAnimationDuration());
      m_BusyIndicatorAnimationValues[0] = flCameraInitialPos.GetX();
      m_BusyIndicatorAnimationValues[1] = flCameraFinalPos;
   }
}


/****************************************************************************
*   Function    :
*   Description :
*   Parameters  :
*   Return      :
****************************************************************************/
void BusyIndicatorWidget2D::OnAnimationTimeDispatcherChanged()
{
   if ((NULL != m_pAnimationTimeDispatcher) && (!m_pAnimPlayer.PointsToNull()))
   {
      m_pAnimationTimeDispatcher->AddPlayer(m_pAnimPlayer);
      m_pAnimPlayer->AddAnimationPlayerListener(this);
   }
}


/****************************************************************************
*       Function    : OnMessage
*       Description : Courier messages will be routed to this function.
*                 If widget is interested in any message then, on
*                 receiving message will act accordigly.
*       Parameters  : Message object to be processed.
*       Return      : true if the message is consumed
*                false if the message should be forwarded
****************************************************************************/
bool BusyIndicatorWidget2D::OnMessage(const Message& msg)
{
   bool bMsgConsumed = false;

   switch (msg.GetId())
   {
      case TimerExpiredMsg::ID :
      {
         const TimerExpiredMsg* timerMsg = message_cast<const TimerExpiredMsg*>(&msg);

         if ((timerMsg != 0) && (m_BusyIndicator_Timer != 0))
         {
            // TODO - Add a unique timer ID for each instance of the widget
            if ((timerMsg->GetTimer() == m_BusyIndicator_Timer) && (m_BusyIndicator_Timer->getStatus() != ::Util::Timer::Stopped))
            {
               m_bMinimumAnimationDuration_TimerExpired = true;
               ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BusyIndicatorWidget2D: [%s] Message is received.", GetLegacyName()));
               bMsgConsumed = true;
            }
         }

         break;
      }

      case AnimationWidgetReqMsg::ID:
      {
         const AnimationWidgetReqMsg* animationReqMsg = message_cast<const AnimationWidgetReqMsg*>(&msg);

         if (animationReqMsg != NULL)
         {
            if (Courier::Identifier(GetLegacyName()) == animationReqMsg->GetSender() &&
                  GetParentView()->GetId() == animationReqMsg->GetView())
            {
               if (animationReqMsg->GetAnimationAction() == Courier::AnimationAction ::Start)
               {
                  StartBusyIndicatorTimer();
                  StartBusyIndicatorAnimation();
                  bMsgConsumed = true;
               }
               else if (animationReqMsg->GetAnimationAction() == Courier::AnimationAction::Stop)
               {
                  if (m_bMinimumAnimationDuration_TimerExpired)
                  {
                     StopBusyIndicatorAnimation();
                  }
                  bMsgConsumed = true;
               }
            }
         }

         break;
      }

      default:
         break;
   }   // end of switch case

   return bMsgConsumed;
}


/****************************************************************************
*   Function    :
*   Description :
*   Parameters  :
*   Return      :
****************************************************************************/
void BusyIndicatorWidget2D::OnPastEnd(Candera::Animation::AnimationPlayerBase* /*animationPlayer*/, ::FeatStd::Int32 /*completedIterationsCount*/)
{
   StartBusyIndicatorAnimation();

   if (m_sText.GetCharCount() != 0)
   {
      FeatStd::TChar* tempText = FEATSTD_NEW_ARRAY(TChar, m_sText.GetCharCount() + stDot[m_u8DotArrayIndex].GetCharCount() + 1);

      if (tempText)
      {
         SECURE_FEATSTD_STRING_ACCESS_BEGIN(m_sText);
         SECURE_FEATSTD_STRING_ACCESS_BEGIN(stDot[m_u8DotArrayIndex]);
         Candera::StringPlatform::Copy((Char*)tempText, m_sText.GetCString());
         Candera::StringPlatform::Copy((Char*)tempText + Candera::StringPlatform::Length(m_sText.GetCString()), stDot[m_u8DotArrayIndex].GetCString());
         SECURE_FEATSTD_STRING_ACCESS_END();
         SECURE_FEATSTD_STRING_ACCESS_END();
         m_u8DotArrayIndex++;

         if (m_TextWidgetPtr != NULL)
         {
            m_TextWidgetPtr->SetText(tempText);
         }

         if (m_u8DotArrayIndex > 3)
         {
            m_u8DotArrayIndex = 0;
         }

         FEATSTD_DELETE_ARRAY(tempText);
      }
   }
}


/****************************************************************************
*   Function    : InitIncCirNodeInfo
*   Description : Retrieve the Node pointers for the Incrementing Circular
Busy Indicator
*   Parameters  :    void
*   Return      : bool
****************************************************************************/
bool BusyIndicatorWidget2D::InitIncCirNodeInfo()
{
   Candera::Node2D* tempNode = NULL;

   if (NULL != m_pNode)
   {
      tempNode = m_pNode;
   }

   if (NULL != tempNode)
   {
      // Get the node count
      if (m_pMainNode != NULL)
      {
         m_u8IncCirNodeCount = m_pMainNode->GetChildCount();
      }

      if (m_u8IncCirNodeCount > 1) // node count should be minimum 2
      {
         for (FeatStd::UInt32 i = 0; i < m_u8IncCirNodeCount && tempNode != NULL; ++i)
         {
            m_ptCircularBusyIndicatorNodePtr.Add(::Candera::Dynamic_Cast<Candera::RenderNode*>(tempNode));
            tempNode = tempNode->GetNextSibling();
         }

         return true;
      }
      else
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "BusyIndicatorWidget : No child nodes present for Incremental circular BusyIndicator"));
         return false;
      }
   }
   else
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "BusyIndicatorWidget : No ChildNodes present"));
      return false;
   }
}


/******************************************************************************
*       Function      : CreateAnimationKeyFrameSequence
*       Description   :
*       Parameters     :
*       Return        : void
*****************************************************************************/
void BusyIndicatorWidget2D::CreateAnimationKeyFrameSequence()
{
   if (GetBusyWaitIndicatorType() == Candera::IncrementallyCircularBusyIndicator)
   {
      CreateAlphaAnimationKeyFrameSequence();

      if (GetEnableScaling())
      {
         CreateScaleAnimationKeyFrameSequence();
      }
   }
   else
   {
      m_pBusyIndicatorKfs = Candera::Animation::AnimationKeyframeSequence::Create();

      if (!m_pBusyIndicatorKfs.PointsToNull())
      {
         m_pBusyIndicatorKfs->SetKeyframes(1, 2 , m_BusyIndicatorAnimationKfs, 0, m_BusyIndicatorAnimationValues, 0);
      }
   }
}


/******************************************************************************
*       Function      : CreateAnimationProperties
*       Description   :
*       Parameters     :
*       Return        : void
*****************************************************************************/
void BusyIndicatorWidget2D::CreateAnimationProperties()
{
   switch (GetBusyWaitIndicatorType())
   {
      // [switch]
      case Candera::IncrementallyCircularBusyIndicator :
         if (GetEnableScaling())
         {
            CreateScalingAnimationProperties();
         }

         CreateAlphaAnimationProperties();
         break;

      case Candera::CircularBusyIndicator:
         m_pAnimProperty = Candera::Animation::AnimationBlendedProperty::Create();

         if (!m_pAnimProperty.PointsToNull())
         {
            m_pAnimProperty->AddKeyframeSequence(m_pBusyIndicatorKfs);  //add key frames to property
         }

         if (NULL != m_pNode)
         {
            m_RotationPS.SetTransformable(m_pNode);
         }

         break;

      case Candera::HorizontalBusyIndicator:
         m_pAnimProperty = Candera::Animation::AnimationBlendedProperty::Create();

         if (!m_pAnimProperty.PointsToNull())
         {
            m_pAnimProperty->AddKeyframeSequence(m_pBusyIndicatorKfs);  //add key frames to property
         }

         if (NULL != m_pCameraNode)
         {
            m_TranslatePS.SetTransformable(m_pCameraNode);
         }

         break;

      default:
         break;
   } // ![switch]
}


/******************************************************************************
*       Function      : CreateAnimationController
*       Description   :
*       Parameters     :
*       Return        : void
*****************************************************************************/
void BusyIndicatorWidget2D::CreateAnimationController()
{
   if (GetBusyWaitIndicatorType() == IncrementallyCircularBusyIndicator)
   {
      if (m_pAnimationController.PointsToNull() == false)
      {
         for (FeatStd::UInt8 i = 0; i < m_u8IncCirNodeCount; i++)
         {
            if (GetEnableScaling())
            {
               m_pAnimationController->AddProperty(m_animScaleProperty[i]);
            }

            m_pAnimationController->AddProperty(m_animAlphaProperty[i]);
         }
      }
   }
   else
   {
      if (m_pAnimationController.PointsToNull() == false)
      {
         m_pAnimationController->AddProperty(m_pAnimProperty);
      }
   }

   if (m_pAnimationController != NULL)
   {
      m_pAnimationController->SetSequenceReferenceTime(0);
   }
}


/******************************************************************************
*       Function      : CreateAnimationPlayer
*       Description   :
*       Parameters     :
*       Return        : void
*****************************************************************************/
void BusyIndicatorWidget2D::CreateAnimationPlayer()
{
   m_pAnimPlayer = Candera::Animation::AnimationPlayer::Create();

   if (!m_pAnimPlayer.PointsToNull())
   {
      //setup player
      m_pAnimPlayer->SetDirection((Candera::Animation::AnimationPlayer::PlayDirection)(GetPlayDirection()));
      m_pAnimPlayer->SetRepeatCount(c_repeatCount);
      m_pAnimPlayer->SetSpeedFactor(GetSpeedFactor());
      m_pAnimPlayer->SetSequenceDurationMs(::FeatStd::Int32(GetAnimationDuration()));
   }
}


/******************************************************************************
*       Function      : CreateScalingAnimationProperties
*       Description   :
*       Parameters     :
*       Return        : void
*****************************************************************************/
void BusyIndicatorWidget2D::CreateScalingAnimationProperties()
{
   for (FeatStd::UInt8 i = 0; i < m_u8IncCirNodeCount; i++)
   {
      m_animScaleProperty.Add(Animation::AnimationBlendedProperty::Create());

      if (m_pScalePS != NULL)
      {
         m_pScalePS[i].SetTransformable(m_ptCircularBusyIndicatorNodePtr[i]);

         _TODO("SetAnimationPropertySetter in CGI 3")

         m_animScaleProperty[i]->AddKeyframeSequence(m_pIncCirScaleKfs[i]);
      }
   }
}


/******************************************************************************
*       Function      : CreateAlphaAnimationProperties
*       Description   :
*       Parameters     :
*       Return        : void
*****************************************************************************/
void BusyIndicatorWidget2D::CreateAlphaAnimationProperties()
{
   for (FeatStd::UInt8 i = 0; i < m_u8IncCirNodeCount; i++)
   {
      m_animAlphaProperty.Add(Candera::Animation::AnimationBlendedProperty::Create());

      if (m_pAlphaPS != NULL)
      {
         m_pAlphaPS[i].SetNode(m_ptCircularBusyIndicatorNodePtr[i]);

         _TODO("SetAnimationPropertySetter in CGI 3")

         m_animAlphaProperty[i]->AddKeyframeSequence(m_pIncCirBusyIndicatorKfs[i]);
      }
   }
}


/******************************************************************************
*       Function      : CreateAlphaAnimationKeyFrameSequence
*       Description   :
*       Parameters     :
*       Return        : void
*****************************************************************************/
void BusyIndicatorWidget2D::CreateAlphaAnimationKeyFrameSequence()
{
   for (FeatStd::UInt8 i = 0; i < m_u8IncCirNodeCount; i++)
   {
      m_pIncCirBusyIndicatorKfs.Add(Candera::Animation::AnimationKeyframeSequence::Create());

      _TODO("AnimationKeyframeSequence in CGI 3")

      if (m_IncCirAnimationValues != NULL)
      {
         m_pIncCirBusyIndicatorKfs[i]->SetKeyframes(
            1,
            ::FeatStd::Int32(m_u8IncCirNodeCount),
            m_IncCirAnimationKfs,
            0,
            &m_IncCirAnimationValues[i * m_u8IncCirNodeCount],
            0
         );
      }
   }
}


/******************************************************************************
*       Function      : CreateScaleAnimationKeyFrameSequence
*       Description   :
*       Parameters     :
*       Return        : void
*****************************************************************************/
void BusyIndicatorWidget2D::CreateScaleAnimationKeyFrameSequence()
{
   for (FeatStd::UInt8 i = 0; i < m_u8IncCirNodeCount; i++)
   {
      m_pIncCirScaleKfs.Add(Candera::Animation::AnimationKeyframeSequence::Create());
      // set the pivot offset for all the nodes
      m_ptCircularBusyIndicatorNodePtr[i]->SetPivotOffset(m_pivotOffset);

      _TODO("AnimationKeyframeSequence in CGI 3")

      if (m_IncCirAnimationValues != NULL)
      {
         m_pIncCirScaleKfs[i]->SetKeyframes(
            1,
            ::FeatStd::Int32(m_u8IncCirNodeCount),
            m_IncCirAnimationKfs,
            0,
            &m_IncCirAnimationValues[i * m_u8IncCirNodeCount],
            0
         );
      }
   }
}


/****************************************************************************
*   Function    : OnChanged
*   Description :
*   Parameters  : ::FeatStd::Int32
*    Return      : void
****************************************************************************/
void BusyIndicatorWidget2D::OnChanged(FeatStd::UInt32 propertyId)
{
   switch (propertyId)
   {
      //this widget has notifierPolicy=onUpdate which means that changing its specific animation related properties will trigger Update so OnChanged will be called from Update instead from the setter
      //for the other properties like Visible, Enabled => OnChanged is called from the setter
      case VisiblePropertyId:
      case EnabledPropertyId:
         Base::OnChanged(propertyId);
         break;

      case SpeedFactorPropertyId:
         if (!m_pAnimPlayer.PointsToNull())
         {
            m_pAnimPlayer->SetSpeedFactor(GetSpeedFactor());
         }

         break;

      case PlayDirectionPropertyId:
         if (!m_pAnimPlayer.PointsToNull())
         {
            m_pAnimPlayer->SetDirection((Candera::Animation::AnimationPlayer::PlayDirection)(GetPlayDirection()));
         }

         break;

      case TextPropertyId:
         m_sText = GetText();

         if ((m_TextWidgetPtr != NULL) && (m_sText.GetCharCount() != 0))
         {
            m_TextWidgetPtr->SetText(m_sText);
         }

         break;

      default:
         break;
   }
}


/****************************************************************************
*   Function    : AdjustNodePivotOffset
*   Description : Set the Node pivot offset to the center of the node
*   Parameters  :
*    Return      : void
****************************************************************************/
void BusyIndicatorWidget2D::AdjustNodePivotOffset()
{
   Candera::Rectangle     nodeBoundingRect;
   Candera::Vector2             GroupNodePos;
   // Get the width and height of any node ( assuming all the nodes have the same dimension)
   m_ptCircularBusyIndicatorNodePtr[0]->GetEffectiveBoundingRectangle(nodeBoundingRect);

   if (NULL != m_pMainNode)
   {
      GroupNodePos = m_pMainNode->GetPosition();
   }

   ::Candera::Float flHalfNodeWidth = (nodeBoundingRect.GetWidth() / 2);
   ::Candera::Float flHalfNodeHeight = (nodeBoundingRect.GetHeight() / 2);
   m_pivotOffset.SetX(-flHalfNodeWidth);
   m_pivotOffset.SetY(-flHalfNodeHeight);

   // Adjust the node position due to change in pivot offset
   if (NULL != m_pMainNode)
   {
      m_pMainNode->SetPosition(GroupNodePos.GetX() + flHalfNodeWidth, GroupNodePos.GetY() + flHalfNodeHeight);
   }
}


/****************************************************************************
*   Function    : OnParentViewRenderingEnabled
*   Description :
*   Parameters  :
*   Return      : void
****************************************************************************/
void BusyIndicatorWidget2D::OnParentViewRenderingEnabled(bool enable)
{
   if (!enable)
   {
      if (m_BusyIndicator_Timer != NULL)
      {
         m_BusyIndicator_Timer->stop();
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BusyIndicatorWidget : MinimumAnimationDuration Timer KILLED"));
      }

      StopBusyIndicatorAnimation();
   }
}


void BusyIndicatorWidget2D::StartBusyIndicatorTimer()
{
   // start a timer
   if ((m_BusyIndicator_Timer != NULL) && (GetMinimumAnimationDuration() > 0))
   {
      m_bMinimumAnimationDuration_TimerExpired = false;
      m_BusyIndicator_Timer->start();
   }
}
