/* ***************************************************************************************
* FILE:          BlurViewScene2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  BlurViewScene2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "widget2D_std_if.h"
#include "BlurWidget2D.h"

#ifdef COURIER_ENHANCED_ENABLED

#include "BlurViewScene2D.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_BLUR
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/BlurViewScene2D.cpp.trc.h"
#endif

namespace hmibase {
namespace widget {
namespace blur {

using namespace Candera;


BlurViewScene2D::BlurViewScene2D(bool managed)
   : Base(managed),
     m_blurWidgets(),
     m_maximalBlurRadius(0)
{}

BlurViewScene2D::~BlurViewScene2D()
{}

void BlurViewScene2D::Finalize()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurViewScene2D: Finalize"));
   BlurWidgetVector::iterator widgetIterator = m_blurWidgets.begin();

   while (widgetIterator != m_blurWidgets.end())
   {
      BlurWidget2D* widget = (*widgetIterator);
      if (widget != NULL)
      {
         widget->Finalize();
      }

      // Finalize also executes "RemoveWidget" internally, so incrementing the iterator to iterate the list is impossible
      widgetIterator = m_blurWidgets.begin();
   }

   m_blurWidgets.clear();
   Base::Finalize();
}


bool BlurViewScene2D::IsUseDirtyAreaEnabled()
{
   return IsDirtyRectUsed();
}


bool BlurViewScene2D::IsBlurEnabled()
{
   for (BlurWidgetVector::iterator it = m_blurWidgets.begin(); it != m_blurWidgets.end(); ++it)
   {
      if ((*it != NULL) && (*it)->GetBlurEnabled())
      {
         return true;
      }
   }
   return false;
}


bool BlurViewScene2D::DistributeMessage(const Courier::Message& msg)
{
   return DistributeMessageDirect(msg);
}


void BlurViewScene2D::Update(Courier::RenderHint* renderHint)
{
   if (IsActive() || IsInactiveViewUpdateEnabled())
   {
      Base::Update(renderHint);
   }
   else
   {
      BlurWidgetVector::iterator widgetIterator = m_blurWidgets.begin();
      BlurWidgetVector::iterator widgetIteratorEnd = m_blurWidgets.end();

      while (widgetIterator != widgetIteratorEnd)
      {
         BlurWidget2D* widget = (*widgetIterator);
         if (widget != NULL)
         {
            if (widget->GetUpdateAfterInactive())
            {
               ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurViewScene2D: Last Update after inactivation of view '%s'", GetId().CStr()));
               Courier::ViewScene2D::Update(renderHint);
            }
         }
      }

      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurViewScene2D: Update of inactive view '%s' is not enabled", GetId().CStr()));
   }
}


void BlurViewScene2D::AddBlurWidget(BlurWidget2D* blurWidget)
{
   if (NULL != blurWidget)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurViewScene2D: Add BlurWidget[%s]", blurWidget->GetLegacyName()));
      Courier::IViewHandler* viewHandler = GetViewHandler();

      if (NULL != viewHandler)
      {
         Courier::Renderer* renderer = viewHandler->GetRenderer();

         if (NULL != renderer)
         {
            DeleteBlurRenderJobs(blurWidget, renderer);
         }
      }
      m_blurWidgets.push_back(blurWidget);
      blurWidget->AddBlurPassGdus();
   }
}


void BlurViewScene2D::RemoveBlurWidget(BlurWidget2D* blurWidget)
{
   if (blurWidget != NULL)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurViewScene2D: Remove BlurWidget[%s]", blurWidget->GetLegacyName()));
      Courier::IViewHandler* viewHandler = GetViewHandler();

      if (NULL != viewHandler)
      {
         Courier::Renderer* renderer = viewHandler->GetRenderer();

         if (NULL != renderer)
         {
            BlurWidgetVector::iterator widgetIterator = m_blurWidgets.begin();
            BlurWidgetVector::iterator widgetIteratorEnd = m_blurWidgets.end();

            while (widgetIterator != widgetIteratorEnd)
            {
               BlurWidget2D* widget = (*widgetIterator);
               if (widget == blurWidget)
               {
                  DeleteBlurRenderJobs(widget, renderer);

                  //widgetIterator = m_blurWidgets.erase(widgetIterator);//Not compilable on the target
                  m_blurWidgets.erase(widgetIterator);
                  widgetIterator = m_blurWidgets.begin();
                  widgetIteratorEnd = m_blurWidgets.end();
               }
               else
               {
                  ++widgetIterator;
               }
            }
         }
      }
      blurWidget->RemoveBlurPassGdus();
   }
}


void BlurViewScene2D::Invalidate(Courier::UInt8 renderCounter, const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurViewScene2D: Invalidate Dirty Area: use dirty area=%5s, dirtyArea=%5s {%f,%f,%f,%f}",
                       (IsUseDirtyAreaEnabled() == true) ? "true" : "false",
                       (dirtyArea == true) ? "true" : "false",
                       (*dirtyArea).GetLeft(),
                       (*dirtyArea).GetTop(),
                       (*dirtyArea).GetWidth(),
                       (*dirtyArea).GetHeight()));

   // invalidate all invalidation dependencies.
   if (IsUseDirtyAreaEnabled())
   {
      ViewScene::Invalidate(dirtyArea);
   }
   else
   {
      ViewScene::Invalidate();
   }

   Courier::IViewHandler* viewHandler = GetViewHandler();
   if (NULL != viewHandler)
   {
      Courier::Renderer* renderer = viewHandler->GetRenderer();

      if (NULL != renderer)
      {
         BlurViewScene2D::CameraPtrVector& cams = GetCameraPtrVector();
         FeatStd::Internal::Vector<Candera::Camera2D*>::Iterator cameraIterator = cams.Begin();
         FeatStd::Internal::Vector<Candera::Camera2D*>::Iterator cameraIteratorEnd = cams.End();

         while (cameraIterator != cameraIteratorEnd)
         {
            OriginalCamera2D* camera = (*cameraIterator);

            if (camera != 0)
            {
               bool cameraIsBlurred = false;
               if (camera->IsEffectiveRenderingEnabled() == true)
               {
                  ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurViewScene2D: Invalidate Dirty Area -> Invalidating"));
                  if (m_blurWidgets.empty() == false)
                  {
                     BlurWidgetVector::iterator widgetIterator = m_blurWidgets.begin();
                     BlurWidgetVector::iterator widgetIteratorEnd = m_blurWidgets.end();

                     while (widgetIterator != widgetIteratorEnd)
                     {
                        BlurWidget2D* widget = (*widgetIterator);
                        if (widget != 0)
                        {
                           if (widget->IsCameraPartOfBlur((*cameraIterator)))
                           {
                              cameraIsBlurred = true;
                              InvalidateWidget(camera, widget, renderCounter, dirtyArea, renderer);
                           }
                        }
                        ++widgetIterator;
                     }
                  }

                  if (cameraIsBlurred == false)
                  {
                     InvalidateCamera(camera, renderCounter, dirtyArea, renderer);
                  }
               }
               else
               {
                  ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurViewScene2D: Invalidate Dirty Area -> Skipping invalidate -> Original Camera effective rendering disabled"));
               }
            }

            ++cameraIterator;
         }
      }
   }
}


void BlurViewScene2D::Invalidate(const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
   // (IsUseDirtyAreaEnabled()) will be evaluated inside the actual invalidation function
   Invalidate(View::cDefaultRender2DCounter, dirtyArea);
}


void BlurViewScene2D::Invalidate(Candera::Camera2D* camera, const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
   if ((camera != 0) && (camera->IsEffectiveRenderingEnabled()))
   {
      if (IsUseDirtyAreaEnabled())
      {
         Invalidate(camera, View::cDefaultRender2DCounter, dirtyArea);
      }
      else
      {
         Invalidate(camera, View::cDefaultRender2DCounter);
      }
   }
}


void BlurViewScene2D::Invalidate(Candera::Camera2D* camera, Candera::UInt8 renderCounter, const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
   // invalidate all invalidation dependencies.
   if (IsUseDirtyAreaEnabled())
   {
      ViewScene::Invalidate(dirtyArea);
   }
   else
   {
      ViewScene::Invalidate();
   }

   if ((camera != 0) && (camera->IsEffectiveRenderingEnabled()))
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurViewScene2D:  Invalidate, camera='%30s'(%p), use dirty area=%u, dirtyArea=%u {%f,%f,%f,%f}\n",
                          ((0 != camera) && (0 != camera->GetName())) ? camera->GetName() : "",
                          camera,
                          IsUseDirtyAreaEnabled(),
                          dirtyArea == true,
                          (*dirtyArea).GetLeft(),
                          (*dirtyArea).GetTop(),
                          (*dirtyArea).GetWidth(),
                          (*dirtyArea).GetHeight()));

      Courier::IViewHandler* viewHandler = GetViewHandler();
      if (0 != viewHandler)
      {
         Courier::Renderer* renderer = viewHandler->GetRenderer();

         if (0 != renderer)
         {
            bool cameraIsBlurred = false;

            BlurWidgetVector::iterator widgetIterator = m_blurWidgets.begin();
            BlurWidgetVector::iterator widgetIteratorEnd = m_blurWidgets.end();

            while (widgetIterator != widgetIteratorEnd)
            {
               BlurWidget2D* widget = (*widgetIterator);

               if (widget != 0)
               {
                  if (widget->IsCameraPartOfBlur(camera))
                  {
                     cameraIsBlurred = true;
                     InvalidateWidget(camera, widget, renderCounter, dirtyArea, renderer);
                  }
               }
               ++widgetIterator;
            }

            if (cameraIsBlurred == false)
            {
               InvalidateCamera(camera, renderCounter, dirtyArea, renderer);
            }
         }
      }
   }
   else
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurViewScene2D: Invalidate -> Skipping invalidate -> Original Camera effective rendering disabled"));
   }
}


void BlurViewScene2D::Invalidate(Candera::RenderTarget* renderTarget, const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
   if (IsUseDirtyAreaEnabled())
   {
      ViewScene::Invalidate(dirtyArea);
   }
   else
   {
      ViewScene::Invalidate();
   }

   if (renderTarget != 0)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurViewScene2D: Invalidate Rendertarget (%p), use dirty area=%5s, dirtyArea=%5s {%f,%f,%f,%f}\n",
                          renderTarget,
                          (IsUseDirtyAreaEnabled() == true) ? "true" : "false",
                          (dirtyArea == true) ? "true" : "false",
                          (*dirtyArea).GetLeft(),
                          (*dirtyArea).GetTop(),
                          (*dirtyArea).GetWidth(),
                          (*dirtyArea).GetHeight()));

      Courier::IViewHandler* viewHandler = GetViewHandler();
      if (NULL != viewHandler)
      {
         Courier::Renderer* renderer = viewHandler->GetRenderer();

         if (NULL != renderer)
         {
            renderer->Invalidate(renderTarget);

            BlurWidgetVector::iterator widgetIterator = m_blurWidgets.begin();
            BlurWidgetVector::iterator widgetIteratorEnd = m_blurWidgets.end();

            while (widgetIterator != widgetIteratorEnd)
            {
               BlurWidget2D* widget = (*widgetIterator);

               if (widget != NULL)
               {
                  if (widget->IsRenderTargetPartOfBlur(renderTarget))
                  {
                     OriginalCamera2DVectorType cameras = widget->GetOriginalCameras();

                     OriginalCamera2DVectorType::iterator cameraIterator = cameras.begin();
                     OriginalCamera2DVectorType::iterator cameraIteratorEnd = cameras.end();

                     while (cameraIteratorEnd != cameraIteratorEnd)
                     {
                        Candera::Camera2D* camera = (*cameraIterator);

                        if ((camera != 0) && (camera->IsEffectiveRenderingEnabled()))
                        {
                           // If Rendertarget is invalid, we need to redraw everything inside that rendertarget, if the camera is active
                           InvalidateWidget(camera, widget, View::cDefaultRender2DCounter, FeatStd::Optional<Candera::Rectangle>(), renderer);
                        }
                        else
                        {
                           ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurViewScene2D: Invalidate Rendertarget (%p), camera %30s not invalidated, effective rendering disabled",
                                               renderTarget,
                                               camera != 0 ? camera->GetName() : "NullPtr"));
                        }

                        ++cameraIterator;
                     }
                  }
               }
               ++widgetIterator;
            }
         }
      }
   }
}


void BlurViewScene2D::InvalidateWidget(Candera::Camera2D* camera, BlurWidget2D* widget, Candera::UInt8 renderCounter, const FeatStd::Optional<Candera::Rectangle>& dirtyArea, Courier::Renderer* renderer)
{
   if (0 != renderer)
   {
      if ((widget != 0) && (camera != 0))
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurViewScene2D: Invalidate Widget [%50s], use dirty area=%u, dirtyArea=%u {%f,%f,%f,%f}\n",
                             widget->GetLegacyName(),
                             IsUseDirtyAreaEnabled(),
                             dirtyArea == true,
                             (*dirtyArea).GetLeft(),
                             (*dirtyArea).GetTop(),
                             (*dirtyArea).GetWidth(),
                             (*dirtyArea).GetHeight()));

         bool isWidgetMultipassEnabled = widget->GetMultipass();

         FeatStd::Optional<Candera::Rectangle> originalRadiusEnhancedDirtyArea = dirtyArea;
         FeatStd::Optional<Candera::Rectangle> radiusEnhancedDirtyArea = dirtyArea;

         if ((IsUseDirtyAreaEnabled() == true) && (dirtyArea == true))
         {
            Candera::Rectangle radiusEnhancedDirtyAreaRect = (*dirtyArea);
            Candera::Rectangle originalViewport = widget->GetOriginalCameraViewport(camera).GetRectInPixel();

            RetrieveMaximalBlurRadius();

            radiusEnhancedDirtyAreaRect.SetLeft(radiusEnhancedDirtyAreaRect.GetLeft() - m_maximalBlurRadius);
            radiusEnhancedDirtyAreaRect.SetTop(radiusEnhancedDirtyAreaRect.GetTop() - m_maximalBlurRadius);
            radiusEnhancedDirtyAreaRect.SetWidth(radiusEnhancedDirtyAreaRect.GetWidth() + (2 * m_maximalBlurRadius));
            radiusEnhancedDirtyAreaRect.SetHeight(radiusEnhancedDirtyAreaRect.GetHeight() + (2 * m_maximalBlurRadius));

            Candera::Rectangle originalRadiusEnhancedDirtyAreaRect = radiusEnhancedDirtyAreaRect;
            originalRadiusEnhancedDirtyAreaRect.SetLeft(originalViewport.GetLeft() + radiusEnhancedDirtyAreaRect.GetLeft());
            originalRadiusEnhancedDirtyAreaRect.SetTop(originalViewport.GetTop() + radiusEnhancedDirtyAreaRect.GetTop());

            radiusEnhancedDirtyArea = radiusEnhancedDirtyAreaRect;
            originalRadiusEnhancedDirtyArea = originalRadiusEnhancedDirtyAreaRect;
         }
         else if ((IsUseDirtyAreaEnabled() == false) && (dirtyArea == true))
         {
            // Dirty area ignored, due to settings
            radiusEnhancedDirtyArea = FeatStd::Optional<Candera::Rectangle>();
            originalRadiusEnhancedDirtyArea  = FeatStd::Optional<Candera::Rectangle>();
         }
         else
         {
            // Dirty area not set (usage flag will be ignored)
         }

         // Unmodified Dirty Area, as the InvalidateCamera will take care of this
         InvalidateCamera(camera, renderCounter, dirtyArea, renderer);

         // Iterates passes (first pass == 1, second pass == 2...)
         for (Candera::Int passIndex = 1; passIndex <= 2; ++passIndex)
         {
            Camera2DVectorType cameras = widget->GetBlurCameras(camera, passIndex);

            if (!cameras.empty())
            {
               Camera2DVectorType::iterator iterator = cameras.begin();
               Camera2DVectorType::iterator iteratorEnd = cameras.end();

               while (iterator != iteratorEnd)
               {
                  // If the pass is the last blur pass (i.e. the one, that renders to the original RT), then dirty area can be used
                  // as the viewport is the original viewport.
                  if ((isWidgetMultipassEnabled == true) && (passIndex == 2))
                  {
                     renderer->Queue3DJob(this, *iterator, renderCounter, originalRadiusEnhancedDirtyArea, false, true);
                  }
                  else if ((isWidgetMultipassEnabled == false) && (passIndex == 1))
                  {
                     renderer->Queue3DJob(this, *iterator, renderCounter, originalRadiusEnhancedDirtyArea, false, true);
                  }
                  else
                  {
                     // If camera is not the last pass, invalidate the full viewport
                     renderer->Queue3DJob(this, *iterator, renderCounter, radiusEnhancedDirtyArea, false, true);
                  }

                  ++iterator;
               }
            }
         }
      }
   }
}


void BlurViewScene2D::InvalidateCamera(Candera::Camera2D* camera, Candera::UInt8 renderCounter, const FeatStd::Optional<Candera::Rectangle>& dirtyArea, Courier::Renderer* renderer)
{
   if (camera != 0)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurViewScene2D: Invalidate Camera [%30s], use dirty area=%u, dirtyArea=%u {%f,%f,%f,%f}\n",
                          camera->GetName(),
                          IsUseDirtyAreaEnabled(),
                          dirtyArea == true,
                          (*dirtyArea).GetLeft(),
                          (*dirtyArea).GetTop(),
                          (*dirtyArea).GetWidth(),
                          (*dirtyArea).GetHeight()));
      if (renderer != 0)
      {
         FeatStd::Optional<Candera::Rectangle> radiusEnhancedDirtyArea = dirtyArea;

         if ((IsUseDirtyAreaEnabled() == true) && (dirtyArea == true))
         {
            RetrieveMaximalBlurRadius();

            Candera::Rectangle radiusEnhancedDirtyAreaRect = (*dirtyArea);

            radiusEnhancedDirtyAreaRect.SetLeft(radiusEnhancedDirtyAreaRect.GetLeft() - m_maximalBlurRadius);
            radiusEnhancedDirtyAreaRect.SetTop(radiusEnhancedDirtyAreaRect.GetTop() - m_maximalBlurRadius);
            radiusEnhancedDirtyAreaRect.SetWidth(radiusEnhancedDirtyAreaRect.GetWidth() + (2 * m_maximalBlurRadius));
            radiusEnhancedDirtyAreaRect.SetHeight(radiusEnhancedDirtyAreaRect.GetHeight() + (2 * m_maximalBlurRadius));

            radiusEnhancedDirtyArea = radiusEnhancedDirtyAreaRect;
         }
         else if ((IsUseDirtyAreaEnabled() == false) && (dirtyArea == true))
         {
            // Dirty area ignored, due to settings
            radiusEnhancedDirtyArea = FeatStd::Optional<Candera::Rectangle>();
         }
         else
         {
            // Dirty area not set (usage flag will be ignored)
         }
         renderer->Queue2DJob(this, camera, renderCounter, radiusEnhancedDirtyArea, false, true);
      }
   }
}


void BlurViewScene2D::RetrieveMaximalBlurRadius()
{
   m_maximalBlurRadius = 0;

   BlurWidgetVector::iterator widgetIterator = m_blurWidgets.begin();
   BlurWidgetVector::iterator widgetIteratorEnd = m_blurWidgets.end();

   while (widgetIterator != widgetIteratorEnd)
   {
      BlurWidget2D* widget = (*widgetIterator);

      if (widget != 0)
      {
         if (widget->GetBlurRadius() > m_maximalBlurRadius)
         {
            m_maximalBlurRadius = widget->GetBlurRadius();
         }
      }
      ++widgetIterator;
   }
}


void BlurViewScene2D::DeleteWidgetRenderJobs(Candera::Camera2D* camera, BlurWidget2D* widget, Courier::Renderer* renderer)
{
   if ((widget != 0) && (renderer != 0))
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurViewScene2D: Delete Renderjobs [%s]", widget->GetLegacyName()));
      for (Candera::Int passIndex = 1; passIndex <= 2; ++passIndex)
      {
         Camera2DVectorType cameras = widget->GetBlurCameras(camera, passIndex);

         if (!cameras.empty())
         {
            Camera2DVectorType::iterator iterator = cameras.begin();
            Camera2DVectorType::iterator iteratorEnd = cameras.end();

            while (iterator != iteratorEnd)
            {
               renderer->DeleteRenderJob(*iterator);
               ++iterator;
            }
         }
      }
   }
}


void BlurViewScene2D::DeleteBlurRenderJobs(BlurWidget2D* widget, Courier::Renderer* renderer)
{
   if ((widget != 0) && (renderer != 0))
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurViewScene3D: Delete Renderjobs [%s]", widget->GetLegacyName()));
      OriginalCamera2DVectorType cameras = widget->GetOriginalCameras();

      if (!cameras.empty())
      {
         OriginalCamera2DVectorType::iterator iterator = cameras.begin();
         OriginalCamera2DVectorType::iterator iteratorEnd = cameras.end();

         while (iterator != iteratorEnd)
         {
            Camera2D* camera = (*iterator);
            if (camera != 0)
            {
               DeleteWidgetRenderJobs(camera, widget, renderer);
               renderer->DeleteRenderJob(camera);
            }

            ++iterator;
         }
      }
   }
}


}   /* hmibase */
}   /* widget */
}   /* blur */

#endif  //#ifdef COURIER_ENHANCED_ENABLED
