/* ***************************************************************************************
* FILE:          BaseWidget2D.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  BaseWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifndef _HMIBASE_BASEWIDGET2D_H_
#define _HMIBASE_BASEWIDGET2D_H_

#include <Courier/Version.h>
#include <Courier/Base.h>
#include <CanderaWidget/Widget.h>

#include <BaseContract/generated/BaseMsgs.h>

#include <Widgets/generated/AllWidgetTypes.h>
#include <Widgets/generated/AllWidgetMessages.h>
#include <Widgets/2D/ControlTemplate/ControlTemplateCloneableWidget.h>
#include <Widgets/2D/Appearance/AppearanceState.h>


namespace Focus {
class FListItemData;
class FManager;
class FWidgetConfig;
}


namespace hmibase {
namespace widget {
namespace flags {
enum Enum_BaseWidget2D
{
   VisibleEnabled = _LastValue_Widget,
   LastEffectiveRenderingEnabled,

   //last value
   _LastValue_BaseWidget2D
};


}
namespace appearance {
class Appearance;
}


}
}


class BaseWidget2D : public Candera::Widget2D
{
   public:
      //type declaration according to CGI Studio Standards
      CGI_WIDGET_RTTI_DECLARATION(BaseWidget2D, Candera::Widget2D);

      HMIBASE_TYPEDEF_OBJECT_ACCESSOR(BaseWidget2D);

      BaseWidget2D();
      virtual ~BaseWidget2D();

      /************************************************************************/
      /* overrides from Courier::FrameworkWidget                              */
      /************************************************************************/
      virtual bool OnMessage(const Courier::Message& msg) override;
      virtual void OnFocus() override;
      virtual void OnLostFocus() override;
      virtual void Invalidate(const FeatStd::Optional<Candera::Rectangle>& dirtyArea = FeatStd::Optional<Candera::Rectangle>()) override;

      /************************************************************************/
      /* overrides from Candera::WidgetBase                                   */
      /************************************************************************/
      //initializes this widget
      virtual void Init(Candera::AssetProvider* assetProvider) override;

      //destroys this widget
      virtual void Finalize() override;

      //updates widget and its node before rendering
      virtual void Update() override;

      /************************************************************************/
      /* overrides from hmibase::widget::Widget                               */
      /************************************************************************/
      //prevents the dynamic cast
      virtual const BaseWidget2D* ToBaseWidget2D() const override;

      //prevents the dynamic cast
      virtual BaseWidget2D* ToBaseWidget2D() override;

      //returns true because this is a 2D widget
      virtual bool Is2D() const override;

      //returns the node associated with this 2D widget
      virtual Candera::Node2D* GetNode2D() const override;

      //returns false because this is not a 3D widget
      virtual bool Is3D() const override;

      //returns NULL because this is not a 3D widget
      virtual Candera::Node* GetNode3D() const override;

      //required for list cloning
      virtual bool CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap) override;

      //widget node and all its ancestors have rendering enabled
      virtual bool IsEffectiveVisible() const override;

      //name of a Widget as it used to be in Candera 2.x
      virtual const FeatStd::Char* GetLegacyName() override;

      //returns the asset provider
      Candera::AssetProvider* GetAssetProvider();

      //returns the widget from which this one was cloned
      const ControlTemplateCloneableWidget* GetOriginal() const;

      //returns the list item information for this widget
      const Focus::FListItemData* GetListItemInfo() const;

      //enables controlling of the visibility (set this to true if you want Visible to have any effect)
      DefWidgetFlag(VisibleEnabled, 0)

      //appearance id
      DefWidgetProperty(AppearanceId, FeatStd::UInt32, 0xE00D5379)
      DefWidgetProperty_SetterWithCheckAndNotifier(AppearanceId)

#define CdaProperty_AppearanceId()\
      CdaBindableProperty(AppearanceId, AppearanceIdType, GetAppearanceId, SetAppearanceId)\
      CdaDescription("UNDER DEVELOPMENT! Specifies the id of the appearance responsible to change, based on widget state (enabled, pressed, active, focused), the images and the colors for the widget node and the descendant nodes. The appearances are registered at start-up.")\
      CdaBindablePropertyEnd()

      void SetFocus(bool focus);

      //state used for the widget appearance
      typedef hmibase::widget::appearance::AppearanceState AppearanceState;
      virtual AppearanceState GetAppearanceState();

      //deprecated
      HMIBASE_DEPRECATED("Use SetEnabled", void SetEnableStatus(bool enabled));
      HMIBASE_DEPRECATED("Use IsEnabled", bool GetEnableStatus() const);
      HMIBASE_DEPRECATED("Use SetVisible", void SetVisibleStatus(bool visible));
      HMIBASE_DEPRECATED("Use IsVisible", bool GetVisibleStatus() const);

      //meta info
      CdaWidget2DDef(BaseWidget2D, Candera::Widget2D)
      CdaCategory("")

      CdaProperties()

      CdaBindableProperty(Enable, bool, IsEnabled, SetEnabled)
      CdaDescription("Enables this widget. Some widgets (for example button) use this property while other ignore it.")
      CdaVisibilityTest(composerPropVisibleFilter_Enabled)
      CdaBindablePropertyEnd()

      CdaProperty(VisibleEnabled, bool, IsVisibleEnabled, SetVisibleEnabled)
      CdaDescription("Enables the configuration of the node property EnableRendering which is used to determine if the node is rendered or not. A node is effectively rendered if it and all its ancestors have rendering enabled.\n Please notice that if the same property of a node is set from multiple sources then the result is unpredictable.")
      CdaPropertyEnd()

      CdaBindableProperty(Visible, bool, IsVisible, SetVisible)
      CdaDescription("Configures the node property EnableRendering which is used to determine if the node is rendered or not. A node is effectively rendered if it and all its ancestors have rendering enabled.\n Please notice that if the same property of a node is set from multiple sources then the result is unpredictable.")
      CdaVisibilityTest(IsVisibleEnabled)
      CdaBindablePropertyEnd()

      CdaPropertiesEnd()
      CdaWidgetDefEnd()

   protected:
      virtual bool composerPropVisibleFilter_Enabled() const;

      virtual bool IsPickIntersectingNode(const Candera::Camera2D& camera2D, const Candera::Node2D* node, const Candera::Vector2& point) const;

      // OnParentViewXYZ methods are called only for widgets available in the ViewScene.
      // The widgets cloned by the list are not there, however the notification is also received in OnMessage.
      // This method calls OnParentViewXYZ for these cloned widgets.
      virtual void DispatchViewEvent(const Courier::Message& msg);

      /************************************************************************/
      /* overrides from Candera::Widget2D                                     */
      /************************************************************************/
      virtual void OnBeforeNodeChanged() override;

      virtual void OnChanged(FeatStd::UInt32 propertyId) override;

      virtual void SetFlagValue(FeatStd::UInt32 propertyId, size_t pos, bool value) override;

      //language/culture has changed
      virtual void CultureChanged() {}
      //widget is being initialized
      virtual void InitWidget() {}

      void Invalidate(bool forceInvalidate, const FeatStd::Optional<Candera::Rectangle>& dirtyArea = FeatStd::Optional<Candera::Rectangle>());

      hmibase::widget::appearance::Appearance* GetAppearance();
      void SetAppearance(hmibase::widget::appearance::Appearance* app);
      void ClearAppearance();

   private:
      void InitLegacyName();
      void InitCachedStringIdentifier();
      void CleanupCachedStringIdentifier();

      void Invalidate(Courier::ViewScene2D& view, const Candera::Matrix3x2& worldTransformation, const Candera::Rectangle& boundingRectangle);

      DefWidgetFlag(LastEffectiveRenderingEnabled, 0)

      Candera::AssetProvider* _assetProvider;
      const ControlTemplateCloneableWidget* _original;
      Focus::FListItemData* _listItemInfo;

      // manages the appearance of this button by updating the descendant render nodes
      hmibase::widget::appearance::Appearance* _appearance;
      DefWidgetProperty_Field(AppearanceId)

      std::string _legacyName;
      const FeatStd::Char* _cachedName;
      //holds a deep clone of the StringIdentifier hierarchy of this widget.
      const Candera::StringIdentifier* _cachedStringIdentifier;

      FeatStd::Optional<Candera::Rectangle> _occupiedArea;
      Candera::Matrix3x2 _worldTransformation;

      //the bounding rectangle of the invalidation container is used for calculating the dirty area instead of the node bounds
      //this is useful if there are layouters which change the bounds of other nodes in the scene and this widget does not know about them
      typedef FeatStd::Int8 InvalidationContainerLevelType;
      static const InvalidationContainerLevelType INVALIDATION_CONTAINER_INVALID = -1;
      static const InvalidationContainerLevelType INVALIDATION_CONTAINER_NOT_USED = -2;
      InvalidationContainerLevelType _invalidationContainerLevel;
      Candera::Node2D* GetInvalidationContainer();
};


typedef hmibase::util::ObjectAccessor<Candera::Widget2D> Widget2DAccessor;

#endif // _HMIBASE_BASEWIDGET2D_H_
