/* ***************************************************************************************
* FILE:          AppearanceState.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  AppearanceState.h is part of HMI-Base widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifndef HMIBASE_APPEARANCE_STATE_H
#define HMIBASE_APPEARANCE_STATE_H


#include <FeatStd/MemoryManagement/SharedPointer.h>


namespace hmibase {
namespace widget {
namespace appearance {

class AppearanceState
{
   public:
      AppearanceState(bool focused = false, bool active = false, bool pressed = false, bool enabled = true) : Focused(focused), Active(active), Pressed(pressed), Enabled(enabled)
      {
      }

      bool operator ==(const AppearanceState& other) const
      {
         return (Focused == other.Focused) && (Active == other.Active) && (Pressed == other.Pressed) && (Enabled == other.Enabled);
      }
      bool operator !=(const AppearanceState& other) const
      {
         return !operator==(other);
      }

      enum ENUM
      {
         /*0000*/enDisabled = 0,
         /*0001*/enNormal,
         /*0010*/enPressedDisabled,//
         /*0011*/enPressed,

         /*0100*/enActiveDisabled,
         /*0101*/enActiveNormal,
         /*0110*/enActivePressedDisabled,//
         /*0111*/enActivePressed,

         /*1000*/enFocusedDisabled,//
         /*1001*/enFocusedNormal,
         /*1010*/enFocusedPressedDisabled,//
         /*1011*/enFocusedPressed,

         /*1100*/enFocusedActiveDisabled,//
         /*1101*/enFocusedActiveNormal,
         /*1110*/enFocusedActivePressedDisabled,//
         /*1111*/enFocusedActivePressed
      };

      AppearanceState(ENUM enumValue) : Focused((enumValue & (1u << 3)) != 0), Active((enumValue & (1u << 2)) != 0), Pressed((enumValue & (1u << 1)) != 0), Enabled((enumValue & (1u << 0)) != 0)
      {
      }

      ENUM asEnum() const
      {
         return static_cast<ENUM>((Focused ? (1u << 3) : 0u) | (Active ? (1u << 2) : 0u) | (Pressed ? (1u << 1) : 0u) | (Enabled ? (1u << 0) : 0u));
      }

      bool isFocused() const
      {
         return Focused;
      }
      void setFocused(bool value)
      {
         Focused = value;
      }
      bool isActive() const
      {
         return Active;
      }
      void setActive(bool value)
      {
         Active = value;
      }
      bool isPressed() const
      {
         return Pressed;
      }
      void setPressed(bool value)
      {
         Pressed = value;
      }
      bool isEnabled() const
      {
         return Enabled;
      }
      void setEnabled(bool value)
      {
         Enabled = value;
      }

   private:
      bool Focused;
      bool Active;
      bool Pressed;
      bool Enabled;
};


/**************************************************************************/
template <typename TData>
class AppearanceStateData
{
   public:
      typedef TData DataType;

      /**************************************************************************/
      AppearanceStateData(const TData& normal = TData(),
                          const TData& pressed = TData(),
                          const TData& disabled = TData(),
                          const TData& pressedDisabled = TData(),
                          const TData& activeNormal = TData(),
                          const TData& activePressed = TData(),
                          const TData& activeDisabled = TData(),
                          const TData& activePressedDisabled = TData(),
                          const TData& focusedNormal = TData(),
                          const TData& focusedPressed = TData(),
                          const TData& focusedDisabled = TData(),
                          const TData& focusedPressedDisabled = TData(),
                          const TData& focusedActiveNormal = TData(),
                          const TData& focusedActivePressed = TData(),
                          const TData& focusedActiveDisabled = TData(),
                          const TData& focusedActivePressedDisabled = TData());

      /**************************************************************************/
      TData& getData(const AppearanceState& state);
      const TData& getData(const AppearanceState& state) const;

      /**************************************************************************/
      TData Normal;
      TData Pressed;
      TData Disabled;
      TData PressedDisabled;
      TData ActiveNormal;
      TData ActivePressed;
      TData ActiveDisabled;
      TData ActivePressedDisabled;
      TData FocusedNormal;
      TData FocusedPressed;
      TData FocusedDisabled;
      TData FocusedPressedDisabled;
      TData FocusedActiveNormal;
      TData FocusedActivePressed;
      TData FocusedActiveDisabled;
      TData FocusedActivePressedDisabled;
};


/**************************************************************************/
template <typename TData>
class ShareableAppearanceStateData : public AppearanceStateData<TData>
{
      typedef AppearanceStateData<TData> Base;

   public:
      FEATSTD_TYPEDEF_SHARED_POINTER(ShareableAppearanceStateData);

      static SharedPointer create(const TData& normal = TData(),
                                  const TData& pressed = TData(),
                                  const TData& disabled = TData(),
                                  const TData& pressedDisabled = TData(),
                                  const TData& activeNormal = TData(),
                                  const TData& activePressed = TData(),
                                  const TData& activeDisabled = TData(),
                                  const TData& activePressedDisabled = TData(),
                                  const TData& focusedNormal = TData(),
                                  const TData& focusedPressed = TData(),
                                  const TData& focusedDisabled = TData(),
                                  const TData& focusedPressedDisabled = TData(),
                                  const TData& focusedActiveNormal = TData(),
                                  const TData& focusedActivePressed = TData(),
                                  const TData& focusedActiveDisabled = TData(),
                                  const TData& focusedActivePressedDisabled = TData())
      {
         return SharedPointer(FEATSTD_NEW(ShareableAppearanceStateData)(normal, pressed, disabled, pressedDisabled, activeNormal, activePressed, activeDisabled, activePressedDisabled, focusedNormal, focusedPressed, focusedDisabled, focusedPressedDisabled, focusedActiveNormal, focusedActivePressed, focusedActiveDisabled, focusedActivePressedDisabled));
      }

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(ShareableAppearanceStateData);
      FEATSTD_SHARED_POINTER_DECLARATION();

      ShareableAppearanceStateData(const TData& normal, const TData& pressed, const TData& disabled, const TData& pressedDisabled, const TData& activeNormal, const TData& activePressed, const TData& activeDisabled, const TData& activePressedDisabled, const TData& focusedNormal, const TData& focusedPressed, const TData& focusedDisabled, const TData& focusedPressedDisabled, const TData& focusedActiveNormal, const TData& focusedActivePressed, const TData& focusedActiveDisabled, const TData& focusedActivePressedDisabled)
         : Base(normal, pressed, disabled, pressedDisabled, activeNormal, activePressed, activeDisabled, activePressedDisabled, focusedNormal, focusedPressed, focusedDisabled, focusedPressedDisabled, focusedActiveNormal, focusedActivePressed, focusedActiveDisabled, focusedActivePressedDisabled)
      {
      }
};


/**************************************************************************/
template <typename TData>
AppearanceStateData<TData>::AppearanceStateData(const TData& normal,
      const TData& pressed,
      const TData& disabled,
      const TData& pressedDisabled,
      const TData& activeNormal,
      const TData& activePressed,
      const TData& activeDisabled,
      const TData& activePressedDisabled,
      const TData& focusedNormal,
      const TData& focusedPressed,
      const TData& focusedDisabled,
      const TData& focusedPressedDisabled,
      const TData& focusedActiveNormal,
      const TData& focusedActivePressed,
      const TData& focusedActiveDisabled,
      const TData& focusedActivePressedDisabled)
   : Normal(normal),
     Pressed(pressed),
     Disabled(disabled),
     PressedDisabled(pressedDisabled),
     ActiveNormal(activeNormal),
     ActivePressed(activePressed),
     ActiveDisabled(activeDisabled),
     ActivePressedDisabled(activePressedDisabled),
     FocusedNormal(focusedNormal),
     FocusedPressed(focusedPressed),
     FocusedDisabled(focusedDisabled),
     FocusedPressedDisabled(focusedPressedDisabled),
     FocusedActiveNormal(focusedActiveNormal),
     FocusedActivePressed(focusedActivePressed),
     FocusedActiveDisabled(focusedActiveDisabled),
     FocusedActivePressedDisabled(focusedActivePressedDisabled)
{
}


/**************************************************************************/
template <typename TData>
TData& AppearanceStateData<TData>::getData(const AppearanceState& state)
{
   switch (state.asEnum())
   {
      default:
      case AppearanceState::enNormal:
         return Normal;
      case AppearanceState::enPressed:
         return Pressed;
      case AppearanceState::enDisabled:
         return Disabled;
      case AppearanceState::enPressedDisabled:
         return PressedDisabled;

      case AppearanceState::enActiveNormal:
         return ActiveNormal;
      case AppearanceState::enActivePressed:
         return ActivePressed;
      case AppearanceState::enActiveDisabled:
         return ActiveDisabled;
      case AppearanceState::enActivePressedDisabled:
         return ActivePressedDisabled;

      case AppearanceState::enFocusedNormal:
         return FocusedNormal;
      case AppearanceState::enFocusedPressed:
         return FocusedPressed;
      case AppearanceState::enFocusedDisabled:
         return FocusedDisabled;
      case AppearanceState::enFocusedPressedDisabled:
         return FocusedPressedDisabled;

      case AppearanceState::enFocusedActiveNormal:
         return FocusedActiveNormal;
      case AppearanceState::enFocusedActivePressed:
         return FocusedActivePressed;
      case AppearanceState::enFocusedActiveDisabled:
         return FocusedActiveDisabled;
      case AppearanceState::enFocusedActivePressedDisabled:
         return FocusedActivePressedDisabled;
   }
}


/**************************************************************************/
template <typename TData>
const TData& AppearanceStateData<TData>::getData(const AppearanceState& state) const
{
   switch (state.asEnum())
   {
      default:
      case AppearanceState::enNormal:
         return Normal;
      case AppearanceState::enPressed:
         return Pressed;
      case AppearanceState::enDisabled:
         return Disabled;
      case AppearanceState::enPressedDisabled:
         return PressedDisabled;

      case AppearanceState::enActiveNormal:
         return ActiveNormal;
      case AppearanceState::enActivePressed:
         return ActivePressed;
      case AppearanceState::enActiveDisabled:
         return ActiveDisabled;
      case AppearanceState::enActivePressedDisabled:
         return ActivePressedDisabled;

      case AppearanceState::enFocusedNormal:
         return FocusedNormal;
      case AppearanceState::enFocusedPressed:
         return FocusedPressed;
      case AppearanceState::enFocusedDisabled:
         return FocusedDisabled;
      case AppearanceState::enFocusedPressedDisabled:
         return FocusedPressedDisabled;

      case AppearanceState::enFocusedActiveNormal:
         return FocusedActiveNormal;
      case AppearanceState::enFocusedActivePressed:
         return FocusedActivePressed;
      case AppearanceState::enFocusedActiveDisabled:
         return FocusedActiveDisabled;
      case AppearanceState::enFocusedActivePressedDisabled:
         return FocusedActivePressedDisabled;
   }
}


}
}


}
#endif
