/* ***************************************************************************************
* FILE:          AnimationBaseHelper.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  AnimationBaseHelper is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "AnimationBaseHelper.h"

#include <Candera/Engine2D/AnimationPropertySetters/Transformable2DRotatePropertySetter.h>
#include <Candera/Engine2D/AnimationPropertySetters/Transformable2DScalePropertySetter.h>
#include <Candera/Engine2D/AnimationPropertySetters/Transformable2DTranslatePropertySetter.h>
#include <Candera/Engine2D/AnimationPropertySetters/RenderingEnabledNode2DPropertySetter.h>
#include "Candera/Engine2D/AnimationPropertySetters/AlphaNode2DPropertySetter.h"

using namespace Candera;


#include "hmi_trace_if.h"
#include "Widgets/widget_etg_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_ANIMATION
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/AnimationBaseHelper.cpp.trc.h"
#endif


// ------------------------------------------------------------------------
Candera::Animation::AnimationPlayer::SharedPointer AnimationBaseHelper::createAnimationPlayer()
{
   return Candera::Animation::AnimationPlayer::Create();
}


// ------------------------------------------------------------------------
Candera::Animation::AnimationController::SharedPointer AnimationBaseHelper::createAnimationController()
{
   return Candera::Animation::AnimationController::Create();
}


// ------------------------------------------------------------------------
void AnimationBaseHelper::setupAnimationPlayer(Candera::Animation::AnimationPlayer::SharedPointer& player,
      Candera::Animation::AnimationPlayer::PlayDirection direction,
      Candera::UInt32 repeatCount,
      FeatStd::Float speedFactor,
      FeatStd::Int32 duration)
{
   if (!player.PointsToNull())
   {
      player->SetDirection(direction);
      player->SetRepeatCount(static_cast<Candera::Int32>(repeatCount));
      player->SetSpeedFactor(speedFactor);
      player->SetSequenceDurationMs(duration);
   }
}


// ------------------------------------------------------------------------
bool AnimationBaseHelper::addPlayerToAnimationTimeDispatcher(const Candera::Animation::AnimationPlayer::SharedPointer& player, Candera::Animation::AnimationTimeDispatcher::SharedPointer& timeDispatcher)
{
   if (!player.PointsToNull())
   {
      if (!timeDispatcher.PointsToNull())
      {
         if (timeDispatcher->AddPlayer(player))
         {
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "Player added to time dispatcher successfully"));
            return true;
         }
      }
   }
   return false;
}


// ------------------------------------------------------------------------
bool AnimationBaseHelper::removePlayerFromAnimationTimeDispatcher(const Candera::Animation::AnimationPlayer::SharedPointer player, Candera::Animation::AnimationTimeDispatcher::SharedPointer& timeDispatcher)
{
   if (!player.PointsToNull())
   {
      if (!timeDispatcher.PointsToNull())
      {
         if (timeDispatcher->RemovePlayer(player))
         {
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "Player removed from time dispatcher successfully"));
            return true;
         }
      }
   }
   return false;
}


// ------------------------------------------------------------------------
bool AnimationBaseHelper::addListenerToPlayer(Candera::Animation::AnimationPlayer::SharedPointer& player, Candera::Animation::AnimationPlayerListener* listener)
{
   bool result = false;

   if (!player.PointsToNull() && listener != 0)
   {
      result = player->AddAnimationPlayerListener(listener);
   }
   return result;
}


bool AnimationBaseHelper::removeListenerFromPlayer(Candera::Animation::AnimationPlayer::SharedPointer player, Candera::Animation::AnimationPlayerListener* listener)
{
   bool result = false;

   if (!player.PointsToNull() && listener != 0)
   {
      result = player->RemoveAnimationPlayerListener(listener);
   }
   return result;
}


// ------------------------------------------------------------------------
void AnimationBaseHelper::setAnimationController(Candera::Animation::AnimationPlayer::SharedPointer& player, const Candera::Animation::AnimationController::SharedPointer& controller)
{
   if (!player.PointsToNull() && !controller.PointsToNull())
   {
      player->SetController(controller);
   }
}


// ------------------------------------------------------------------------
bool AnimationBaseHelper::addPropertyToAnimationController(Candera::Animation::AnimationController::SharedPointer& controller, const Candera::Animation::AnimationBlendedProperty::SharedPointer& prop)
{
   if (!controller.PointsToNull() && !prop.PointsToNull())
   {
      if (controller->AddProperty(prop))
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "Property Added successfully to AnimationController"));
         return true;
      }
      else
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "Property not added to AnimationController"));
      }
   }
   return false;
}


// ------------------------------------------------------------------------
Candera::Animation::AnimationBlendedProperty::SharedPointer AnimationBaseHelper::createAnimationBlendedProperty()
{
   return Candera::Animation::AnimationBlendedProperty::Create();
}


// ------------------------------------------------------------------------
Candera::Animation::AnimationKeyframeSequence::SharedPointer AnimationBaseHelper::createAnimationKeyFrameSequence()
{
   return Candera::Animation::AnimationKeyframeSequence::Create();
}


void AnimationBaseHelper::assignNodeToAnimate(const Candera::Animation::AnimationBlendedProperty::SharedPointer& animationblendedProperty, Candera::Node2D* node)
{
   if (!animationblendedProperty.PointsToNull() && node != 0)
   {
      if (animationblendedProperty != 0)
      {
         Candera::Animation::AnimationPropertySetter::SharedPointer propertySetter = animationblendedProperty->GetAnimationPropertySetter();
         if (propertySetter != 0)
         {
            if (propertySetter->IsTypeOf(Candera::Animation::Transformable2DScalePropertySetter::GetTypeId()) ||
                  propertySetter->IsTypeOf(Candera::Animation::Transformable2DTranslatePropertySetter::GetTypeId()) ||
                  propertySetter->IsTypeOf(Candera::Animation::Transformable2DRotatePropertySetter::GetTypeId()))
            {
               Candera::Animation::BaseTransformable2DPropertySetter* baseTps = dynamic_cast<Animation::BaseTransformable2DPropertySetter*>(propertySetter.GetPointerToSharedInstance());
               if (baseTps != 0)
               {
                  baseTps->SetTransformable(node);
               }
            }
            else
            {
               Candera::Animation::BaseNode2DPropertySetter* baseNps = dynamic_cast<Animation::BaseNode2DPropertySetter*>(propertySetter.GetPointerToSharedInstance());
               if (baseNps != 0)
               {
                  baseNps->SetNode(node);
               }
            }
         }
      }
   }
}


void AnimationBaseHelper::setAnimationRepeatCount(Candera::Animation::AnimationPlayer::SharedPointer player, Candera::UInt32 repeatCount)
{
   if (!player.PointsToNull())
   {
      player->SetRepeatCount(repeatCount);
   }
}


void AnimationBaseHelper::setAnimationSequenceDuration(Candera::Animation::AnimationPlayer::SharedPointer player, Candera::Animation::SequenceTimeType duration)
{
   if (!player.PointsToNull())
   {
      player->SetSequenceDurationMs(duration);
   }
}


bool AnimationBaseHelper::addKeyframeSequence(Candera::Animation::AnimationBlendedProperty::SharedPointer animationBPS, const FeatStd::MemoryManagement::SharedPointer<Candera::Animation::AnimationKeyframeSequence> animationKFS)
{
   if (!animationBPS.PointsToNull() && !animationKFS.PointsToNull())
   {
      return animationBPS->AddKeyframeSequence(animationKFS);
   }
   return false;
}


void AnimationBaseHelper::setInterpolationStrategy(Candera::Animation::InterpolationStrategy::SharedPointer strategy, Candera::Animation::AnimationKeyframeSequence::SharedPointer& animationKfs)
{
   if (!strategy.PointsToNull() && !animationKfs.PointsToNull())
   {
      if (animationKfs != NULL)
      {
         animationKfs->SetInterpolationStrategy(strategy);
      }
   }
}


bool AnimationBaseHelper::createAnimationPropertySetter(const Candera::Animation::AnimationBlendedProperty::SharedPointer& animationblendedProperty, Candera::Node2D* node, enAnimationProperty type)
{
   bool result = true;
   switch (type)
   {
      case enAnimateScale:
         AnimationBaseHelper::createPropertySetter<Candera::Animation::Transformable2DScalePropertySetter>(animationblendedProperty);
         AnimationBaseHelper::assignNodeToAnimate(animationblendedProperty, node);
         break;

      case enAnimateAlphaValue:
         AnimationBaseHelper::createPropertySetter<Candera::Animation::AlphaNode2DPropertySetter>(animationblendedProperty);
         AnimationBaseHelper::assignNodeToAnimate(animationblendedProperty, node);
         break;

      case enAnimatePosition:
         AnimationBaseHelper::createPropertySetter<Candera::Animation::Transformable2DTranslatePropertySetter>(animationblendedProperty);
         AnimationBaseHelper::assignNodeToAnimate(animationblendedProperty, node);
         break;

      case enAnimateRotation:
         AnimationBaseHelper::createPropertySetter<Candera::Animation::Transformable2DRotatePropertySetter>(animationblendedProperty);
         AnimationBaseHelper::assignNodeToAnimate(animationblendedProperty, node);
         break;

      case enAnimateRendering:
         AnimationBaseHelper::createPropertySetter<Candera::Animation::RenderingEnabledNode2DPropertySetter>(animationblendedProperty);
         AnimationBaseHelper::assignNodeToAnimate(animationblendedProperty, node);
         break;

      default:
         result = false;
         break;
   }
   return result;
}
