/* ***************************************************************************************
* FILE:          Adorner.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  Adorner.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#if !defined(Adorner_h)
#define Adorner_h

#include <FeatStd/MemoryManagement/SharedPointer.h>
#include <Candera/EngineBase/Animation/AnimationPlayerListener.h>
#include <AppUtils/DataSet.h>
#include <Widgets/utils/WidgetFunctors.h>
#include <Widgets/2D/BaseWidget2D.h>

//forward declarations
class AppViewHandler;


namespace hmibase {
namespace widget {
namespace adorner {

typedef hmibase::util::DataSet AdornerDataSet;
typedef hmibase::widget::utils::ComplexId4<unsigned int> AdornerMarkerId;
typedef hmibase::widget::utils::functors::MatchesComplexId4<unsigned int> AdornerMarkerFilter;


class AdornerAnimation;


/*****************************************************************************/
class Adorner
{
   public:
      FEATSTD_TYPEDEF_SHARED_POINTER(Adorner);
      CANDERA_RTTI_DECLARATION(Adorner);

      Adorner();
      virtual ~Adorner();

      /*****************************************************************************/
      virtual void onNodeDeleted(Candera::Node2D* adornerNode) = 0;

      /*****************************************************************************/
      virtual void show() = 0;
      virtual void hide() = 0;
      virtual void destroy() = 0;
      virtual void move(const Candera::Vector2& delta) = 0;

      /*****************************************************************************/
      bool isVisible() const
      {
         return _visible;
      }
      Candera::Widget2D* getSourceWidget()
      {
         return _sourceWidget.getObjectSafely();
      }
      Candera::Node2D* getNode()
      {
         return _node;
      }
      AdornerAnimation* getAnimation()
      {
         return _animation;
      }
      void setAnimation(AdornerAnimation* animation)
      {
         _animation = animation;
      }

      /*****************************************************************************/
      AdornerDataSet Data;

   protected:
      Widget2DAccessor _sourceWidget;
      Candera::Node2D* _node;
      AdornerAnimation* _animation;
      bool _visible;

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(Adorner);
      FEATSTD_SHARED_POINTER_DECLARATION();
};


/*****************************************************************************/
class AdornerAnimation
{
   public:
      CANDERA_RTTI_DECLARATION(AdornerAnimation);

      AdornerAnimation() {}
      virtual ~AdornerAnimation() {}

      /*****************************************************************************/
      /* Animation control */
      virtual bool start() = 0;
      virtual void stop() = 0;
      virtual void reset() = 0;
      virtual bool isRunning() const = 0;

      /*****************************************************************************/
      /* Attach/detach adorner */
      virtual bool attachToAdorner(Adorner::SharedPointer adorner) = 0;
      virtual void detachFromAdorner() = 0;
      virtual void onAdornerNodeDeleted(Candera::Node2D* adornerNode) = 0;

      /*****************************************************************************/
      /* Animation configuration */
      virtual Candera::Animation::SequenceTimeType getDuration() const = 0;
      virtual void setDuration(Candera::Animation::SequenceTimeType duration) = 0;

      virtual Candera::Vector2 getBeginValues() const = 0;
      virtual void setBeginValues(const Candera::Vector2& values) = 0;

      virtual Candera::Vector2 getEndValues() const = 0;
      virtual void setEndValues(const Candera::Vector2& values) = 0;

      /*****************************************************************************/
      Adorner::SharedPointer getAdorner() const
      {
         return _adorner;
      }

      /*****************************************************************************/
      class AnimationListener
      {
         public:
            virtual ~AnimationListener() {}
            virtual void onAnimationEnded(AdornerAnimation* animation) = 0;
      };

      virtual void addListener(AnimationListener& listener) = 0;
      virtual void removeListener(AnimationListener& listener) = 0;

      /*****************************************************************************/
      AdornerDataSet Data;

   protected:
      virtual void notifyAnimationEnded() = 0;

      Adorner::SharedPointer _adorner;

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(AdornerAnimation);
};


/*****************************************************************************/
struct AdornerMovementCapabilities
{
   AdornerMovementCapabilities(bool horizontal = true, bool vertical = true)
      : IsHorizontalMovementEnabled(horizontal), IsVerticalMovementEnabled(vertical)
   {
   }

   bool IsHorizontalMovementEnabled;
   bool IsVerticalMovementEnabled;
};


/*****************************************************************************/
class AdornerManager
{
   public:
      AdornerManager();
      virtual ~AdornerManager() {}

      /*****************************************************************************/
      /* Adorner control */
      virtual Adorner::SharedPointer createAdorner(Candera::Widget2D& widget, Candera::Node2D* adornerNode = NULL, Candera::Node2D* adornerContainer = NULL) = 0;
      virtual void destroyAdorner(Adorner::SharedPointer adorner) = 0;
      virtual void showAdorner(Adorner::SharedPointer adorner) = 0;
      virtual void hideAdorner(Adorner::SharedPointer adorner) = 0;
      virtual void moveAdorner(Adorner::SharedPointer adorner, const Candera::Vector2& delta) = 0;

      void setDefaultMovementCapabilities(const AdornerMovementCapabilities& defaultMovementCapabilities);
      const AdornerMovementCapabilities& getDefaultMovementCapabilities() const;

      /*****************************************************************************/
      /* Animation control */
      virtual AdornerAnimation* createAnimation(Candera::Widget2D& widget) = 0;
      virtual void destroyAnimation(AdornerAnimation* animation) = 0;
      virtual bool startAnimation(AdornerAnimation& animation) = 0;
      virtual void stopAnimation(AdornerAnimation& animation) = 0;
      virtual void resetAnimation(AdornerAnimation& animation) = 0;

      /*****************************************************************************/
      /* Attach/detach adorner */
      virtual bool attachAnimationToAdorner(AdornerAnimation& animation, Adorner::SharedPointer adorner) = 0;
      virtual void detachAnimationFromAdorner(AdornerAnimation& animation) = 0;

      /*****************************************************************************/
      /* Animation configuration */
      virtual void setAnimationDuration(AdornerAnimation& animation, Candera::Animation::SequenceTimeType duration) = 0;
      virtual void setAnimationBeginValues(AdornerAnimation& animation, const Candera::Vector2& values) = 0;
      virtual void setAnimationEndValues(AdornerAnimation& animation, const Candera::Vector2& values) = 0;

      void setDefaultAnimationDuration(Candera::Animation::SequenceTimeType defaultAnimationDuration);
      Candera::Animation::SequenceTimeType getDefaultAnimationDuration() const;

   protected:
      AdornerMovementCapabilities _defaultMovementCapabilities;
      Candera::Animation::SequenceTimeType _defaultAnimationDuration;
};


/*****************************************************************************/
class AdornerDynamicProperties
{
   public:
      static bool ClearAdorner(Candera::DynamicProperties::DynamicPropertyHost& object)
      {
         return object.ClearValue(CdaDynamicPropertyInstance(InternalAdornerWrapper));
      }

      static bool SetAdorner(Candera::DynamicProperties::DynamicPropertyHost& object, const Adorner::SharedPointer& adorner)
      {
         AdornerWrapper wrapper;
         wrapper._adorner = adorner;
         return object.SetValue(CdaDynamicPropertyInstance(InternalAdornerWrapper), wrapper);
      }

      static Adorner::SharedPointer GetAdorner(const Candera::DynamicProperties::DynamicPropertyHost& object)
      {
         return object.GetValue(CdaDynamicPropertyInstance(InternalAdornerWrapper))._adorner;
      }

      static const Candera::DynamicProperties::DynamicPropertyHost* ParentProvider(const Candera::DynamicProperties::DynamicPropertyHost*)
      {
         return NULL;
      }

      struct AdornerWrapper
      {
         Adorner::SharedPointer _adorner;

         bool operator==(const AdornerWrapper& other) const
         {
            return _adorner.GetPointerToSharedInstance() == other._adorner.GetPointerToSharedInstance();
         }

         bool operator!=(const AdornerWrapper& other) const
         {
            return !operator==(other);
         }
      };

   private:
      static const AdornerWrapper& DefaultInternalAdornerWrapper()
      {
         FEATSTD_UNSYNCED_STATIC_OBJECT(AdornerWrapper, _defaultValue);
         return _defaultValue;
      }

      CdaDynamicProperties(AdornerDynamicProperties, Candera::DynamicProperties::DynamicPropertyHost);

      CdaDynamicProperty(InternalAdornerWrapper, AdornerWrapper);
      CdaDynamicPropertyDefaultValue(DefaultInternalAdornerWrapper());
      CdaDynamicPropertyEnd();

      CdaDynamicPropertiesEnd();
};


}//adorner
}//widget
}//hmibase

namespace Candera {
namespace MetaInfo {

template<> struct DataType<hmibase::widget::adorner::AdornerDynamicProperties::AdornerWrapper>
{
   static inline const Char* GetEditor()
   {
      return "";
   }

   static inline bool ConvertToString(hmibase::widget::adorner::AdornerDynamicProperties::AdornerWrapper, Char*, UInt)
   {
      return true;
   }

   static inline bool ConvertFromString(hmibase::widget::adorner::AdornerDynamicProperties::AdornerWrapper, const Char*)
   {
      return true;
   }
};


}
}


#endif
