/* ***************************************************************************************
* FILE:          ImageBehavior.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ImageBehavior is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "ImageBehavior.h"
#include <CanderaPlatform/Device/Common/Effects/BitmapBrush.h>
#include <Candera/Engine2D/Core/RenderNode.h>

#include <Trace/ToString.h>
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_IMAGE
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ImageBehavior.cpp.trc.h"
#endif

namespace hmibase
{
namespace behavior
{
namespace image
{

CGI_BEHAVIOR_RTTI_DEFINITION(ImageBehavior2D)

ImageBehavior2D::ImageBehavior2D() : _invalid(true)
{
}

ImageBehavior2D::~ImageBehavior2D()
{
}

void ImageBehavior2D::Update()
{
   Base::Update();

   if (_invalid && (GetNode().ToNode2D() != NULL))
   {
      SetImage(*this, GetImage());

      _invalid = false;
   }
}

void ImageBehavior2D::SetImage(Candera::Behavior& behavior, const Candera::MemoryManagement::SharedPointer<Candera::Image2D>& image)
{
   Candera::RenderNode* renderNode = Candera::Dynamic_Cast<Candera::RenderNode*>(behavior.GetNode().ToNode2D());
   if ((renderNode != NULL) && (renderNode->GetEffect(0) != NULL))
   {
      Candera::BitmapBrush* bitmapBrush = Candera::Dynamic_Cast<Candera::BitmapBrush*>(renderNode->GetEffect(0)->GetBrushEffect2D());
      if (bitmapBrush != NULL)
      {
         SetImage(behavior, *bitmapBrush, image);
      }
   }
}

void ImageBehavior2D::SetImage(Candera::Behavior& behavior, Candera::BitmapBrush& bitmapBrush, const Candera::MemoryManagement::SharedPointer<Candera::Image2D>& image)
{
   Candera::Image2D* oldImage = bitmapBrush.Image().Get();

   //if both bitmaps are NULL or they point to same image=> do nothing
   if (oldImage != image.GetPointerToSharedInstance())
   {
      if (image.PointsToNull())
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SetImage image=null %s", HMIBASE_TO_STRING_VW(&behavior)));
      }
      else
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SetImage %s", HMIBASE_TO_STRING_VW(&behavior)));
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SetImage image=%s", HMIBASE_TO_STRING((const Candera::Image2D*)image.GetPointerToSharedInstance())));
      }
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "SetImage node=%s", HMIBASE_TO_STRING_SPN2D(behavior.GetNode().ToNode2D())));

      bool invalidateLayout = false;
      //if one of the bitmaps is NULL and the other one non NULL or if size is different=>Invalidate layout
      if ((oldImage == NULL)
            || image.PointsToNull()
            || (oldImage->GetWidth() != image->GetWidth())
            || (oldImage->GetHeight() != image->GetHeight()))
      {
         invalidateLayout = true;
      }

      bitmapBrush.Unload();
      bitmapBrush.Image().Set(image);
      bitmapBrush.Upload();

      if (invalidateLayout)
      {
         //invalidate the layout because size changed
         Candera::Layouter::InvalidateLayout(behavior.GetNode());
      }

      behavior.Invalidate();
   }
}


void ImageBehavior2D::OnEvent(const FeatStd::Event& /*event*/, Candera::EventDispatchResult& /*dispatchResult*/)
{
   //const Candera::PreNodeChangeEvent* preNodeChangedEvent = Candera::Dynamic_Cast<const Candera::PreNodeChangeEvent*>(&event);
   //if (preNodeChangedEvent != 0) {
   //    if (0 != GetNode().ToNode2D()) {
   //        SetPivotPointFromPivotHint();
   //    }
   //}
}

void ImageBehavior2D::OnChanged(PropertyChangeState state, FeatStd::UInt32 propertyId)
{
   Base::OnChanged(state, propertyId);

   if ((state == BeforeChange) && (propertyId == ImagePropertyId))
   {
      _invalid = true;
   }

}


}
}
}
