/******************************************************************
*COPYRIGHT: (C) 2017 Robert Bosch GmbH
*The reproduction, distribution and utilization of this file as
*well as the communication of its contents to others without express
*authorization is prohibited. Offenders will be held liable for the
*payment of damages. All rights reserved in the event of the grant
*of a patent, utility model or design.
******************************************************************/
#include "hmibase/gadget/videobuffer/DrmFdBuffer.h"
#include "hmibase/gadget/videobuffer/ApxInstance.h"
#include "hmibase/util/Error.h"
#include <unistd.h>

#include "hmibase/util/Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW
#include "trcGenProj/Header/DrmFdBuffer.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

using namespace hmibase::util;

namespace hmibase {
namespace gadget {
namespace videobuffer {

DrmFdBuffer::DrmFdBuffer(uint16_t width, uint16_t height, uint16_t depth, uint16_t bpp)
   : VideoBuffer(width, height, depth, bpp)
   , mApxPtr(NULL)
   , mFd(-1)
   , mIsOwner(false)
   , mDataPtr(NULL)
   , mDataSize(0)
   , mPitch(0)
{
   mApxPtr = new ApxInstance();
}


DrmFdBuffer::~DrmFdBuffer()
{
   if (mApxPtr != NULL)
   {
      mApxPtr->unmapBuffer(mDataPtr, mDataSize);

      if (mIsOwner)
      {
         uint32_t handle = mApxPtr->getHandle(mFd);
         mApxPtr->destroyDump(handle);
         close(mFd);
      }

      delete mApxPtr;
      mApxPtr = NULL;
   }

   mFd       = -1;
   mIsOwner  = false;
   mDataPtr  = NULL;
   mDataSize = 0;
   mPitch    = 0;
}


int32_t DrmFdBuffer::create()
{
   if (mFd != -1)
   {
      ETG_TRACE_ERR_THR(("Could not create buffer because it already exists. mFd=%d", mFd));
      return -1;
   }

   if (mApxPtr == NULL)
   {
      ETG_TRACE_ERR_THR(("Could not create buffer because mApxPtr == NULL. mFd=%d", mFd));
      return -1;
   }

   uint32_t size    = 0;
   uint32_t pitch   = 0;
   uint32_t minSize = getWidth() * getHeight() * getBpp();
   uint32_t handle  = mApxPtr->createDump(getWidth(), getHeight(), getBpp(), size, pitch);
   uint64_t offset  = mApxPtr->getDumpOffset(handle);
   void*    dataPtr = mApxPtr->mapBuffer(offset, size);
   int32_t  fd      = mApxPtr->getFd(handle);

   if (fd == -1 || dataPtr == NULL)
   {
      ETG_TRACE_ERR_THR(("Could not create buffer. mFd=%d, fd=%d, size=%d, minSize=%d, handle=%d, offset=%d, pitch=%d, dataPtr=%p", mFd, fd, size, minSize, handle, offset, pitch, dataPtr));
      if (dataPtr != NULL)
      {
         mApxPtr->unmapBuffer(dataPtr, size);
         dataPtr = NULL;
      }
      return -1;
   }

   mFd       = fd;
   mIsOwner  = true;
   mDataPtr  = (uint8_t*)dataPtr;
   mDataSize = size;
   mPitch    = pitch;

   return mFd;
}


void DrmFdBuffer::attach(int32_t fd, uint32_t size, uint32_t pitch)
{
   if (mFd != -1)
   {
      ETG_TRACE_ERR_THR(("Could not attach buffer because it already exists. mFd=%d, fd=%d, size=%d, pitch=%d", mFd, fd, size, pitch));
      return;
   }

   if (mApxPtr == NULL)
   {
      ETG_TRACE_ERR_THR(("Could not attach buffer because mApxPtr == NULL. mFd=%d, fd=%d, size=%d, pitch=%d", mFd, fd, size, pitch));
      return;
   }

   uint32_t handle  = mApxPtr->getHandle(fd);
   uint64_t offset  = mApxPtr->getDumpOffset(handle);
   void*    dataPtr = mApxPtr->mapBuffer(offset, size);

   if (dataPtr == NULL)
   {
      ETG_TRACE_ERR_THR(("Could not attach buffer. mFd=%d, fd=%d, size=%d, pitch=%d, handle=%d, offset=%d", mFd, fd, size, pitch, handle, offset));
      return;
   }

   mFd       = fd;
   mIsOwner  = false;
   mDataPtr  = (uint8_t*)dataPtr;
   mDataSize = size;
   mPitch    = pitch;
}


uint8_t* DrmFdBuffer::getDataPtr() const
{
   return mDataPtr;
}


uint32_t DrmFdBuffer::getDataSize() const
{
   return mDataSize;
}


uint32_t DrmFdBuffer::getPitch() const
{
   return mPitch;
}


int32_t DrmFdBuffer::getFd() const
{
   return mFd;
}


} // namespace
} // namespace
} // namespace
