/******************************************************************
*COPYRIGHT: (C) 2017 Robert Bosch GmbH
*The reproduction, distribution and utilization of this file as
*well as the communication of its contents to others without express
*authorization is prohibited. Offenders will be held liable for the
*payment of damages. All rights reserved in the event of the grant
*of a patent, utility model or design.
******************************************************************/
#include "hmibase/gadget/videobuffer/ApxContext.h"
#include "hmibase/util/Error.h"
#include <stdint.h>
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>

extern "C" {
#include <apx.h>
}


#include <drm.h>
#include <xf86drm.h>
#include <xf86drmMode.h>
#include <sys/mman.h>


#include "hmibase/util/Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW
#include "trcGenProj/Header/ApxContext.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

using namespace hmibase::util;

namespace hmibase {
namespace gadget {
namespace videobuffer {

ApxContext::ApxContext()
   : mApxFd(-1)
   , mApxPtr(NULL)
   , mWlDisplayPtr(NULL)
{
   Error::reset();
   mWlDisplayPtr = wl_display_connect(NULL);
   if (mWlDisplayPtr == NULL || Error::is())
   {
      ETG_TRACE_ERR_THR(("Cannot connect wl_display. %s", Error::getMessage().cPtr()));
      return;
   }

   Error::reset();
   wl_registry* wlRegistryPtr = wl_display_get_registry(mWlDisplayPtr);
   if (wlRegistryPtr == NULL || Error::is())
   {
      ETG_TRACE_ERR_THR(("Cannot get wl_registry. %s", Error::getMessage().cPtr()));
      return;
   }

   mApxPtr = new apx();
   memset(mApxPtr, 0, sizeof(apx));

   wl_registry_listener wlRegistryListener;
   wlRegistryListener.global        = ApxContext::staticCallback;
   wlRegistryListener.global_remove = NULL;
   wl_registry_add_listener(wlRegistryPtr, &wlRegistryListener, this);

   wl_display_dispatch(mWlDisplayPtr);
   wl_display_roundtrip(mWlDisplayPtr);

   Error::reset();
   mApxFd = open("/dev/dri/card0", O_RDWR);
   if (mApxFd < 0 || Error::is())
   {
      ETG_TRACE_ERR_THR(("Cannot open /dev/dri/card0. %s", Error::getMessage().cPtr()));
      return;
   }

   Error::reset();
   drm_magic_t magic = 0;
   int ret0 = drmGetMagic(mApxFd, &magic);
   if (ret0 != 0 || Error::is())
   {
      ETG_TRACE_ERR_THR(("Cannot get magic for mApxFd=%d. %s", mApxFd, Error::getMessage().cPtr()));
      return;
   }

   Error::reset();
   apx_init(mApxPtr);
   if (Error::is())
   {
      ETG_TRACE_ERR_THR(("Cannot initialize apx. %s", Error::getMessage().cPtr()));
      return;
   }

   Error::reset();
   int ret1 = apx_authenticate(mApxPtr, mWlDisplayPtr, magic);
   if (ret1 != 0 || Error::is())
   {
      ETG_TRACE_ERR_THR(("Cannot Authenticate. %s", Error::getMessage().cPtr()));
      return;
   }
}


ApxContext::~ApxContext()
{
   if (mApxPtr != NULL)
   {
      apx_deinit(mApxPtr);
      delete mApxPtr;
      mApxPtr = NULL;
   }

   if (mWlDisplayPtr != NULL)
   {
      wl_display_disconnect(mWlDisplayPtr);
      mWlDisplayPtr = NULL;
   }

   if (mApxFd != -1)
   {
      close(mApxFd);
      mApxFd = -1;
   }
}


int ApxContext::getApxFd()
{
   return mApxFd;
}


void ApxContext::staticCallback(void* dataPtr, wl_registry* wlRegistryPtr, uint32_t name, const char* namePtr, uint32_t version)
{
   ApxContext* thisPtr = (ApxContext*)dataPtr;
   if (thisPtr != NULL)
   {
      thisPtr->objectCallback(wlRegistryPtr, name, namePtr, version);
   }
}


void ApxContext::objectCallback(wl_registry* wlRegistryPtr, uint32_t name, const char* namePtr, uint32_t version)
{
   apx_global_handler(mApxPtr, wlRegistryPtr, name, namePtr, version);
}


} // namespace
} // namespace
} // namespace
