/* ***************************************************************************************
* FILE:          UniqueUID.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  UniqueUID.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "hmibase/util/UniqueUID.h"
#include <string.h>

namespace hmibase {
namespace util {

UniqueUID::UniqueUID()
{
#ifdef _MSC_VER
   UuidCreate(&_uuid);
#else
   uuid_generate_random(_uuid);
#endif
}


UniqueUID::UniqueUID(const uuid_t& uuid)
{
   copy(_uuid, uuid);
}


UniqueUID::UniqueUID(const UniqueUID& rhs)
{
   copy(_uuid, rhs._uuid);
}


UniqueUID::~UniqueUID()
{
}


bool UniqueUID::operator == (const UniqueUID& rhs)
{
   return compare(_uuid, rhs._uuid) == 0 ? true : false;
}


bool UniqueUID::operator != (const UniqueUID& rhs)
{
   return compare(_uuid, rhs._uuid) != 0 ? true : false;
}


UniqueUID& UniqueUID::operator = (const UniqueUID& rhs)
{
   copy(_uuid, rhs._uuid);

   return *this;
}


bool UniqueUID::isNull()
{
#ifdef _MSC_VER
   RPC_STATUS status;
   int ret = UuidIsNil(&_uuid, &status);

   return (status == RPC_S_OK) ? ((ret == 1) ? true : false) : false;
#else
   return uuid_is_null(_uuid);
#endif
}


void UniqueUID::clear()
{
#ifdef _MSC_VER
   UuidCreateNil(&_uuid);
#else
   uuid_clear(_uuid);
#endif
}


void UniqueUID::getUUID(uuid_t& uuid)
{
   copy(uuid, _uuid);
}


std::string UniqueUID::toString(const hmibase::util::UniqueUID& uniqueUID)
{
   return toString(uniqueUID._uuid);
}


std::string UniqueUID::toString(const uuid_t& uuid)
{
#ifdef _MSC_VER
   unsigned char* str;
   std::string result;
   UuidToStringA(&uuid, &str);

   result = (char*)str;

   RpcStringFreeA(&str);

   return result;
#else
   char s[37];
   uuid_unparse(uuid, s);

   return s;
#endif
}


int UniqueUID::compare(const uuid_t& a, const uuid_t& b)
{
#ifdef _MSC_VER
   RPC_STATUS status;
   int ret = UuidEqual(const_cast<uuid_t*>(&a), const_cast<uuid_t*>(&b), &status);
   if (status == RPC_S_OK)
   {
      return ret == 1 ? 0 : -1;
   }
   else
   {
      return -1; // an error occured during comparison
   }
#else
   return uuid_compare(a, b);
#endif
}


void UniqueUID::copy(uuid_t& dst, const uuid_t& src)
{
#ifdef _MSC_VER
   memcpy(&dst, &src, sizeof(uuid_t));
#else
   uuid_copy(dst, src);
#endif
}


}
}
