/******************************************************************
*COPYRIGHT: (C) 2017 Robert Bosch GmbH
*The reproduction, distribution and utilization of this file as
*well as the communication of its contents to others without express
*authorization is prohibited. Offenders will be held liable for the
*payment of damages. All rights reserved in the event of the grant
*of a patent, utility model or design.
******************************************************************/
#include "hmibase/util/TypeCounter.h"

#include "hmibase/util/Trace.h"
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW_UTIL
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/TypeCounter.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

namespace hmibase {
namespace util {

TypeCounter::TypeCounter()
   : mMutex()
   , mObjectPtrs()
{
}


TypeCounter::~TypeCounter()
{
}


void TypeCounter::clear()
{
   mMutex.lock();
   mObjectPtrs.clear();
   mMutex.unlock();
}


void TypeCounter::registerObject(TypeCountable* objectPtr)
{
   mMutex.lock();
   mObjectPtrs.insert(objectPtr);
   mMutex.unlock();
}


void TypeCounter::releaseObject(TypeCountable* objectPtr)
{
   mMutex.lock();
   mObjectPtrs.erase(objectPtr);
   mMutex.unlock();
}


int TypeCounter::getTypeCount(const SimpleString& typeName)
{
   std::map<SimpleString, int> typeCounts = getTypeCounts();
   if (typeCounts.find(typeName) == typeCounts.end())
   {
      return 0;
   }
   else
   {
      return getTypeCounts()[typeName];
   }
}


std::map<SimpleString, int> TypeCounter::getTypeCounts()
{
   std::map<SimpleString, int> result;
   mMutex.lock();
   for (std::set<TypeCountable*>::const_iterator iter = mObjectPtrs.begin(); iter != mObjectPtrs.end(); iter++)
   {
      TypeCountable* tcPtr = *iter;
      if (tcPtr == NULL)
      {
         result["NULL"]++;
      }
      else
      {
         result[tcPtr->getTypeName()]++;
      }
   }
   mMutex.unlock();
   return result;
}


void TypeCounter::print()
{
   std::map<SimpleString, int> typeCounts = getTypeCounts();

   // First loop through all typeNames and find the number of chars in the longest one
   int maxLength = 0;
   for (std::map<SimpleString, int>::const_iterator iter = typeCounts.begin(); iter != typeCounts.end(); iter++)
   {
      SimpleString typeName  = iter->first;
      //int          typeCount = iter->second;

      if (maxLength < (int)typeName.length())
      {
         maxLength = (int)typeName.length();
      }
   }

   // Print intro
   int numTypes  = (int)typeCounts.size();
   ETG_TRACE_USR4_THR(("Printing number of instances of %d types:", numTypes));

   // Second loop through all typeNames, this time e print them to log
   for (std::map<SimpleString, int>::const_iterator iter = typeCounts.begin(); iter != typeCounts.end(); iter++)
   {
      SimpleString typeName  = iter->first;
      int          typeCount = iter->second;

      SimpleString align;
      for (int i = (int)typeName.length(); i < maxLength; i++)
      {
         align += " ";
      }

      SimpleString line = SS("    ") + typeName + align + SS(" = ") + SS(typeCount);
      ETG_TRACE_USR4_THR(("%s", line.cPtr()));
   }
}


} // namespace
} // namespace
