/* ***************************************************************************************
* FILE:          TimerThreaded.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TimerThreaded.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "hmibase/util/TimerThreaded.h"
#include <stdio.h>
#include <string.h>

#include "hmibase/util/Trace.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_UTIL
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/TimerThreaded.cpp.trc.h"
#endif


namespace hmibase {
namespace util {

namespace Internal {

class LocalWorkThread : ::hmibase::util::Thread, ::hmibase::util::Thread::IThreadFunction
{
   public:
      static LocalWorkThread& getInstance(bool createTheInstance = false);

      void add(TimerThreaded& timer);
      void remove(const TimerThreaded& timer);

   protected:
      virtual bool threadWorkProc();

      typedef std::vector<class TimerThreaded*> TimerList;

      ::hmibase::util::Condition  _cond;
      TimerList	_timerList;
      bool	      _running;
      LocalWorkThread();
      virtual ~LocalWorkThread();
      HMIBASE_UNCOPYABLE(LocalWorkThread);
};


class ScopedCriticalSection
{
   public:
      ScopedCriticalSection()
      {
         lock();
      }
      ~ScopedCriticalSection()
      {
         unlock();
      }
      static void lock()
      {
         s_criticalSection.lock();
      }
      static void unlock()
      {
         s_criticalSection.unlock();
      }
   private:
      static hmibase::util::Mutex s_criticalSection;
      HMIBASE_UNCOPYABLE(ScopedCriticalSection);
};


hmibase::util::Mutex ScopedCriticalSection::s_criticalSection;
} // namespace Internal


/******************************************************************************
*  Create the instance
******************************************************************************/
bool TimerThreaded::createTimerThread()
{
   return &Internal::LocalWorkThread::getInstance(true) != 0;
}


/******************************************************************************
*  Constructor
******************************************************************************/
TimerThreaded::TimerThreaded() :
   _status(Stopped),
   _repeatTime(0),
   _timIndex(0),
   _repeatCounter(0),
   _expiringTime(0),
   _remainingTime(0),
   _fktTimerCallback(0),
   _timerId(-1)
{
   memset(_timeout, 0, sizeof(_timeout));
}


/******************************************************************************
*  Destructor
******************************************************************************/
TimerThreaded::~TimerThreaded()
{
   stop();
   resetTimeouts();
}


/******************************************************************************
*  setName
******************************************************************************/
void TimerThreaded::setName(const char* compName, const char* instanceName)
{
   char temp[50];
   SNPRINTF(temp, sizeof(temp), "%s:%s", compName ? compName : "?", instanceName ? instanceName : "?");
   _name = temp;
   ETG_TRACE_USR4_THR(("TimerThreaded::setName ->'%s'", _name.c_str()));
}


/******************************************************************************
*  setTimeout
******************************************************************************/
bool TimerThreaded::setTimeout(uint32_t i, uint32_t timeout, TimerThreaded::ITimerFunction* fktTimerCallback)
{
   ETG_TRACE_USR4_THR(("TimerThreaded::setTimeout() i=%d t=%d ->'%s'", i, timeout, _name.c_str()));
   _fktTimerCallback = fktTimerCallback;
   if (i >= (TABSIZE(_timeout) - 1))     //since there has to be one empty element at the end
   {
      return false;
   }
   if (i > 0 && timeout <= _timeout[i - 1])
   {
      return false;
   }
   if (i == 0)
   {
      stop();
   }
   _repeatTime = 0;
   _timeout[i + 0] = timeout;
   _timeout[i + 1] = 0;
   return true;
}


/******************************************************************************
*  setTimeout
******************************************************************************/
bool TimerThreaded::setTimeout(uint32_t timeout, TimerThreaded::ITimerFunction* fktTimerCallback)
{
   return setTimeout(0, timeout, fktTimerCallback);
}


/******************************************************************************
*  setTimeoutWithRepeat
******************************************************************************/
bool TimerThreaded::setTimeoutWithRepeat(uint32_t timeout, uint32_t repeatTime, TimerThreaded::ITimerFunction* fktTimerCallback)
{
   ETG_TRACE_USR4_THR(("TimerThreaded::setTimeoutWithRepeat() t=%d  repeat=%d ->'%s'", timeout, repeatTime, _name.c_str()));
   _fktTimerCallback = fktTimerCallback;
   bool rc = setTimeout(0, timeout, fktTimerCallback);
   _repeatTime = repeatTime;
   return rc;
}


/******************************************************************************
*  resetTimeouts
******************************************************************************/
void TimerThreaded::resetTimeouts()
{
   stop();
   memset(&_timeout, 0, sizeof(_timeout));
   _repeatTime = 0;
}


/******************************************************************************
*  start
******************************************************************************/
void TimerThreaded::start()
{
   _startTicks.start();
   startInternal(_timeout[0]);
}


/******************************************************************************
*  restart
******************************************************************************/
void TimerThreaded::restart()
{
   _timIndex = 0;
   _repeatCounter = 0;
   stop();
   start();
}


/******************************************************************************
*  stop
******************************************************************************/
void TimerThreaded::stop()
{
   ETG_TRACE_USR4_THR(("TimerThreaded::stop ->'%s'", _name.c_str()));
   _timIndex = 0;
   _repeatCounter = 0;
   stopInternal();
}


/******************************************************************************
*  stopInternal
******************************************************************************/
void TimerThreaded::stopInternal()
{
   ETG_TRACE_USR4_THR(("TimerThreaded::stopinternal ->'%s'", _name.c_str()));
   Internal::ScopedCriticalSection scs;

   Internal::LocalWorkThread::getInstance().remove(*this);
   _remainingTime = 0;
   _status = Stopped;
}


/******************************************************************************
*  pause
******************************************************************************/
void TimerThreaded::pause()
{
   ETG_TRACE_USR4_THR(("TimerThreaded::pause ->'%s'", _name.c_str()));
   Internal::ScopedCriticalSection scs;

   uint32_t now = hmibase::util::Ticker::getTickCountMsec();
   stopInternal();
   if (now < _expiringTime)
   {
      _remainingTime = _expiringTime - now;
      _status = Paused;
   }
}


/******************************************************************************
*  resume
******************************************************************************/
void TimerThreaded::resume()
{
   ETG_TRACE_USR4_THR(("TimerThreaded::resume ->'%s'", _name.c_str()));
   Internal::ScopedCriticalSection scs;

   if (_status == Paused && _remainingTime > 0)
   {
      startInternal(_remainingTime);
   }
}


/******************************************************************************
*  startInternal
******************************************************************************/
void TimerThreaded::startInternal(uint32_t timeout)
{
   if (timeout == 0)
   {
      return;
   }

   ETG_TRACE_USR4_THR(("TimerThreaded::startInternal ->'%s'", _name.c_str()));

   Internal::ScopedCriticalSection scs;

   if (_status != Running)
   {
      _expiringTime = hmibase::util::Ticker::getTickCountMsec() + timeout;
      Internal::LocalWorkThread::getInstance().add(*this);
   }
   _status = Running;
}


/******************************************************************************
*  expire
******************************************************************************/
void TimerThreaded::expire()
{
   if (_status != Running)
   {
      return;
   }

   ETG_TRACE_USR4_THR(("TimerThreaded::expire ->'%s'", _name.c_str()));

   stopInternal();
   _status = Expired;

   bool multiTimeOuts = _timeout[1] != 0;

   if (_timIndex < (TABSIZE(_timeout) - 1) && _timeout[_timIndex])
   {
      if (_fktTimerCallback)
      {
         _fktTimerCallback->timerExpired(_timerId, _timIndex);
      }

      if (multiTimeOuts)
      {
         _timIndex++;
         if (_timeout[_timIndex])
         {
            uint32_t elapsed = getElapsedTimeSinceStart();
            if (elapsed < _timeout[_timIndex])
            {
               uint32_t nextTime = _timeout[_timIndex] - elapsed;
               startInternal(nextTime);
            }
         }
      }
   }
   if (_repeatTime > 0)
   {
      _repeatCounter++;
      startInternal(_repeatTime);
   }
}


namespace Internal {

/******************************************************************************
*  LocalWorkThread constructor
******************************************************************************/
LocalWorkThread::LocalWorkThread() :
   _running(false)
{
   //create(this, 0, "LocalWorkThread");
}


/******************************************************************************
*  LocalWorkThread destructor
******************************************************************************/
LocalWorkThread::~LocalWorkThread()
{
   Internal::ScopedCriticalSection scs;
   _running = false;

   TimerList::iterator it(_timerList.begin());
   while (it != _timerList.end())
   {
      (*it)->stopInternal();
      it = _timerList.begin();
   }
   _timerList.clear();
}


/******************************************************************************
*  LocalWorkThread::threadWorkProc
******************************************************************************/
bool LocalWorkThread::threadWorkProc()
{
   _running = true;

   while (_running)
   {
      uint32_t timeout(0);
      {
         Internal::ScopedCriticalSection scs;

         TimerList::iterator it(_timerList.begin());

         // look for all timers in the list which are expired, fire msg and remove them from the list
         while (it != _timerList.end() && (hmibase::util::Ticker::getTickCountMsec() >= (*it)->getExpiringTime()))
         {
            Internal::ScopedCriticalSection::lock();
            (*it)->expire();
            Internal::ScopedCriticalSection::unlock();
            it = _timerList.begin();
         }

         // calculate the time until next timer expires
         // if no timer is in the list, semaphore will be obtained for infinite time (semaphore gets released by adding a timer to the list)
         it = _timerList.begin();

         if (it != _timerList.end())
         {
            uint32_t tNow = hmibase::util::Ticker::getTickCountMsec();
            if (tNow < (*it)->getExpiringTime())
            {
               timeout = (*it)->getExpiringTime() - tNow;
            }
         }
      }

      if (timeout > 0)
      {
         _cond.wait((uint32_t)timeout);
      }
      else
      {
         if (_timerList.size() == 0)
         {
            _cond.wait(0xffffffff);
         }
      }
   }

   return 0;
}


/******************************************************************************
*  LocalWorkThread::getInstance
******************************************************************************/
LocalWorkThread& LocalWorkThread::getInstance(bool createTheInstance)
{
   static LocalWorkThread* g_timerThread;

#ifdef WIN32
   // !! SCHOST build
   if (g_timerThread == 0)
#else
   // !! Timer thread has to be created at startup of application
   if (createTheInstance == true && g_timerThread == 0)
#endif
   {
      g_timerThread = new LocalWorkThread();
      if (g_timerThread)
      {
         g_timerThread->create(g_timerThread, 0, "LocalWorkThread");
      }
   }
   if (g_timerThread == 0)
   {
      ETG_TRACE_ERRMEM(("FATAL - Timer thread has to be created at startup of application"));
      HMI_APP_ASSERT_ALWAYS();
   }
   return *g_timerThread;
}


/******************************************************************************
*  LocalWorkThread::add
******************************************************************************/
void LocalWorkThread::add(TimerThreaded& timer)
{
   {
      ScopedCriticalSection scs;

      TimerList::iterator it = _timerList.begin();

      while (it != _timerList.end())
      {
         if (timer.getExpiringTime() < (*it)->getExpiringTime())
         {
            break;
         }
         ++it;
      }

      _timerList.insert(it, &timer);
   }

   _cond.signal(); // release, notify, post, ...
}


/******************************************************************************
*  LocalWorkThread::remove
******************************************************************************/
void LocalWorkThread::remove(const TimerThreaded& timer)
{
   {
      ScopedCriticalSection scs;

      for (TimerList::iterator it(_timerList.begin()); it != _timerList.end(); ++it)
      {
         if ((*it) == &timer)
         {
            _timerList.erase(it);
            break;
         }
      }
   }
   _cond.signal();
}


} // namespace internal
} // namespace util
} // namespace hmibase
