/******************************************************************
*COPYRIGHT: (C) 2017 Robert Bosch GmbH
*The reproduction, distribution and utilization of this file as
*well as the communication of its contents to others without express
*authorization is prohibited. Offenders will be held liable for the
*payment of damages. All rights reserved in the event of the grant
*of a patent, utility model or design.
******************************************************************/
#include "hmibase/util/SimpleThreadImpl.h"

#include "hmibase/util/Trace.h"
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW_UTIL
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/SimpleThreadImpl.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

namespace hmibase {
namespace util {

SimpleThreadImpl::SimpleThreadImpl(Threadable& threadable, const SimpleString& name)
   : mName(name)
   , mThreadable(threadable)
   , mKeepRunning(true)
   , mStateMutex()
   , mState(NONE)
{
}


SimpleThreadImpl::~SimpleThreadImpl()
{
   // Derived classes must call finish() in their destructor.
}


SimpleThreadImpl::State SimpleThreadImpl::getState()
{
   mStateMutex.lock();
   State state = mState;
   mStateMutex.unlock();
   return state;
}


void SimpleThreadImpl::setState(SimpleThreadImpl::State state)
{
   mStateMutex.lock();
   mState = state;
   mStateMutex.unlock();
}


const SimpleString& SimpleThreadImpl::getName() const
{
   return mName;
}


void SimpleThreadImpl::start()
{
   if (getState() != NONE)
   {
      ETG_TRACE_ERR_THR(("%s", createMessage("Cannot start thread again. (state != NONE)").cPtr()));
      return;
   }

   setState(INITIALIZING);

   if (createThread())
   {
      while (getState() == INITIALIZING)
      {
         sleep(1);
      }
   }
}


void SimpleThreadImpl::finish()
{
   mKeepRunning = false;
   ETG_TRACE_USR4_THR(("%s", createMessage("Finishing Thread").cPtr()));

   joinThread();
   setState(JOINED);

   ETG_TRACE_USR4_THR(("%s", createMessage("Finished Thread").cPtr()));
}


bool SimpleThreadImpl::isRunning()
{
   State state = getState();
   switch (state)
   {
      case NONE:
      case FINISHED:
      case JOINED:
         return false;
      default:
         return true;
   }
}


void SimpleThreadImpl::localThreadRun()
{
   setState(STARTING);
   mThreadable.onStart(mKeepRunning);

   setState(RUNNING);
   while (mThreadable.onRun(mKeepRunning) && mKeepRunning)
   {
      sleep(0);
   }

   setState(FINISHING);
   mThreadable.onFinish(mKeepRunning);

   setState(FINISHED);
}


SimpleString SimpleThreadImpl::createMessage(const SimpleString& message)
{
   return SimpleString::format("SimpleThreadImpl[%s], state=%d, mKeepRunning=%d: %s", getName().cPtr(), getState(), mKeepRunning, message.cPtr());
}


} // namespace
} // namespace
