/******************************************************************
*COPYRIGHT: (C) 2017 Robert Bosch GmbH
*The reproduction, distribution and utilization of this file as
*well as the communication of its contents to others without express
*authorization is prohibited. Offenders will be held liable for the
*payment of damages. All rights reserved in the event of the grant
*of a patent, utility model or design.
******************************************************************/
#include "hmibase/util/SharedMemory.h"
#include "hmibase/util/Error.h"
#include "hmibase/util/Permissions.h"
#include <sys/mman.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>

#include "hmibase/util/Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW_UTIL
#include "trcGenProj/Header/SharedMemory.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

namespace hmibase {
namespace util {

SharedMemory::SharedMemory(const char* name, const size_t numBytes, bool isMaster)
   : mName(fixName(name)), mIsOwner(isMaster), mFileDescriptor(-1), mNumBytes(0), mDataPtr(NULL)
{
   // Open or create shared memory
   int oFlags = O_RDWR;
   int mode   = 0;

   // get current umask
   // this is needed as setting write permissions does not work with current umask,
   // so we set umask to 0 open shm with desired mode and restore umask afterwards
   mode_t old_umask = umask(0);
   if (isMaster)
   {
      // Just in case there are leftovers from last run
      SharedMemory::purge(name);

      oFlags |= O_CREAT | O_EXCL;

      mode   |= S_IRWXU | S_IRWXG; // Read Write Execute for user and group
   }

   Error::reset();
   mFileDescriptor = shm_open(mName.cPtr(), oFlags, mode);

   // restore umask
   umask(old_umask);

   if (Error::is() || mFileDescriptor == -1)
   {
      ETG_TRACE_ERR_THR(("%s", createErrorMessage("shm_open failed.").cPtr()));
      return;
   }

   // If we created the shared memory, we also have to set the access rights
   if (isMaster)
   {
      Error::reset();
      int fchownResult = fchown(mFileDescriptor, (uid_t) - 1, hmibase::util::group::aid_hmibase);
      if (Error::is() || fchownResult == -1)
      {
         ETG_TRACE_ERR_THR(("%s", createErrorMessage("fchown(hmibase::util::group::aid_hmibase) failed.").cPtr()));
         // Keep going, even if we could not set user.
         // Some platforms do not have the user, eg unittests.
         //return;
      }

      Error::reset();
      int fchmodResult = fchmod(mFileDescriptor, 0660);
      if (Error::is() || fchmodResult == -1)
      {
         ETG_TRACE_ERR_THR(("%s", createErrorMessage("fchmod failed.").cPtr()));
         return;
      }
   }

   Error::reset();
   int truncResult = ftruncate(mFileDescriptor, numBytes);
   if (Error::is() || truncResult == -1)
   {
      ETG_TRACE_ERR_THR(("%s", createErrorMessage("ftruncate failed.").cPtr()));
      return;
   }

   // Map local Memory
   int prot  = PROT_READ | PROT_WRITE;
   int flags = MAP_SHARED;
   Error::reset();
   mDataPtr  = mmap(NULL, numBytes, prot, flags, mFileDescriptor, 0);
   mNumBytes = numBytes;

   if (mDataPtr == MAP_FAILED || Error::is())
   {
      ETG_TRACE_ERR_THR(("%s", createErrorMessage("mmap failed.").cPtr()));
      mDataPtr  = NULL;
      mNumBytes = 0;
   }
}


SharedMemory::~SharedMemory()
{
   if (mDataPtr != NULL)
   {
      Error::reset();
      munmap(mDataPtr, mNumBytes);
      if (Error::is())
      {
         ETG_TRACE_ERR_THR(("%s", createErrorMessage("munmap failed.").cPtr()));
      }

      mDataPtr  = NULL;
      mNumBytes = 0;
   }

   if (mFileDescriptor > -1)
   {
      Error::reset();
      close(mFileDescriptor);
      if (Error::is())
      {
         ETG_TRACE_ERR_THR(("%s", createErrorMessage("close failed.").cPtr()));
      }
   }

   if (mIsOwner)
   {
      Error::reset();
      shm_unlink(mName.cPtr());
      if (Error::is())
      {
         ETG_TRACE_ERR_THR(("%s", createErrorMessage("shm_unlink failed.").cPtr()));
      }
   }
}


void SharedMemory::purge(const char* name)
{
   shm_unlink(fixName(name).cPtr());
}


SimpleString SharedMemory::createErrorMessage(const SimpleString& message) const
{
   return SimpleString::format("SharedMemory[%s]: %s %s", mName.cPtr(), message.cPtr(), Error::getMessage().cPtr());
}


SimpleString SharedMemory::fixName(const SimpleString& name)
{
   if (name.startsWith("/"))
   {
      return name;
   }
   else
   {
      return SimpleString("/") + name;
   }
}


} // namespace
} // namespace
