/* ***************************************************************************************
* FILE:          Semaphore.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  Semaphore.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "hmibase/util/Semaphore.h"
#include "hmibase/util/Permissions.h"

#ifdef WIN32
#include <windows.h>
#include <WinBase.h>
#else
#include <semaphore.h>
#include <unistd.h>
#include <sys/stat.h>
#endif

#include <fcntl.h>
#include <errno.h>

#include "hmibase/util/Trace.h"
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW_UTIL
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/Semaphore.cpp.trc.h"
#endif

namespace hmibase {
namespace util {

/////////////////////////////////////////////////////////////////////////////////////
///// Semaphore Implementation //////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////

namespace internal {

#ifndef WIN32
class SemaPosix : public ::hmibase::util::internal::SemaphoreImpl
{
   public:
      SemaPosix() : _sem(0), _isCreator(false)
      {
      }

      virtual ~SemaPosix()
      {
         destroy();
         _sem = 0;
      }

      bool create(unsigned int nCount, const char* name)
      {
         HMI_APP_ASSERT(name != 0);

         _name = name;

         // use unlink only when semaphore should be removed from the system, will unlock and close all references in use
         //sem_unlink(name);
         _sem = ::sem_open(name, O_EXCL | O_CREAT, 0660 , nCount);

         if (_sem == SEM_FAILED)
         {
            if ((errno == EEXIST) || (errno == EACCES))
            {
               // named semaphore already exists or is owned by another user, so simply try to open
               _sem = ::sem_open(name, 0);
            }
         }
         else
         {
            std::string namedSemPath = "/dev/shm/sem.";
            namedSemPath.append(name);
            _isCreator = true;

            if (chown(namedSemPath.c_str(), (uid_t) - 1, hmibase::util::group::aid_hmibase) == -1)
            {
               ETG_TRACE_FATAL_THR(("hmibase::util::SemaPosix, sem_open chown failed: errno %d", errno));
            }
            if (chmod(namedSemPath.c_str(), 0660) == -1)
            {
               ETG_TRACE_FATAL_THR(("hmibase::util::SemaPosix, sem_open chmod failed: errno %d", errno));
            }
         }

         if (_sem == SEM_FAILED)
         {
            ETG_TRACE_FATAL_THR(("hmibase::util::SemaPosix, sem_open failed: errno %d", errno));
            return false;
         }
         return true;
      }

      void destroy()
      {
         if (_sem)
         {
            if (sem_close(_sem) == -1)
            {
               ETG_TRACE_ERR_THR(("hmibase::util::SemaPosix, sem_close failed! errno %d", errno));
            }

            if (_name.length() > 0)
            {
               // use unlink only when semaphore should be removed from the system, will unlock and close all references in use
               //if (sem_unlink(_name.c_str()) == -1)
               //{
               //   ETG_TRACE_ERR_THR(("hmibase::util::SemaPosix, sem_unlink failed!"));
               //}
            }
            _sem = NULL;
         }
      }

      bool wait()   // decrement
      {
         int res = -1;
         if (_sem)
         {
            while ((res =::sem_wait(_sem)) == -1 &&  errno == EINTR)
            {
               continue;
            }
            if (res != 0)
            {
               ETG_TRACE_ERR_THR(("hmibase::util::SemaPosix, sem_wait failed! errno %d", errno));
            }
         }
         return (res == 0);
      }

      bool wait(unsigned long timeout)
      {
         int res = -1;
         if (_sem)
         {
            struct timespec ts, ts2;
            if (clock_gettime(CLOCK_REALTIME, &ts) == -1)
            {
               ETG_TRACE_ERR_THR(("hmibase::util::SemaPosix, clock_gettime failed! errno %d", errno));
               return false;
            }
            ts2.tv_sec = ts.tv_sec + timeout / 1000;
            ts2.tv_nsec = ts.tv_nsec + (timeout % 1000) * 1000000U;
            if (ts2.tv_nsec >= 1000000000)
            {
               ts2.tv_nsec -= 1000000000U;
               ++ts2.tv_sec;
            }
            while ((res = ::sem_timedwait(_sem, &ts2)) == -1 && errno == EINTR)
            {
               continue;
            }
            if (res != 0)
            {
               ETG_TRACE_ERR_THR(("hmibase::util::SemaPosix, sem_timedwait failed! errno %d", errno));
            }
         }
         return (res == 0);
      }

      bool waitTry()
      {
         int res = -1;
         if (_sem)
         {
            if ((res = sem_trywait(_sem)) != 0)
            {
               ETG_TRACE_ERR_THR(("hmibase::util::SemaPosix, sem_trywait failed! errno %d", errno));
            }
         }
         return (res == 0);
      }

      bool post()    // increment, release one's share of the semaphore's count
      {
         int res = -1;
         if (_sem)
         {
            if ((res = sem_post(_sem)) != 0)
            {
               ETG_TRACE_ERR_THR(("hmibase::util::SemaPosix, sem_post failed! errno %d", errno));
            }
         }
         return (res == 0);
      }

   private:
      sem_t        _mSem;
      sem_t*       _sem;
      bool         _isCreator;
      std::string  _name;
};


#else


class SemaWindows : public ::hmibase::util::internal::SemaphoreImpl
{
   public:
      SemaWindows() : _sem(0)
      {
      }

      virtual ~SemaWindows()
      {
         destroy();
         _sem = 0;
      }

      bool create(unsigned int nCount, const char* name)
      {
         destroy();
         _name = name;
         _sem = ::CreateSemaphore(0, nCount, 0xFFFF, _name.c_str());
         if (_sem == 0)
         {
            ETG_TRACE_ERR_THR(("hmibase::util::SemaWindows, create failed!"));
         }
         return _sem != 0;
      }

      void destroy()
      {
         if (_sem != 0)
         {
            ::CloseHandle(_sem);
            _sem = 0;
         }
      }

      bool wait()   // reservation, decrement
      {
         return ::WaitForSingleObject(_sem, INFINITE) == static_cast<DWORD>(WAIT_OBJECT_0);
      }

      bool wait(unsigned long timeout) // reservation, decrement
      {
         return ::WaitForSingleObject(_sem, timeout) == static_cast<DWORD>(WAIT_OBJECT_0);
      }

      bool waitTry() // try reservation
      {
         ETG_TRACE_ERR_THR(("hmibase::util::SemaWindows, waitTry not supported, failed!"));
         return false;
      }

      bool post()    // increment, release one's share of the semaphore's count
      {
         bool rc = (FALSE != ::ReleaseSemaphore(_sem, 1, NULL));
         if (rc == false)
         {
            ETG_TRACE_ERR_THR(("hmibase::util::SemaWindows, post failed!"));
         }
         return rc;
      }

   private:
      std::string  _name;
      HANDLE       _sem;
};


#endif // WIN32
} // namespace internal


/////////////////////////////////////////////////////////////////////////////////////
///// Semaphore Interface  //////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////

Semaphore::Semaphore() : _impl(0)
{
   setImplementation();
}


Semaphore::Semaphore(unsigned int nCount, const char* name) : _impl(0)
{
   setImplementation();
   create(nCount, name);
}


Semaphore::~Semaphore()
{
   if (_impl)
   {
      destroy();
      delete _impl;
   }
   _impl = 0;
}


bool Semaphore::create(unsigned int nCount, const char* name)
{
   return _impl ? _impl->create(nCount, name) : false;
}


void Semaphore::destroy()
{
   if (_impl)
   {
      _impl->destroy();
   }
}


bool Semaphore::wait()
{
   // obtain
   return _impl ? _impl->wait() : false;
}


bool Semaphore::wait(unsigned long timeout)
{
   return _impl ? _impl->wait(timeout) : false;
}


bool Semaphore::waitTry()
{
   return _impl ? _impl->waitTry() : false;
}


bool Semaphore::post()
{
   // release
   return _impl ? _impl->post() : false;
}


void Semaphore::purge(const char* name)
{
#ifndef WIN32
   sem_unlink(name);
#endif
}


void Semaphore::setImplementation()
{
#ifndef WIN32
   _impl = new ::hmibase::util::internal::SemaPosix();
#else
   _impl = new ::hmibase::util::internal::SemaWindows();
#endif
   HMI_APP_ASSERT(_impl);
}


}
}
