/******************************************************************
*COPYRIGHT: (C) 2017 Robert Bosch GmbH
*The reproduction, distribution and utilization of this file as
*well as the communication of its contents to others without express
*authorization is prohibited. Offenders will be held liable for the
*payment of damages. All rights reserved in the event of the grant
*of a patent, utility model or design.
******************************************************************/
#include "hmibase/util/Path.h"
#include <algorithm>

#include "hmibase/util/Trace.h"
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW_UTIL
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/Path.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

namespace hmibase {
namespace util {

Path::Path()
{
}


Path::Path(const Path& path)
{
   *this = path;
}


Path::Path(const SimpleString& str)
{
   const char* cStr = str.cPtr();

   SimpleString element;
   for (size_t i = 0; i < str.length(); i++)
   {
      char c = cStr[i];
      if (c == '/' || c == '\\')
      {
         mElements.push_back(element);
         element.clear();
      }
      else
      {
         char cc[2];
         cc[0] = c;
         cc[1] = 0;
         element += cc;
      }
   }

   if (!element.isEmpty() || !mElements.empty())
   {
      mElements.push_back(element);
   }
}


Path::~Path()
{
}


void Path::clear()
{
   mElements.clear();
}


void Path::normalize()
{
   std::vector<SimpleString> elements = mElements;
   mElements.clear();
   for (size_t i = 0; i < elements.size(); i++)
   {
      appendNormalized(elements[i]);
   }
}


void Path::appendNormalized(const SimpleString& element)
{
   SimpleString trimmed = element;
   trimmed.trim();

   if (mElements.empty())
   {
      mElements.push_back(trimmed);
   }
   else if (trimmed == "")
   {
      // Skip
   }
   else if (trimmed == ".")
   {
      // Skip
   }
   else if (trimmed == ".." && mElements.back() == "")
   {
      mElements.pop_back();
      mElements.push_back(trimmed);
   }
   else if (trimmed == ".." && mElements.back() == ".")
   {
      mElements.pop_back();
      mElements.push_back(trimmed);
   }
   else if (trimmed == ".." && mElements.back() == "..")
   {
      mElements.push_back(trimmed);
   }
   else if (trimmed == "..")
   {
      mElements.pop_back();
   }
   else
   {
      mElements.push_back(trimmed);
   }
}


bool Path::isEmpty() const
{
   return mElements.empty();
}


SimpleString Path::toString() const
{
#ifdef WIN32
   SimpleString separator("\\");
#else
   SimpleString separator("/");
#endif

   SimpleString path;
   for (size_t i = 0; i < mElements.size(); i++)
   {
      if (i > 0)
      {
         path += separator;
      }
      path += mElements[i];
   }
   return path;
}


SimpleString Path::getName() const
{
   if (mElements.empty())
   {
      return "";
   }
   else
   {
      return mElements.back();
   }
}


Path Path::getParent() const
{
   Path path(*this);
   path.normalize();
   if (!path.mElements.empty())
   {
      path.mElements.pop_back();
   }
   return path;
}


Path& Path::operator=(const Path& right)
{
   mElements = right.mElements;
   return *this;
}


Path& Path::operator+=(const Path& right)
{
   for (size_t i = 0; i < right.mElements.size(); i++)
   {
      mElements.push_back(right.mElements[i]);
   }
   return *this;
}


Path Path::operator+(const Path& right) const
{
   Path path(*this);
   path += right;
   return path;
}


bool Path::operator==(const Path& right) const
{
   int comparison = compare(*this, right);
   return comparison == 0;
}


bool Path::operator!=(const Path& right) const
{
   int comparison = compare(*this, right);
   return comparison != 0;
}


bool Path::operator<(const Path& right) const
{
   int comparison = compare(*this, right);
   return comparison < 0;
}


bool Path::operator>(const Path& right) const
{
   int comparison = compare(*this, right);
   return comparison > 0;
}


bool Path::operator<=(const Path& right) const
{
   int comparison = compare(*this, right);
   return comparison <= 0;
}


bool Path::operator>=(const Path& right) const
{
   int comparison = compare(*this, right);
   return comparison >= 0;
}


int Path::compare(const Path& left, const Path& right)
{
   // It should be safe to convert to int here.
   // If the value is bigger than 1000 or so, things are totally messed up.
   int maxLevel = (int)std::max(left.mElements.size(), right.mElements.size());

   for (int i = 0; i < maxLevel; i++)
   {
      int comparison = compare(left, right, i);
      if (comparison != 0)
      {
         return comparison;
      }
   }

   return 0;
}


int Path::compare(const Path& left, const Path& right, int level)
{
   SimpleString leftStr;
   SimpleString rightStr;

   if (left.mElements.size() > (size_t)level)
   {
      leftStr = left.mElements[level];
   }

   if (right.mElements.size() > (size_t)level)
   {
      rightStr = right.mElements[level];
   }

   if (leftStr < rightStr)
   {
      return -1;
   }

   if (leftStr > rightStr)
   {
      return 1;
   }

   return 0;
}


void Path::print() const
{
   SimpleString message("Path[");
   for (size_t i = 0; i < mElements.size(); i++)
   {
      if (i > 0)
      {
         message += SS(", ");
      }
      message += SS("\"") + mElements[i] + SS("\"");
   }
   message += SS("]");
   ETG_TRACE_USR4(("%s", message.cPtr()));
}


} // namespace
} // namespace
