/******************************************************************
*COPYRIGHT: (C) 2017 Robert Bosch GmbH
*The reproduction, distribution and utilization of this file as
*well as the communication of its contents to others without express
*authorization is prohibited. Offenders will be held liable for the
*payment of damages. All rights reserved in the event of the grant
*of a patent, utility model or design.
******************************************************************/
#include "hmibase/util/FixedSizeMutex.h"
#include "hmibase/util/Error.h"
#include <string.h>
#include <errno.h>

#include "hmibase/util/Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW_UTIL
#include "trcGenProj/Header/FixedSizeMutex.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

namespace hmibase {
namespace util {

FixedSizeMutex::FixedSizeMutex()
{
   // Create Struct for FixedSizeMutex Attributes
   pthread_mutexattr_t mutexAttributes;
   pthread_mutexattr_init(&mutexAttributes);
   pthread_mutexattr_settype(&mutexAttributes, PTHREAD_MUTEX_ERRORCHECK);
   pthread_mutexattr_setpshared(&mutexAttributes, PTHREAD_PROCESS_SHARED);

   // Initialize Struct for FixedSizeMutex
   Error::reset();
   int errorCode = pthread_mutex_init(&mMutexStruct, &mutexAttributes);
   if (errorCode != 0 || Error::is())
   {
      ETG_TRACE_ERR_THR(("Cannot initialize FixedSizeMutex. %s", getErrorMessage(errorCode).cPtr()));
   }

   // Destroy FixedSizeMutex Attributes Struct
   pthread_mutexattr_destroy(&mutexAttributes);
}


FixedSizeMutex::~FixedSizeMutex()
{
   Error::reset();
   int errorCode = pthread_mutex_destroy(&mMutexStruct);
   if (errorCode != 0 || Error::is())
   {
      ETG_TRACE_ERR_THR(("Cannot destroy FixedSizeMutex. %s", getErrorMessage(errorCode).cPtr()));
   }
}


void FixedSizeMutex::lock()
{
   Error::reset();
   int errorCode = pthread_mutex_lock(&mMutexStruct);
   if (errorCode != 0 || Error::is())
   {
      ETG_TRACE_ERR_THR(("Cannot lock FixedSizeMutex. %s", getErrorMessage(errorCode).cPtr()));
   }
   else
   {
//      ETG_TRACE_USR4_THR(("FixedSizeMutex Locked"));
   }
}


bool FixedSizeMutex::tryLock()
{
   Error::reset();
   int errorCode = pthread_mutex_trylock(&mMutexStruct);

   if (errorCode == EBUSY)
   {
      ETG_TRACE_USR4_THR(("Cannot lock FixedSizeMutex, because it is locked"));
      return false;
   }
   else if (errorCode != 0 || Error::is())
   {
      ETG_TRACE_ERR_THR(("Cannot lock FixedSizeMutex. %s", getErrorMessage(errorCode).cPtr()));
      return false;
   }
   else
   {
      ETG_TRACE_USR4_THR(("FixedSizeMutex Locked"));
      return true;
   }
}


bool FixedSizeMutex::tryLock(int timeoutMilliseconds)
{
   // Fill timespec struct with seconds and nanoseconds
   // 1 second      = 1000 milliseconds
   // 1 millisecond = 1000 nanoseconds
   timespec waitTime;
   waitTime.tv_sec  = (timeoutMilliseconds / 1000);
   waitTime.tv_nsec = (timeoutMilliseconds % 1000) * 1000;

   Error::reset();
   int errorCode = pthread_mutex_timedlock(&mMutexStruct, &waitTime);

   if (errorCode == ETIMEDOUT)
   {
      ETG_TRACE_USR4_THR(("Cannot lock FixedSizeMutex, because it timed out after %d milliseconds", timeoutMilliseconds));
      return false;
   }
   else if (errorCode != 0 || Error::is())
   {
      ETG_TRACE_ERR_THR(("Cannot lock FixedSizeMutex. %s", getErrorMessage(errorCode).cPtr()));
      return false;
   }
   else
   {
      ETG_TRACE_USR4_THR(("FixedSizeMutex Locked"));
      return true;
   }
}


void FixedSizeMutex::unlock()
{
   Error::reset();
   int errorCode = pthread_mutex_unlock(&mMutexStruct);
   if (errorCode != 0 || Error::is())
   {
      ETG_TRACE_ERR_THR(("Cannot unlock FixedSizeMutex. %s", getErrorMessage(errorCode).cPtr()));
   }
   else
   {
//      ETG_TRACE_USR4_THR(("FixedSizeMutex Unlocked"));
   }
}


SimpleString FixedSizeMutex::getErrorMessage(int errorCode) const
{
   if (errorCode == 0)
   {
      return Error::getMessage();
   }
   else
   {
      return Error::getMessage(errorCode);
   }
}


} // namespace
} // namespace
