/******************************************************************
*COPYRIGHT: (C) 2017 Robert Bosch GmbH
*The reproduction, distribution and utilization of this file as
*well as the communication of its contents to others without express
*authorization is prohibited. Offenders will be held liable for the
*payment of damages. All rights reserved in the event of the grant
*of a patent, utility model or design.
******************************************************************/
#include "hmibase/util/DynamicLibrary.h"
#include "hmibase/util/Error.h"

#ifdef WIN32
#include <windows.h>
#else
#include <dlfcn.h>
#endif

#include "hmibase/util/Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW_UTIL
#include "trcGenProj/Header/DynamicLibrary.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

namespace hmibase {
namespace util {

DynamicLibrary::DynamicLibrary(const SimpleString& filepath)
   : mFilePath(filepath)
   , mLibraryHandle(NULL)
{
   openLibrary();
}


DynamicLibrary::~DynamicLibrary()
{
   if (mLibraryHandle != NULL)
   {
      closeLibrary();
      mLibraryHandle = NULL;
   }
}


bool DynamicLibrary::isLoaded() const
{
   if (mLibraryHandle == NULL)
   {
      return false;
   }
   else
   {
      return true;
   }
}


void DynamicLibrary::openLibrary()
{
#ifdef WIN32
   mLibraryHandle = LoadLibrary(mFilePath.cPtr());
#else
   dlerror();
   mLibraryHandle = dlopen(mFilePath.cPtr(), RTLD_LAZY);
   if (mLibraryHandle == NULL)
   {
      ETG_TRACE_ERR_THR(("%s", getErrorMessage(SS("Cannot load shared object: ") + mFilePath).cPtr()));
   }
#endif
}


void DynamicLibrary::closeLibrary()
{
#ifdef WIN32
   FreeLibrary(mLibraryHandle);
#else
   dlclose(mLibraryHandle);
#endif
}


SimpleString DynamicLibrary::getErrorMessage(const SimpleString& message)
{
#ifdef WIN32
   // TODO
#else
   SimpleString errorMessage(dlerror());
   if (!errorMessage.isEmpty())
   {
      return message + SS(" (") + errorMessage + SS(")");
   }
#endif
   return message;
}


void* DynamicLibrary::getFunctionPtr(const SimpleString& name)
{
   if (mLibraryHandle == NULL)
   {
      return NULL;
   }

#ifdef WIN32
   return GetProcAddress(mLibraryHandle, name.cPtr());
#else
   dlerror();
   void* symbolPtr = dlsym(mLibraryHandle, name.cPtr());
   if (symbolPtr == NULL)
   {
      ETG_TRACE_ERR_THR(("%s", getErrorMessage(SS("Cannot load symbol \"") + name + ("\" from shared object: ") + mFilePath).cPtr()));
   }
   return symbolPtr;
#endif
}


} // namespace
} // namespace
