/* ***************************************************************************************
* FILE:          Condition.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  Condition.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "hmibase/util/Condition.h"
#include "hmibase/util/Mutex.h"

#ifdef WIN32
#include <windows.h>
#include <WinBase.h>
#else
#include <sys/time.h>
#endif
#include <errno.h>

#include "hmibase/util/Trace.h"
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW_UTIL
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/Condition.cpp.trc.h"
#endif


namespace hmibase {
namespace util {


/////////////////////////////////////////////////////////////////////////////////////
///// Condition Implementation///////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////

#define MAX_EVENTS 10

struct EventRefEntry
{
   EventRefEntry() : _numRefs(0)
   {
   }
   void set()
   {
      _numRefs++;
   }
   void clear()
   {
      if (_numRefs > 0)
      {
         _numRefs--;
      }
   }
   bool get() const
   {
      return _numRefs > 0;
   }
   unsigned long _numRefs;
};


namespace internal {

#ifndef WIN32
class ConditionPosix : public ::hmibase::util::internal::ConditionImpl
{
   public:
      ConditionPosix() : _evCounter(0)
      {
         pthread_mutexattr_t mutexAttr;
         pthread_mutexattr_init(&mutexAttr);
         pthread_mutex_init(&_mutex, &mutexAttr);
         pthread_condattr_t condAttr;
         pthread_condattr_init(&condAttr);
         pthread_condattr_setclock(&condAttr, CLOCK_MONOTONIC);
         pthread_cond_init(&_cond, &condAttr);
         addEvent();
      }
      virtual ~ConditionPosix()
      {
         pthread_mutex_destroy(&_mutex);
         pthread_cond_destroy(&_cond);
      }
      bool signal(EventRef evRef)
      {
         pthread_mutex_lock(&_mutex);
         if (evRef <= 0 || evRef > _evCounter)
         {
            ETG_TRACE_ERR_THR(("hmibase::util::Condition, invalid condition/event failed! rc=%d", evRef));
            pthread_mutex_unlock(&_mutex);
            return false;
         }
         _evRefs[--evRef].set();
         // pthread_cond_broadcast((pthread_cond_t*)&_cond);
         pthread_cond_signal(&_cond);
         pthread_mutex_unlock(&_mutex);
         return true;
      }
      EventRef wait(unsigned long timeout_msec)
      {
         pthread_mutex_lock(&_mutex);
         EventRef evRefResult = -1;
         bool signaled = false;
         for (int i = 0; i < MAX_EVENTS; i++)
         {
            if (_evRefs[i].get())
            {
               signaled = true;
               break;
            }
         }

         if (!signaled)
         {
            if (timeout_msec == 0xFFFFFFFF)
            {
               pthread_cond_wait(&_cond, &_mutex);
            }
            else
            {
               struct timespec twait;

               // get current time
               clock_gettime(CLOCK_MONOTONIC, &twait);

               // update twait with wait duration
               twait.tv_sec += timeout_msec / 1000;
               twait.tv_nsec += ((timeout_msec % 1000) * 1000000);
               if (twait.tv_nsec >= 1000000000)
               {
                  twait.tv_sec += twait.tv_nsec / 1000000000;
                  twait.tv_nsec = twait.tv_nsec % 1000000000;
               }

               //condition wait will get releases when timeout occurs or if a message arrives in queue
               int rc = pthread_cond_timedwait(&_cond, &_mutex, &twait);

               if (rc == ETIMEDOUT)
               {
                  evRefResult = 0; // timeout
               }
               else if (rc == 0)
               {
                  //signaled
               }
               else
               {
                  ETG_TRACE_ERR_THR(("hmibase::util::Condition, pthread_cond_timedwait() returned rc=%d)", rc));
               }
            }
         }
         for (int i = 0; i < MAX_EVENTS; i++)
         {
            if (_evRefs[i].get())
            {
               _evRefs[i].clear();
               evRefResult = i + 1;
               break;
            }
         }
         if (evRefResult == -1)
         {
            ETG_TRACE_USR1_THR(("hmibase::util::Condition, waitevent no matching event!"));
         }
         pthread_mutex_unlock(&_mutex);
         return evRefResult;
      }

      EventRef addEvent()
      {
         pthread_mutex_lock(&_mutex);
         if (_evCounter < MAX_EVENTS)
         {
            _evRefs[_evCounter++].clear();
            pthread_mutex_unlock(&_mutex);
            return _evCounter;
         }
         pthread_mutex_unlock(&_mutex);
         ETG_TRACE_ERR_THR(("hmibase::util::Condition, create event failed!"));
         return -1;
      }
   private:
      pthread_cond_t           _cond;
      mutable pthread_mutex_t  _mutex;
      EventRefEntry  _evRefs[MAX_EVENTS];
      //std::vector<EventRefEntry> _evRefTab;
      EventRef       _evCounter;
};


#else

class ConditionWindows : public ::hmibase::util::internal::ConditionImpl
{
   public:
      ConditionWindows() :  _evCounter(0)
      {
         _event[0] = CreateEvent(NULL, FALSE, FALSE, NULL);
         if (_event[0] == 0)
         {
            ETG_TRACE_ERR_THR(("hmibase::util::Condition, create event failed!"));
         }
         addEvent();
      }

      virtual ~ConditionWindows()
      {
         if (_event[0])
         {
            CloseHandle(_event[0]);
            _event[0] = 0;
         }
      }

      bool signal(EventRef evRef)
      {
         ScopedMutex cs(_mutex);
         if (evRef <= 0 || evRef > _evCounter)
         {
            ETG_TRACE_ERR_THR(("hmibase::util::Condition, invalid condition/event failed! rc=%d", evRef));
            return false;
         }
         _evRefs[--evRef].set();
         if (!SetEvent(_event[0]))
         {
            ETG_TRACE_ERR_THR(("hmibase::util::Condition, setvent failed! rc=%d", GetLastError()));
            return false;
         }
         return true;
      }

      EventRef wait(unsigned long timeout_msec)
      {
         _mutex.lock();
         EventRef evRefResult = -1;

         bool signaled = false;
         for (int i = 0; i < MAX_EVENTS; i++)
         {
            if (_evRefs[i].get())
            {
               signaled = true;
               break;
            }
         }

         if (!signaled)
         {
            _mutex.unlock();
            const bool waitForAllEvents = false;
            DWORD dwEvent = WaitForMultipleObjects(1, _event, waitForAllEvents, timeout_msec);
            _mutex.lock();
            if (dwEvent == WAIT_TIMEOUT)
            {
               evRefResult = 0;
            }
            else if (dwEvent >= WAIT_OBJECT_0)
            {
               // signaled
            }
            else
            {
               // error
            }
         }

         for (int i = 0; i < MAX_EVENTS; i++)
         {
            if (_evRefs[i].get())
            {
               _evRefs[i].clear();
               evRefResult = i + 1;
               break;
            }
         }
         if (evRefResult == -1)
         {
            ETG_TRACE_USR1_THR(("hmibase::util::Condition, waitevent no matching event!"));
         }
         _mutex.unlock();
         return evRefResult;
      }

      EventRef addEvent()
      {
         ScopedMutex cs(_mutex);
         if (_evCounter < MAX_EVENTS)
         {
            _evRefs[_evCounter++].clear();
            return _evCounter;
         }

         ETG_TRACE_ERR_THR(("hmibase::util::Condition, create event failed!"));
         return -1;
      }

   private:
      HANDLE         _event[1];
      EventRefEntry  _evRefs[MAX_EVENTS];
      //std::vector<EventRefEntry> _evRefTab;
      EventRef       _evCounter;
      Mutex          _mutex;
};


#endif // WIN32
} // namespace internal


/////////////////////////////////////////////////////////////////////////////////////
///// Condition Interface  //////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////

Condition::Condition() : _impl(0)
{
#ifndef WIN32
   _impl = new ::hmibase::util::internal::ConditionPosix();
#else
   _impl = new ::hmibase::util::internal::ConditionWindows();
#endif
}


Condition::~Condition()
{
   if (_impl)
   {
      delete _impl;
   }
   _impl = 0;
}


bool Condition::signal(EventRef evRef) const
{
   return _impl ? _impl->signal(evRef) : false;
}


EventRef Condition::wait(unsigned long timeout_msec) const
{
   return _impl ? _impl->wait(timeout_msec) : -1;
}


EventRef Condition::addEvent() const
{
   return _impl ? _impl->addEvent() : -1;
}


}  // namespace util
}  // namespace hmibase
