/* ***************************************************************************************
* FILE:          WaylandBackend.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  WaylandBackend.h is part of HMI-Base testimagedaemon
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#ifndef HMIBASEWAYLANDBACKEND_H
#define HMIBASEWAYLANDBACKEND_H

#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string>
#include <unistd.h>
#include <math.h>


#include "EGL/egl.h"
#include <wayland-egl.h>
#include <ilm/ilm_types.h>
#include "compositor-shim.h"

namespace hmibase {
class WaylandBackend
{
   public:
      WaylandBackend(unsigned int screenId, unsigned int layerId, unsigned int surfaceId, unsigned int width, unsigned int height);
      virtual ~WaylandBackend();

      enum tenErrorCode
      {
         NO_ERROR,
         EGL_ERROR,
         WL_ERROR,
         COMPOSITOR_ERROR,
         ILM_ERROR,  //
         IL_ERROR,    // image loader error
         NO_IMAGE
      };

      enum tenImageStatus
      {
         IMAGE_OK,
         IMAGE_NOT_FULLSCREEN,
         IMAGE_FORMAT_ERROR,
         IMAGE_FILE_ERROR,
         IMAGE_NOT_FOUND,
         IMAGE_UNKNOWN,
         UNKNOWN_ERROR
      };

      struct _WaylandGles
      {
         struct wl_display*    wl_display;
         struct wl_surface*    wl_surface;
         struct wl_registry*   wl_registry;
         struct wl_egl_window* wl_native_window;
         struct wl_compositor* wl_compositor;
         unsigned int          surface_id;
         unsigned int          layer_id;

         struct compositor_shim_context*        wlAdapterContext;
         struct compositor_shim_surface_context wlAdapterSurfaceContext;

         _WaylandGles() : wl_display(0), wl_surface(0), wl_registry(0), wl_native_window(0), wl_compositor(0), surface_id(0), layer_id(0), wlAdapterContext(0)
         {
         }
      } ;
      typedef struct _WaylandGles WaylandGles;

      bool init();
      void deinit();
      bool render();

      void setVisibility(bool activate);
      void setCurrentImage(const std::string& imagePath)
      {
         _image = imagePath;
      }

      unsigned int getScreenId() const
      {
         return _screenId;
      }

      unsigned int getLayerId() const
      {
         return _layerId;
      }

      unsigned int getSurfaceId() const
      {
         return _surfaceId;
      }

      tenErrorCode getLastError() const
      {
         return _errorCode;
      }

      tenImageStatus getImageStatus() const
      {
         return _imageStatus;
      }

   private:
      void swap_buffer();
      bool egl_open();
      void egl_close();
      bool makeCurrent();
      void terminate_renderer(void);
      void handle_egl_error(const char*);
      void handle_ilm_error(ilmErrorTypes err, const char*);
      bool init_shader();
      void init_renderer(uint8_t* buffer, int width, int height);
      EGLBoolean create_wl_context(WaylandGles* sink);

      typedef struct egl_setup_t
      {
         EGLContext egl_context;
         EGLSurface egl_surface;
         EGLDisplay egl_display;
         WaylandGles wland_info;
         EGLConfig    eglConfig;

         egl_setup_t() : egl_context(0), egl_surface(0), egl_display(0), eglConfig(0)
         {
         }
      } egl_setup;

      egl_setup egl_object;
      egl_setup egl_object_shared;

      WaylandGles aWayland;

      unsigned int _screenWidth;
      unsigned int _screenHeight;

      unsigned int _surfaceId;
      unsigned int _layerId;
      unsigned int _screenId;

      std::string _image;

      tenErrorCode _errorCode;
      tenImageStatus _imageStatus;
};


}

#endif
