/* ***************************************************************************************
* FILE:          TestImageRenderer.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TestImageRenderer.cpp is part of HMI-Base testimagedaemon
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "TestImageRenderer.h"
#include "ScreenBrokerSettings/Settings.h"
#include "WaylandBackend.h"
#include "ScreenBroker/Client/Client.h"
#include "ScreenBroker/Client/ClientUtil.h"

#include <ilm/ilm_client.h>
#include <ilm/ilm_control.h>

#include <systemd/sd-daemon.h>

#include "TestImageDaemon_Trace.h"

#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_TESTIMAGEDAEMON
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/TestImageRenderer.cpp.trc.h"
#endif

using namespace ::bosch::cm::ai::hmi::TestImages;
using namespace ::bosch::cm::ai::hmi::screenbroker;

namespace hmibase {

TestImageRenderer* _theInstance = 0;

TestImageRenderer::TestImageRenderer() :
   bosch::cm::ai::hmi::TestImages::TestImagesStub("port") ,
   _isInitialized(false)
{
   vInitPlatformEtg();

   ETG_I_REGISTER_CHN(TraceCmd_NotProcessedMsg);
   ETG_I_REGISTER_FILE();

   _theInstance = this;
   //WaylandBackend wlBackend;
   //wlBackend.wayland_init();

   ScreenBroker::Client::GetInstance().InitClientApi(this, "TestImageDaemon", this);
}


TestImageRenderer::~TestImageRenderer()
{
   ETG_I_UNREGISTER_FILE();
   ETG_I_UNREGISTER_CHN();
   WaylandBackend* wlBackend = 0;

   while (_backends.size() > 0)
   {
      wlBackend = _backends.begin()->second;

      if (wlBackend)
      {
         ScreenBroker::RequestArg lRequestArg(ScreenBroker::ClientUtil::GenerateRequestId(),
                                              static_cast<ScreenBroker::UInt32>(-1),
                                              static_cast<ScreenBroker::UInt32>(0));
         DeregisterSurface(lRequestArg, wlBackend->getSurfaceId());

         wlBackend->deinit();
         delete wlBackend;
         wlBackend = 0;
      }
      _backends.erase(_backends.begin());
   }
}


void TestImageRenderer::OnServiceStatusChanged(bool available)
{
   if (available)
   {
      std::vector<ScreenBroker::BuildInLayerInfo> info;
      if (GetBuildInLayerInfo(info))
      {
         OnBuildInLayerInfo(info);
      }
   }
}


void TestImageRenderer::OnBuildInLayerInfo(const std::vector<ScreenBroker::BuildInLayerInfo>& /*info*/)
{
   init();

   sd_notifyf(0, "READY=1\n"  // Tells the init system that daemon startup is finished.
              // This is only used by systemd if the service definition file has Type=notify set.
              "STATUS=Processing requests...\n" // Passes a single-line status string back to the init system that describes the daemon state.
              // This is free-form and can be used for various purposes
              "MAINPID=%lu\nPARENTID=%lu",
              (unsigned long) getpid(),
              (unsigned long) getppid());
}


bool TestImageRenderer::init()
{
   if (_isInitialized)
   {
      return true;
   }

   ilmErrorTypes err = ilm_init();

   if (err != ILM_SUCCESS)
   {
      ETG_TRACE_ERR(("ilm_init failed, error code %u", err));
      return false;
   }

   std::vector< ::ScreenBroker::BuildInLayerInfo > buildInLayers ;
   if (GetBuildInLayerInfo(buildInLayers))
   {
      for (std::vector<ScreenBroker::BuildInLayerInfo>::iterator it = buildInLayers.begin(); it != buildInLayers.end(); ++it)
      {
         std::map<uint32_t, WaylandBackend*>::iterator iter = _backends.find(it->ScreenId());
         if (iter != _backends.end())
         {
            ETG_TRACE_USR2(("WaylandBackend for display %d already available", it->ScreenId()));
            continue;
         }

         // create one backend per display
         unsigned int w = it->Dimension().Width();
         unsigned int h = it->Dimension().Height();
         WaylandBackend* wlBackend = new WaylandBackend(
            it->ScreenId(),
            it->LayerId(),
            it->SurfaceId(),
            w,
            h);

         if (wlBackend)
         {
            if (wlBackend->init())
            {
               _backends[it->ScreenId()] = wlBackend;
               ScreenBroker::RequestArg lRequestArg(ScreenBroker::ClientUtil::GenerateRequestId(),
                                                    static_cast<ScreenBroker::UInt32>(-1),
                                                    static_cast<ScreenBroker::UInt32>(0));
               RegisterSurface(lRequestArg, it->LayerId(), it->SurfaceId());
            }
            else
            {
               ETG_TRACE_ERR(("wlBackend init failed"));

               delete wlBackend;
               wlBackend = 0;

               return false;
            }
         }
         else
         {
            return false;
         }
      }
   }
   else
   {
      // no info about build in layer yet, so don't complete initialization
      return false;
   }

   _isInitialized = true;

   return true;
}


void TestImageRenderer::onShowTestImageRequest(const ::boost::shared_ptr< ::bosch::cm::ai::hmi::TestImages::ShowTestImageRequest >& request)
{
   uint32_t displayId = request->getDisplayId();
   WaylandBackend* backend = getBackend(displayId);
   tenErrorCode errorCode = tenErrorCode__NO_ERROR;

   if (backend != 0)
   {
      ETG_TRACE_USR4(("Start rendering, image %s", request->getImagePath().c_str()));
      backend->setCurrentImage(request->getImagePath());
      backend->render();

      switch (backend->getImageStatus())
      {
         case WaylandBackend::IMAGE_OK:
         {
            errorCode = tenErrorCode__NO_ERROR;
         }
         break;
         case WaylandBackend::IMAGE_FORMAT_ERROR:
         {
            errorCode = tenErrorCode__IMAGE_FORMAT_NOT_SUPPORTED;
         }
         break;
         case WaylandBackend::IMAGE_NOT_FULLSCREEN:
         {
            errorCode = tenErrorCode__IMAGE_NOT_FULLSCSREEN;
         }
         break;
         case WaylandBackend::IMAGE_NOT_FOUND:
         {
            errorCode = tenErrorCode__IMAGE_NOT_FOUND;
         }
         break;
         case WaylandBackend::IMAGE_FILE_ERROR:
         {
            errorCode = tenErrorCode__IMAGE_CORRUPTED;
         }
         break;
         default:
         {
            errorCode = tenErrorCode__UNKNOWN_ERROR;
         }
      }
   }
   else
   {
      errorCode = tenErrorCode__UNKNOWN_DISPLAY;
   }
   ETG_TRACE_ERR(("ErrorCode %u", errorCode));

   ::bosch::cm::ai::hmi::TestImages::TestImageStatus status(errorCode);
   sendShowTestImageResponse(status);
}


void TestImageRenderer::onActivateTestImageProcessingRequest(const ::boost::shared_ptr< ::bosch::cm::ai::hmi::TestImages::ActivateTestImageProcessingRequest >& request)
{
   for (std::map<uint32_t, WaylandBackend*>::iterator it = _backends.begin(); it != _backends.end(); ++it)
   {
      if (request->getActivate())
      {
         ScreenBrokerProtocol::RequestId<ScreenBrokerProtocol::ShowSurfaceMode::Enum> lRequestId;
         lRequestId.Set(ScreenBrokerProtocol::ShowSurfaceMode::Enum(0));
         ScreenBroker::RequestArg lRequestArg(lRequestId.Get(),
                                              static_cast<unsigned int>(getpid()),
                                              0);
         ActivateApplication(lRequestArg, (*it).second->getSurfaceId());
         (*it).second->render();
      }
      else
      {
         ScreenBrokerProtocol::RequestId<ScreenBrokerProtocol::HideSurfaceMode::Enum> lRequestId;
         lRequestId.Clear(ScreenBrokerProtocol::HideSurfaceMode::Enum(0));
         ScreenBroker::RequestArg lRequestArg(lRequestId.Get(),
                                              static_cast<unsigned int>(getpid()),
                                              0);
         ActivateApplication(lRequestArg, (*it).second->getSurfaceId());
      }
   }
}


WaylandBackend* TestImageRenderer::getBackend(uint32_t displayId)
{
   std::map<uint32_t, WaylandBackend*>::iterator it = _backends.find(displayId);

   if (it != _backends.end())
   {
      return it->second;
   }

   if (_backends.size() == 1)
   {
      ETG_TRACE_USR1(("Use default display"));
      return _backends.begin()->second;
   }

   ETG_TRACE_ERR(("No matching display for id %d, total amount of displays %d", displayId, _backends.size()));
   return NULL;
}


}
