/******************************************************************
*COPYRIGHT: (C) 2017 Robert Bosch GmbH
*The reproduction, distribution and utilization of this file as
*well as the communication of its contents to others without express
*authorization is prohibited. Offenders will be held liable for the
*payment of damages. All rights reserved in the event of the grant
*of a patent, utility model or design.
******************************************************************/
#include "hmibase/gadget/syncblock2/socks/Socket.h"
#include "hmibase/gadget/syncblock2/socks/Address.h"
#include "hmibase/util/Error.h"
#include <sys/socket.h>
#include <sys/un.h>
#include <unistd.h>
#include <errno.h>

#include "hmibase/util/Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW
#include "trcGenProj/Header/Socket.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

using namespace hmibase::util;

namespace hmibase {
namespace gadget {
namespace syncblock2 {
namespace socks {

Socket::Socket(int socketFd)
   : mSocketFd(socketFd)
{
}


Socket::~Socket()
{
}


void Socket::close()
{
   if (mSocketFd != -1)
   {
      ::close(mSocketFd);
      mSocketFd = -1;
   }
}


bool Socket::send(Message& message)
{
   Error::reset();
   ssize_t numBytes = sendmsg(mSocketFd, message.getMsgPtr(), MSG_DONTWAIT);
   if (numBytes <= 0 || Error::is())
   {
      return false;
   }
   return true;
}


bool Socket::receive(Message& message)
{
   Error::reset();
   ssize_t numBytes = recvmsg(mSocketFd, message.getMsgPtr(), MSG_DONTWAIT);
   if (numBytes <= 0 || Error::is())
   {
      return false;
   }
   return true;
}


void Socket::create()
{
   Error::reset();
   mSocketFd = socket(AF_UNIX, SOCK_NONBLOCK | SOCK_SEQPACKET, 0);
   if (Error::is() || mSocketFd == -1)
   {
      ETG_TRACE_ERR_THR(("Cannot create Connecting-Socket, resetting connection. %s", Error::getMessage().cPtr()));
      close();
   }

   ETG_TRACE_USR4_THR(("Created Connecting-Socket."));
}


bool Socket::tryConnect(int key)
{
   if (mSocketFd == -1)
   {
      create();
   }

   if (mSocketFd == -1)
   {
      return false;
   }

   sockaddr* addressPtr = Address::getAddress(key);

   Error::reset();
   int result = connect(mSocketFd, addressPtr, sizeof(sockaddr_un));
   delete addressPtr;

   if (Error::is() || result != 0)
   {
      switch (Error::get())
      {
         case EISCONN:
            return true;

         case ENOENT:
            //ETG_TRACE_ERR_THR(("Cannot Connect socket because no listening socket was found, resetting connection. %s", Error::getMessage().cPtr()));
            return false;

         default:
            ETG_TRACE_ERR_THR(("Cannot Connect socket, resetting connection. %s", Error::getMessage().cPtr()));
            close();
            return false;
      }
   }

   ETG_TRACE_USR4_THR(("Connected Socket."));
   return true;
}


} // namespace
} // namespace
} // namespace
} // namespace
