/******************************************************************
*COPYRIGHT: (C) 2017 Robert Bosch GmbH
*The reproduction, distribution and utilization of this file as
*well as the communication of its contents to others without express
*authorization is prohibited. Offenders will be held liable for the
*payment of damages. All rights reserved in the event of the grant
*of a patent, utility model or design.
******************************************************************/
#include "hmibase/gadget/syncblock2/socks/ListeningSocket.h"
#include "hmibase/gadget/syncblock2/socks/Address.h"
#include "hmibase/util/Error.h"
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/un.h>
#include <unistd.h>
#include <errno.h>

#include "hmibase/util/Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW
#include "trcGenProj/Header/ListeningSocket.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

namespace hmibase {
namespace gadget {
namespace syncblock2 {
namespace socks {


ListeningSocket::ListeningSocket(const int key)
   : mKey(key)
   , mSocketFd(-1)
{
   purge();
   create();
}


ListeningSocket::~ListeningSocket()
{
   destroy();
   purge();
}


bool ListeningSocket::exists() const
{
   return mSocketFd != -1;
}


void ListeningSocket::create()
{
   // Create Listening Socket
   Error::reset();
   mSocketFd = ::socket(AF_UNIX, SOCK_NONBLOCK | SOCK_SEQPACKET, 0);
   if (Error::is() || mSocketFd == -1)
   {
      ETG_TRACE_ERR_THR(("[%s] Cannot create Listening-Socket.", Address::getPath(mKey).cPtr()));
      destroy();
      return;
   }

   // Bind Listening Socket
   Error::reset();
   sockaddr* addressPtr = Address::getAddress(mKey);
   int bindResult = bind(mSocketFd, addressPtr, sizeof(sockaddr_un));
   delete addressPtr;
   if (Error::is() || bindResult != 0)
   {
      ETG_TRACE_ERR_THR(("[%s] Cannot bind Listening-Socket.", Address::getPath(mKey).cPtr()));
      destroy();
      return;
   }

   // Change file permissions
   Error::reset();
   int mode = 0;
   mode |= S_IRUSR; // Read  by Owner
   mode |= S_IWUSR; // Write by Owner
   mode |= S_IRGRP; // Read  by Goup
   mode |= S_IWGRP; // Write by Group
   mode |= S_IROTH; // Read  by Other
   mode |= S_IWOTH; // Write by Other

   if (chmod(Address::getPath(mKey).cPtr(), mode) != 0 || Error::is())
   {
      ETG_TRACE_ERR_THR(("[%s] Cannot Set Permissions.", Address::getPath(mKey).cPtr()));
      destroy();
      return;
   }

   // Start Listening
   Error::reset();
   int listenResult = listen(mSocketFd, 1);
   if (Error::is() || listenResult != 0)
   {
      ETG_TRACE_ERR_THR(("[%s] Cannot start Listening-Socket.", Address::getPath(mKey).cPtr()));
      destroy();
      return;
   }
}


void ListeningSocket::destroy()
{
   if (mSocketFd != -1)
   {
      close(mSocketFd);
   }
}


void ListeningSocket::purge()
{
   unlink(Address::getPath(mKey).cPtr());
}


bool ListeningSocket::tryAccept(Socket& socket)
{
   if (!exists())
   {
      create();
   }

   if (!exists())
   {
      return false;
   }

   Error::reset();
   int socketFd = accept(mSocketFd, NULL, NULL);
   if (Error::is() || socketFd == -1)
   {
      if (Error::get() != EWOULDBLOCK)
      {
         ETG_TRACE_ERR_THR(("[%s] Could not Accept.", Address::getPath(mKey).cPtr()));
         close(socketFd);
      }
      return false;
   }

   ETG_TRACE_USR4_THR(("[%s] Accepted.", Address::getPath(mKey).cPtr()));
   socket = Socket(socketFd);
   return true;
}


} // namespace
} // namespace
} // namespace
} // namespace
