/******************************************************************
*COPYRIGHT: (C) 2017 Robert Bosch GmbH
*The reproduction, distribution and utilization of this file as
*well as the communication of its contents to others without express
*authorization is prohibited. Offenders will be held liable for the
*payment of damages. All rights reserved in the event of the grant
*of a patent, utility model or design.
******************************************************************/
#include "hmibase/gadget/syncblock2/SocketProducerInst.h"
#include "hmibase/gadget/syncblock2/socks/GiveBufferMessage.h"
#include "hmibase/util/Timeout.h"
#include <unistd.h>

#include "hmibase/util/Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW
#include "trcGenProj/Header/SocketProducerInst.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

using namespace hmibase::util;
using namespace hmibase::gadget::syncblock2::socks;

namespace hmibase {
namespace gadget {
namespace syncblock2 {

int SocketProducerInst::smBufferCount = 3;


SocketProducerInst::SocketProducerInst(int key, int instanceId)
   : ProducerInst(key, instanceId, SOCKET_PRODUCER)
   , mMutex()
   , mExchangeMutex()
   , mCurrentBuffer(-1)
   , mTerminate(false)
{
   mMutex.lock();
   for (int i = 0; i < smBufferCount; i++)
   {
      mBuffersEmpty.push_back(i);
   }
   mMutex.unlock();
}


SocketProducerInst::~SocketProducerInst()
{
   startTermination();
   while (!isTerminated())
   {
      usleep(10000); // 10 milliseconds
   }
   mThread.finish();
}


bool SocketProducerInst::onRun(const bool& /*keepRunning*/)
{
   mMutex.lock();
   bool terminate = mTerminate;
   mMutex.unlock();

   if (terminate)
   {
      // Send -1 to Consumer to Signal that we want all buffers back
      if (!sendBufferId(-1))
      {
         ETG_TRACE_ERR_THR(("SocketProducer-Thread[%d][%d]: Failed when sending GoodByeMessage to Consumer", mKey, mInstanceId));
      }

      // Receive BufferIds, until we get a -1 from Consumer as confirmation
      bool consumerIsDone = false;
      Timeout timeout(3000);
      while (!consumerIsDone && !timeout.isReached())
      {
         int bufferId = 0;
         if (receiveBufferId(bufferId) && bufferId == -1)
         {
            consumerIsDone = true;
         }
         usleep(10000); // 10 milliseconds
      }

      if (!consumerIsDone)
      {
         ETG_TRACE_ERR_THR(("SocketProducer-Thread[%d][%d]: Timeout when waiting for Consumer GoodByeMessage", mKey, mInstanceId));
      }

      mMutex.lock();
      moveTo(mBuffersAway, mBuffersTerminated);
      mMutex.unlock();
      return false;
   }
   else
   {
      // Get latest filled buffer
      // All other filled buffers become empty buffers
      mMutex.lock();
      int bufferToSend = popBack(mBuffersFilled);
      moveTo(mBuffersFilled, mBuffersEmpty);
      mMutex.unlock();

      // If there was a buffer, give it to the Consumer
      if (bufferToSend != -1)
      {
         if (sendBufferId(bufferToSend))
         {
            // Buffer is away
            mMutex.lock();
            mBuffersAway.push_back(bufferToSend);
            mMutex.unlock();
         }
         else
         {
            // Buffer is still here
            mMutex.lock();
            mBuffersEmpty.push_back(bufferToSend);
            mMutex.unlock();
         }
      }

      // Now try to receive some Buffers from the Consumer
      int bufferId = 0;
      while (receiveBufferId(bufferId))
      {
         if (bufferId == -1)
         {
            // Consumer has shut down
            startTermination();
            mMutex.lock();
            moveTo(mBuffersAway, mBuffersTerminated);
            mMutex.unlock();
            return false;
         }
         else
         {
            // We received a buffer from Consumer
            mMutex.lock();
            mBuffersAway.remove(bufferId);
            mBuffersEmpty.push_back(bufferId);
            mMutex.unlock();
         }
      }

      usleep(10000); // Sleep 10 Milliseconds
      return true;
   }
}


bool SocketProducerInst::sendBufferId(int bufferId)
{
   GiveBufferMessage giveBufferMessage;
   giveBufferMessage.setBufferIndex(bufferId);
   return mSocket.send(giveBufferMessage);
}


bool SocketProducerInst::receiveBufferId(int& bufferId)
{
   GiveBufferMessage giveBufferMessage;
   if (mSocket.receive(giveBufferMessage))
   {
      bufferId = giveBufferMessage.getBufferIndex();
      return true;
   }
   else
   {
      return false;
   }
}


void SocketProducerInst::startTermination()
{
   mMutex.lock();
   ETG_TRACE_USR4_THR(("SocketProducer-Thread[%d][%d][%d]: BufferSizes: empty=%d, filled=%d, away=%d, terminated=%d", mKey, mInstanceId, mObjectId, mBuffersEmpty.size(), mBuffersFilled.size(), mBuffersAway.size(), mBuffersTerminated.size()));
   mTerminate = true;

   // Move all Buffers, except those that are away into mBuffersTerminated
   if (mCurrentBuffer != -1)
   {
      mBuffersTerminated.push_back(mCurrentBuffer);
      mCurrentBuffer = -1;
   }
   moveTo(mBuffersEmpty , mBuffersTerminated);
   moveTo(mBuffersFilled, mBuffersTerminated);

   mMutex.unlock();
}


bool SocketProducerInst::isTerminated()
{
   mMutex.lock();
   bool isTerminated = mBuffersTerminated.size() == static_cast<size_t>(smBufferCount);
   mMutex.unlock();

   return isTerminated;
}


int SocketProducerInst::exchange()
{
   mIsFirstExchange = false;
   mExchangeMutex.lock();

   // Send last Buffer to Producer
   mMutex.lock();
   if (mCurrentBuffer != -1)
   {
      mBuffersFilled.push_back(mCurrentBuffer);
      mCurrentBuffer = -1;
   }
   mMutex.unlock();

   // Wait until a new Buffer becomes available
   int bufferId = -1;
   Timeout timeout(1000);
   while (bufferId == -1 && !isTerminated() && !timeout.isReached())
   {
      mMutex.lock();

      if (!mBuffersEmpty.empty())
      {
         mCurrentBuffer = mBuffersEmpty.front();
         mBuffersEmpty.pop_front();
      }

      if (mCurrentBuffer != -1)
      {
         bufferId = mCurrentBuffer;
      }

      mMutex.unlock();

      if (bufferId == -1)
      {
         usleep(10000); // Sleep 10 Milliseconds
      }
   }

   mExchangeMutex.unlock();
   if (bufferId < 0 || bufferId > 2)
   {
      return -1;
   }

   return mFds[bufferId];
}


int SocketProducerInst::popFront(std::list<int>& buffers)
{
   int buffer = -1;
   if (!buffers.empty())
   {
      buffer = buffers.front();
      buffers.pop_front();
   }
   return buffer;
}


int SocketProducerInst::popBack(std::list<int>& buffers)
{
   int buffer = -1;
   if (!buffers.empty())
   {
      buffer = buffers.back();
      buffers.pop_back();
   }
   return buffer;
}


void SocketProducerInst::moveTo(std::list<int>& buffers, std::list<int>& targetBuffers)
{
   while (!buffers.empty())
   {
      int buffer = buffers.front();
      buffers.pop_front();
      if (buffer != -1)
      {
         targetBuffers.push_back(buffer);
      }
   }
}


} // namespace
} // namespace
} // namespace
