/******************************************************************
*COPYRIGHT: (C) 2017 Robert Bosch GmbH
*The reproduction, distribution and utilization of this file as
*well as the communication of its contents to others without express
*authorization is prohibited. Offenders will be held liable for the
*payment of damages. All rights reserved in the event of the grant
*of a patent, utility model or design.
******************************************************************/
#include "hmibase/gadget/syncblock2/SocketConsumerInst.h"
#include "hmibase/gadget/syncblock2/socks/GiveBufferMessage.h"
#include "hmibase/util/Timeout.h"
#include <unistd.h>

#include "hmibase/util/Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW
#include "trcGenProj/Header/SocketConsumerInst.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN


using namespace std;
using namespace hmibase::util;
using namespace hmibase::gadget::syncblock2::socks;


namespace hmibase {
namespace gadget {
namespace syncblock2 {

SocketConsumerInst::SocketConsumerInst(int key, int instanceId, int fd0, int fd1, int fd2, const BufferInfo& bufferInfo, socks::Socket* socketPtr)
   : ConsumerInst(key, instanceId, fd0, fd1, fd2, bufferInfo)
   , mBufferId(-1)
   , mIsUpdated(false)
   , mSocketPtr(socketPtr)
   , mMutex()
   , mThread(*this, "SocketConsumer")
{
   if (mSocketPtr != NULL)
   {
      mThread.start();
   }
}


SocketConsumerInst::~SocketConsumerInst()
{
   if (mSocketPtr != NULL)
   {
      mThread.finish();

      delete mSocketPtr;
      mSocketPtr = NULL;
   }
}


void SocketConsumerInst::onStart(const bool& /*keepRunning*/)
{
   // Do nothing
}


bool SocketConsumerInst::onRun(const bool& /*keepRunning*/)
{
   // Receive one new Buffer
   int newBufferId = 0;
   if (receiveBufferId(newBufferId))
   {
      //ETG_TRACE_USR4_THR(("SocketConsumerInst-Thread[%d][%d] Received bufferId=%d", mKey, mInstanceId, newBufferId));
      if (newBufferId == -1)
      {
         // Producer wants us to finish
         // Will send our response in onFinish()
         return false;
      }
      else
      {
         mMutex.lock();
         while (!mBuffersNew.empty())
         {
            int bufferId = mBuffersNew.front();
            mBuffersNew.pop_front();
            mBuffersOld.push_back(bufferId);
         }
         mBuffersNew.push_back(newBufferId);
         mIsUpdated = true;
         mMutex.unlock();
      }
   }

   // Send all old Buffers
   bool doneSending = false;
   while (!doneSending)
   {
      mMutex.lock();
      int bufferId = popFront(mBuffersOld);
      mMutex.unlock();

      if (bufferId == -1)
      {
         doneSending = true;
      }
      else
      {
         //ETG_TRACE_USR4_THR(("SocketConsumerInst-Thread[%d][%d] Sending bufferId=%d", mKey, mInstanceId, bufferId));
         if (!sendBufferId(bufferId))
         {
            ETG_TRACE_ERR_THR(("SocketConsumerInst-Thread[%d][%d][%d]: Sending bufferId=%d failed", mKey, mInstanceId, mObjectId, bufferId));
            mMutex.lock();
            mBuffersOld.push_front(bufferId);
            mMutex.unlock();
            doneSending = true;
         }
      }
   }
   usleep(10000); // Sleep 10 Milliseconds
   return true;
}


void SocketConsumerInst::onFinish(const bool& /*keepRunning*/)
{
   ETG_TRACE_USR4_THR(("SocketConsumerInst-Thread[%d][%d][%d]: Finishing...", mKey, mInstanceId, mObjectId));

   // Reset our buffers
   mMutex.lock();
   mBuffersNew.clear();
   mBuffersOld.clear();
   mBufferId  = -1;
   mIsUpdated = true;
   mMutex.unlock();

   // wait for isUpdated to be cleared
   // this will indicate that out Consumer is not using the current buffer anymore.
   bool isUpdated = true;
   Timeout timeout(3000);
   while (!timeout.isReached() && isUpdated)
   {
      mMutex.lock();
      isUpdated = mIsUpdated;
      mMutex.unlock();
      usleep(10000); // 10 milliseconds
   }

   if (isUpdated)
   {
      ETG_TRACE_ERR_THR(("SocketConsumerInst-Thread[%d][%d][%d]: Timed out when waiting for Consumer to Consume currentBuffer.", mKey, mInstanceId, mObjectId));
   }

   // Send -1
   if (!sendBufferId(-1))
   {
      ETG_TRACE_ERR_THR(("SocketConsumerInst-Thread[%d][%d][%d]: Failed to send GoodByeMessage.", mKey, mInstanceId, mObjectId));
   }
   mSocketPtr->close();

   ETG_TRACE_USR4_THR(("SocketConsumerInst-Thread[%d][%d][%d]: Finished", mKey, mInstanceId, mObjectId));
}


bool SocketConsumerInst::sendBufferId(int bufferId)
{
   GiveBufferMessage giveBufferMessage;
   giveBufferMessage.setBufferIndex(bufferId);
   return mSocketPtr->send(giveBufferMessage);
}


bool SocketConsumerInst::receiveBufferId(int& bufferId)
{
   GiveBufferMessage giveBufferMessage;
   if (mSocketPtr->receive(giveBufferMessage))
   {
      bufferId = giveBufferMessage.getBufferIndex();
      return true;
   }
   else
   {
      return false;
   }
}


int SocketConsumerInst::popFront(std::list<int>& buffers)
{
   int buffer = -1;
   if (!buffers.empty())
   {
      buffer = buffers.front();
      buffers.pop_front();
   }
   return buffer;
}


int SocketConsumerInst::popBack(std::list<int>& buffers)
{
   int buffer = -1;
   if (!buffers.empty())
   {
      buffer = buffers.back();
      buffers.pop_back();
   }
   return buffer;
}


int SocketConsumerInst::exchange()
{
   // Select latest buffer
   mMutex.lock();
   while (!mBuffersNew.empty())
   {
      int newBufferId = popFront(mBuffersNew);
      if (mBufferId != -1)
      {
         mBuffersOld.push_back(mBufferId);
      }
      mBufferId = newBufferId;
   }
   mIsUpdated = false;
   int bufferId = mBufferId;
   mMutex.unlock();

   if (bufferId < 0 || bufferId > 2)
   {
      return -1;
   }

   return mFds[bufferId];
}


bool SocketConsumerInst::isUpdated()
{
   mMutex.lock();
   bool isUpdated = mIsUpdated;
   mMutex.unlock();
   return isUpdated;
}


void SocketConsumerInst::clearUpdatedFlag()
{
   mMutex.lock();
   mIsUpdated = false;
   mMutex.unlock();
}


bool SocketConsumerInst::isTerminated()
{
   return !mThread.isRunning();
}


} // namespace
} // namespace
} // namespace
