/******************************************************************
*COPYRIGHT: (C) 2017 Robert Bosch GmbH
*The reproduction, distribution and utilization of this file as
*well as the communication of its contents to others without express
*authorization is prohibited. Offenders will be held liable for the
*payment of damages. All rights reserved in the event of the grant
*of a patent, utility model or design.
******************************************************************/
#include "hmibase/gadget/syncblock2/Producer.h"
#include "hmibase/gadget/syncblock2/ShmProducerInst.h"
#include "hmibase/gadget/syncblock2/SocketProducerInst.h"
#include "hmibase/util/SimpleString.h"

#include "hmibase/util/Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW
#include "trcGenProj/Header/Producer.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

using namespace std;

namespace hmibase {
namespace gadget {
namespace syncblock2 {


const char* Producer::getVersion()
{
   return "1.0";
}


Producer::Producer(int key, ProducerType producerType)
   : mKey(key)
   , mCurrentId(-1)
   , mProducerType(producerType)
{
#ifdef VARIANT_S_FTR_ENABLE_USE_SOCKET_PRODUCER
   mProducerType = SOCKET_PRODUCER;
#endif
}


Producer::~Producer()
{
   for (map<int, ProducerInst*>::iterator iter = mInstPtrs.begin(); iter != mInstPtrs.end(); ++iter)
   {
      ProducerInst* instPtr = iter->second;
      if (instPtr != NULL)
      {
         instPtr->startTermination();
      }
   }

   for (map<int, ProducerInst*>::iterator iter = mInstPtrs.begin(); iter != mInstPtrs.end(); ++iter)
   {
      ProducerInst* instPtr = iter->second;
      if (instPtr != NULL)
      {
         delete instPtr;
      }
   }
   mInstPtrs.clear();
}


bool Producer::attach(int id)
{
   map<int, ProducerInst*>::iterator iter = mInstPtrs.find(id);
   if (iter == mInstPtrs.end() || iter->second == NULL)
   {
      switch (mProducerType)
      {
         case SHM_PRODUCER:
            mInstPtrs[id] = new ShmProducerInst(mKey, id);
            break;
         case SOCKET_PRODUCER:
            mInstPtrs[id] = new SocketProducerInst(mKey, id);
            break;
         default:
            ETG_TRACE_ERR_THR(("Unknown ProducerType: %d", (int)mProducerType));
            return false;
      }
   }

   mCurrentId = id;
   return true;
}


void Producer::repair()
{
   ProducerInst* producerInstPtr = mInstPtrs[mCurrentId];

   if (producerInstPtr == NULL)
   {
      return;
   }

   if (!producerInstPtr->isTerminated())
   {
      return;
   }

   ETG_TRACE_USR4_THR(("Producer[%d][%d]: Consumer is gone, repairing connection.", mKey, mCurrentId));

   // Remember old ProducerInst Infos.
   BufferInfo bufferInfo = producerInstPtr->getBufferInfo();
   int fds[3];
   for (int i = 0; i < 3; i++)
   {
      fds[i] = producerInstPtr->getFd(i);
   }

   // delete old ProducerInst
   delete producerInstPtr;
   mInstPtrs[mCurrentId] = NULL;

   // Create new ProducerInst
   attach(mCurrentId);

   // Get new ProducerInst
   producerInstPtr = mInstPtrs[mCurrentId];

   // Set same infos as old instance had
   if (producerInstPtr != NULL)
   {
      producerInstPtr->setBufferInfo(bufferInfo);
      for (int i = 0; i < 3; i++)
      {
         producerInstPtr->setFd(i, fds[i]);
      }
   }
}


int Producer::exchange()
{
   if (mInstPtrs[mCurrentId] != NULL)
   {
      if (!mInstPtrs[mCurrentId]->isFirstExchange())
      {
         repair();
      }
      return mInstPtrs[mCurrentId]->exchange();
   }

   return -1;
}


void Producer::setBufferInfo(const BufferInfo& bufferInfo)
{
   if (mInstPtrs[mCurrentId] != NULL)
   {
      return mInstPtrs[mCurrentId]->setBufferInfo(bufferInfo);
   }
}


void Producer::addBufHandle(unsigned int bufNum, int bufHandle)
{
   if (mInstPtrs[mCurrentId] != NULL)
   {
      mInstPtrs[mCurrentId]->setFd(bufNum, bufHandle);
   }
}


int Producer::getBufHandle(unsigned int bufNum)
{
   if (mInstPtrs[mCurrentId] != NULL)
   {
      return mInstPtrs[mCurrentId]->getFd(bufNum);
   }

   return -1;
}


bool Producer::isTerminated()
{
   if (mInstPtrs[mCurrentId] != NULL)
   {
      return mInstPtrs[mCurrentId]->isTerminated();
   }
   return true;
}


bool Producer::startTermination()
{
   if (mInstPtrs[mCurrentId] != NULL)
   {
      mInstPtrs[mCurrentId]->startTermination();
   }
   return isTerminated();
}


bool Producer::checkTermination()
{
   return isTerminated();
}


} // namespace
} // namespace
} // namespace
