/******************************************************************
*COPYRIGHT: (C) 2017 Robert Bosch GmbH
*The reproduction, distribution and utilization of this file as
*well as the communication of its contents to others without express
*authorization is prohibited. Offenders will be held liable for the
*payment of damages. All rights reserved in the event of the grant
*of a patent, utility model or design.
******************************************************************/
#include "hmibase/gadget/syncblock2/Consumer.h"
#include "hmibase/gadget/syncblock2/ShmConsumerInst.h"
#include "hmibase/gadget/syncblock2/SocketConsumerInst.h"
#include "hmibase/gadget/syncblock2/socks/HelloMessage.h"
#include "hmibase/gadget/syncblock2/socks/ListeningSocket.h"
#include "hmibase/util/Error.h"
#include "hmibase/util/Timeout.h"
#include <unistd.h>

#include "hmibase/util/Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW
#include "trcGenProj/Header/Consumer.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

using namespace std;
using namespace hmibase::gadget::syncblock2::socks;

namespace hmibase {
namespace gadget {
namespace syncblock2 {


Consumer::Consumer(int key)
   : mKey(key)
   , mListeningSocketPtr(NULL)
   , mMutex()
   , mThread(*this, "Consumer")
   , mUpdateIsImplemented(true)
{
   mThread.start();
}


Consumer::~Consumer(void)
{
   mThread.finish();
}


BufferInfo Consumer::getBufferInfo(int inst)
{
   BufferInfo bufferInfo;

   mMutex.lock();
   ConsumerInst* instPtr = mInstPtrs[inst];
   if (instPtr != NULL)
   {
      bufferInfo = instPtr->getBufferInfo();
   }
   mMutex.unlock();

   return bufferInfo;
}


void Consumer::clearUpdatedFlag(int inst)
{
   mMutex.lock();
   ConsumerInst* instPtr = mInstPtrs[inst];
   if (instPtr != NULL)
   {
      instPtr->clearUpdatedFlag();
   }
   mMutex.unlock();
}


int Consumer::exchange(int inst)
{
   int result = -1;

   mMutex.lock();
   ConsumerInst* instPtr = mInstPtrs[inst];
   if (instPtr != NULL)
   {
      result = instPtr->exchange();
   }
   mMutex.unlock();

   return result;
}


void Consumer::waitForUpdate()
{
   Timeout timeout(1000); // 1 seconds

   std::vector<int> updateIds;
   while (!timeout.isReached() && updateIds.empty())
   {
      usleep(20000); // 10 milliseconds
      isUpdated(updateIds);
   }
}


void Consumer::isUpdated(std::vector<int>& updateIds)
{
   updateIds.clear();

   mMutex.lock();
   for (map<int, ConsumerInst*>::iterator iter = mInstPtrs.begin(); iter != mInstPtrs.end(); ++iter)
   {
      int           instId  = iter->first;
      ConsumerInst* instPtr = iter->second;

      if (instPtr != NULL && instPtr->isUpdated())
      {
         updateIds.push_back(instId);
      }
   }
   mMutex.unlock();
}


void Consumer::cleanupInstances()
{
   mMutex.lock();
   std::vector<int> toBeDeleted;
   for (map<int, ConsumerInst*>::iterator iter = mInstPtrs.begin(); iter != mInstPtrs.end(); ++iter)
   {
      int           instId  = iter->first;
      ConsumerInst* instPtr = iter->second;

      if (instPtr != NULL && instPtr->isTerminated())
      {
         delete instPtr;
         toBeDeleted.push_back(instId);
      }
   }

   for (size_t i = 0; i < toBeDeleted.size(); ++i)
   {
      mInstPtrs.erase(toBeDeleted[i]);
   }

   mMutex.unlock();
}


void Consumer::onStart(const bool& /*keepRunning*/)
{
   ETG_TRACE_USR4_THR(("Consumer-Thread Starting..."));
   mListeningSocketPtr = new ListeningSocket(mKey);
   ETG_TRACE_USR4_THR(("Consumer-Thread Started."));
}


bool Consumer::onRun(const bool& keepRunning)
{
//   ETG_TRACE_USR4_THR(("Consumer-Thread Running..."));
   cleanupInstances();

   if (!keepRunning || mListeningSocketPtr == NULL)
   {
      return false;
   }

   // Accept new Connections
   Socket* socketPtr = acceptSocket();
   if (socketPtr != NULL)
   {
      ETG_TRACE_USR4_THR(("Consumer-Thread: Accepted Socket."));

      Timeout timeout(1000); // 1 Second

      ETG_TRACE_USR4_THR(("Consumer-Thread: Receiving HelloMessage from Producer..."));
      HelloMessage message;
      while (!timeout.isReached() && !socketPtr->receive(message))
      {
         usleep(1000); // 1 Milliseconds
      }

      int id  = message.getInstanceId();
      int fd0 = message.getFd(0);
      int fd1 = message.getFd(1);
      int fd2 = message.getFd(2);
      ProducerType producerType = message.getProducerType();
      BufferInfo   bufferInfo   = message.getBufferInfo();

      ETG_TRACE_USR4_THR(("Consumer-Thread: Received HelloMessage from Producer. (id=%d, fd0=%d, fd1=%d, fd2=%d, type=%d, bufferSize=%d)", id, fd0, fd1, fd2, (int)producerType, bufferInfo.getSize()));

      if (id  == -1 || fd0 == -1 || fd1 == -1 || fd2 == -1)
      {
         ETG_TRACE_ERR_THR(("Consumer-Thread: Invalid HelloMessage from Producer. (id=%d, fd0=%d, fd1=%d, fd2=%d, type=%d, bufferSize=%d)", id, fd0, fd1, fd2, (int)producerType, bufferInfo.getSize()));
      }
      else
      {
         mMutex.lock();

         if (mInstPtrs[id] != NULL)
         {
            delete mInstPtrs[id];
         }

         switch (producerType)
         {
            case SHM_PRODUCER:
               mInstPtrs[id] = new ShmConsumerInst(mKey, id, fd0, fd1, fd2, bufferInfo);
               break;
            case SOCKET_PRODUCER:
               mInstPtrs[id] = new SocketConsumerInst(mKey, id, fd0, fd1, fd2, bufferInfo, socketPtr);
               socketPtr = NULL;
               break;
            default:
               ETG_TRACE_ERR_THR(("Consumer-Thread: Unknown ProducerType: %d", (int)producerType));
               break;
         }

         mMutex.unlock();
      }

      if (socketPtr != NULL)
      {
         socketPtr->close();
         delete socketPtr;
         socketPtr = NULL;
      }
   }

   // Call OnUpdate, if necessary
   vector<int> updateIds;
   isUpdated(updateIds);
   for (size_t i = 0; i < updateIds.size(); i++)
   {
      int inst = updateIds[i];
      onUpdate(inst);
      if (mUpdateIsImplemented)
      {
         clearUpdatedFlag(inst);
      }
   }

   // Wait a little bit.
   usleep(30000); // 30 Milliseconds

   return true;
}


Socket* Consumer::acceptSocket()
{
   Socket* socketPtr = new Socket();
   if (!mListeningSocketPtr->tryAccept(*socketPtr))
   {
      delete socketPtr;
      socketPtr = NULL;
   }
   return socketPtr;
}


void Consumer::onFinish(const bool& /*keepRunning*/)
{
   ETG_TRACE_USR4_THR(("Consumer-Thread Finishing..."));

   if (mListeningSocketPtr != NULL)
   {
      delete mListeningSocketPtr;
      mListeningSocketPtr = NULL;
   }

   mMutex.lock();
   for (map<int, ConsumerInst*>::iterator iter = mInstPtrs.begin(); iter != mInstPtrs.end(); ++iter)
   {
      ConsumerInst* instPtr = iter->second;
      delete instPtr;
   }
   mInstPtrs.clear();
   mMutex.unlock();

   ETG_TRACE_USR4_THR(("Consumer-Thread Finishing."));
}


} // namespace
} // namespace
} // namespace
