/******************************************************************
*COPYRIGHT: (C) 2017 Robert Bosch GmbH
*The reproduction, distribution and utilization of this file as
*well as the communication of its contents to others without express
*authorization is prohibited. Offenders will be held liable for the
*payment of damages. All rights reserved in the event of the grant
*of a patent, utility model or design.
******************************************************************/
#include "hmibase/gadget/syncblock2/BufferStatus.h"

namespace hmibase {
namespace gadget {
namespace syncblock2 {

BufferStatus::BufferStatus()
   : meSoooDirty(true)
{
   for (int i = 0; i < smNumBuffers; i++)
   {
      mStates[i] = EMPTY;
   }
}


BufferStatus::~BufferStatus()
{
   for (int i = 0; i < smNumBuffers; i++)
   {
      mStates[i] = EMPTY;
   }
}


bool BufferStatus::isDirty() const
{
   return meSoooDirty;
}


void BufferStatus::setDirty(bool value)
{
   meSoooDirty = value;
}


int BufferStatus::getNextWriteIndex()
{
   meSoooDirty = true;

   int writingBuffer = -1;
   int writtenBuffer = -1;
   int emptyBuffer   = -1;
   for (int i = 0; i < smNumBuffers; i++)
   {
      switch (mStates[i])
      {
         case WRITING:
            if (writingBuffer == -1)
            {
               writingBuffer = i;
            }
            break;
         case WRITTEN:
            if (writtenBuffer == -1)
            {
               writtenBuffer = i;
            }
            break;
         case EMPTY  :
            if (emptyBuffer == -1)
            {
               emptyBuffer = i;
            }
            break;
         default:
            break;
      }
   }

   // Clear all WRITING and WRITTEN Buffers.
   // This will remove any duplicates.
   for (int i = 0; i < smNumBuffers; i++)
   {
      switch (mStates[i])
      {
         case WRITING:
         case WRITTEN:
            mStates[i] = EMPTY;
            break;
         default:
            break;
      }
   }

   // WRITING buffer becomes new WRITTEN buffer
   if (writingBuffer > -1)
   {
      mStates[writingBuffer] = WRITTEN;
   }

   // EMPTY Buffer becomes new WRITING buffer
   // If there was none, recycle old WRITTEN buffer
   if (emptyBuffer > -1)
   {
      mStates[emptyBuffer] = WRITING;
      return emptyBuffer;
   }
   else if (writtenBuffer > -1)
   {
      mStates[writtenBuffer] = WRITING;
      return writtenBuffer;
   }
   else
   {
      return -1;
   }
}


int BufferStatus::getNextReadIndex()
{
   meSoooDirty = false;

   int writtenBuffer = -1;
   int readingBuffer = -1;
   for (int i = 0; i < smNumBuffers; i++)
   {
      switch (mStates[i])
      {
         case WRITTEN:
            if (writtenBuffer == -1)
            {
               writtenBuffer = i;
            }
            break;
         case READING:
            if (readingBuffer == -1)
            {
               readingBuffer = i;
            }
            break;
         default:
            break;
      }
   }

   // Clear all READING and WRITTEN Buffers.
   // This will remove any duplicates.
   // Also invalidate CLOSING Buffers.
   for (int i = 0; i < smNumBuffers; i++)
   {
      switch (mStates[i])
      {
         case WRITTEN:
         case READING:
         case CLOSING:
            mStates[i] = EMPTY;
            break;
         default:
            break;
      }
   }

   // WRITTEN buffer becomes new READING buffer
   // if there was none, then continue using old READING buffer
   if (writtenBuffer > -1)
   {
      mStates[writtenBuffer] = READING;
      return writtenBuffer;
   }
   else if (readingBuffer > -1)
   {
      mStates[readingBuffer] = READING;
      return readingBuffer;
   }
   else
   {
      return -1;
   }
}


void BufferStatus::releaseReadIndex()
{
   for (int i = 0; i < smNumBuffers; i++)
   {
      if (mStates[i] == READING)
      {
         mStates[i] = EMPTY;
      }
   }
}


void BufferStatus::requestClose()
{
   for (int i = 0; i < smNumBuffers; i++)
   {
      if (mStates[i] == READING)
      {
         mStates[i] = CLOSING;
      }
      else
      {
         mStates[i] = EMPTY;
      }
   }
   meSoooDirty = true;
}


bool BufferStatus::confirmClose()
{
   for (int i = 0; i < smNumBuffers; i++)
   {
      if (mStates[i] != EMPTY)
      {
         return false;
      }
   }
   return true;
}


SimpleString BufferStatus::toString() const
{
   SimpleString result = "BufferStatus:";
   for (int i = 0; i < smNumBuffers; i++)
   {
      SimpleString status;
      switch (mStates[i])
      {
         case EMPTY:
            status = "EMPTY  ";
            break;
         case READING:
            status = "READING";
            break;
         case WRITING:
            status = "WRITING";
            break;
         case WRITTEN:
            status = "WRITTEN";
            break;
         case CLOSING:
            status = "CLOSING";
            break;
         default:
            status = SimpleString::format("%7d", mStates[i]);
            break;
      }
      result += SimpleString::format("\n  [%d]: %s", i, status.cPtr());
   }
   return result;
}


} // namespace
} // namespace
} // namespace
