/* ***************************************************************************************
* FILE:          ProjectSettings.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ProjectSettings.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#if !defined(ProjectSettings_h)
#define ProjectSettings_h

#include <ScreenBroker/ScreenBroker.h>
#include <ScreenBroker/SurfaceState.h>

namespace ScreenBrokerProtocol {
/** Defines the protocol between screen broker plugins and the application transported
 *  via the 32 bits of request ID in the interface RequestCurrentStatus.
 *  Note: This request ID is not bit-coded, hence the template class RequestId<>
 *  does not apply.
 */
namespace CurrentStatusRequestId {
enum Enum
{
   /// Popup of active application & system
   ActivePopups = 0,

   /// Top popup of active application & system
   TopActivePopup,

   /// Top popup of active application & system considering popup filter
   TopActiveVisiblePopup,

   /// Top non-permanent surface
   TopActiveSurfaceNonPermanent,

   /// Top surface incl. permanent surfaces
   TopActiveSurface,

   ///TopActivePopup in foreground app
   PopupInForeground,

   /// Inactive popup that goes to background app
   PopupToBackground,

   EndofCurrentStatusRequestId
};


}

namespace CurrentStatusRequestIdForTTFisCMD {
enum Enum
{
   ActivePopups = CurrentStatusRequestId::EndofCurrentStatusRequestId,

   TopActivePopup ,

   TopActiveVisiblePopup,

   TopActiveSurfaceNonPermanent ,

   TopActiveSurface
};


}

/** Defines the protocol between screen broker plugins and the application transported
 *  via the 32 bits of request ID in the interface ActivateApplication.
 *  Note: The option with value 0 (Bit0) is reserved for the use to transport the information
 *  to show resp. hide a surface via the interface ActivateApplication.
 */
namespace ShowSurfaceMode {
enum Enum
{
   /// Bit 0: Show / Hide requested surface
   Reserved = 0
};


}

/** Defines the protocol between screen broker plugins and the application transported
 *  via the 32 bits of request ID in the interface ActivateApplication.
 *  Note: The option with value 0 (Bit0) is reserved for the use to transport the information
 *  to show resp. hide a surface via the interface ActivateApplication.
 */
namespace HideSurfaceMode {
enum Enum
{
   /// Bit 0: Show / Hide requested surface
   Reserved = 0
};


}

/** Defines the protocol between screen broker plugins and the application transported
 *  via the 32 bits of request ID in the interface ShowPopup.
 */
namespace ShowPopupMode {
enum Enum
{
   /** Bit 0: Activate / Deactivate closing popup,
    *  if it is superseded (e.g. due to a higher priority popup). */
   CloseOnSuperseded = 0,
   /** Bit 1: Activate / Deactivate closing application modal popup,
    *  if its corresponding application is closed. */
   CloseOnAppLeave,
   /** Bit 2: Enable / Disable closing popup on external touch event */
   CloseOnExternalTouch
};


}

/** Defines the protocol between screen broker plugins and the application transported
 *  via the 32 bits of request ID in the interface HidePopup.
 */
namespace HidePopupMode {
enum Enum
{
};


}

/** Defines the protocol between screen broker plugins and the application transported
 * via the 32 bits of actionId in the interface Action.
 */
namespace ActionId {
enum Enum
{
   StopAnimations = 0,
   Commit
};


}

/*namespace SurfaceState {
enum Enum
{
   AnimationStarted = ScreenBroker::SurfaceState::AnimationStarted,
   AnimationStopped = ScreenBroker::SurfaceState::AnimationStopped
};


}*/

namespace InputFocus {
enum Enum
{
   Default = 0,
   // force focus
   Force,
   // reset force flag
   Reset
};


}


/** Wrapper class to perform bit manipulation to a 32 bit data value, here
 *  used especially to configure the various kinds of screen broker request IDs.
 */
template <typename Mode>
class RequestId
{
   public:
      ///
      explicit RequestId(ScreenBroker::UInt32 data = 0ul) : mData(data) {}
      ///
      bool Test(Mode offset) const
      {
         return ((mData & (1u << offset)) != 0);
      }
      ///
      void Set(Mode offset)
      {
         mData |= (1u << static_cast<ScreenBroker::UInt32>(offset));
      }
      ///
      void Clear(Mode offset)
      {
         mData &= ~(1u << static_cast<ScreenBroker::UInt32>(offset));
      }
      ///
      void Reset()
      {
         mData = 0ul;
      }
      ///
      ScreenBroker::UInt32 Get() const
      {
         return mData;
      }
   private:
      ScreenBroker::UInt32 mData;
};


}


///
class ProjectSettings
{
   public:

      static bool IsPermanent(ScreenBroker::UInt32 surfaceId);
      static bool IsPopup(ScreenBroker::UInt32 surfaceId);

      static bool DisableInputReceptionOfUnderlyingSurface();
      static bool DisableInputReceptionOfUnderlyingPopup();

      /*
       * Returns configuration for KeyHandler Plugin.
       * */
      static void GetKeyboardSurfaceConfiguration(ScreenBroker::UInt32& layerId,
            ScreenBroker::UInt32& surfaceId);
};


#endif
