/* ***************************************************************************************
* FILE:          WaylandEventHandler.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  WaylandEventHandler.cpp is part of HMI-Base ScreenBrokerPlugins
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

// =============================================================================
// Using the "do { ... } while(0) construct within macros is a common practice in order
// to make the use of macros safe within conditions.
//lint -emacro({717}, CHECK_WAYLAND_CONTEXT)    "do ... while(0);"
//To avoid renaming of local symbol (hidden symbol (time(long *) is not used)
//lint -esym(578, time) "Declaration of symbol 'time' hides symbol"
// =============================================================================

#include "WaylandEventHandler.h"
#include "WaylandContext.h"

#ifndef VARIANT_S_FTR_ENABLE_IVI_SHELL
#include <ilm/ilm_client.h>
#endif
#include <string.h>

#define CHECK_WAYLAND_CONTEXT(waylandContext)                                          \
    do {                                                                        \
        if (0 == waylandContext) {                                                    \
            ETG_TRACE_ERR(("%40s: Invalid wayland context", __FUNCTION__));\
            return;                                                             \
        }                                                                       \
    } while(0)

#define KEYBOARD_INTERFACE_TRACE(format, ...)    \
    //DEACTIVATED_LOG_DEBUG("%s: data(%p), keyboard(%p), " format, __FUNCTION__, data, keyboard, ## __VA_ARGS__);
#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_PLUGINS
#include "trcGenProj/Header/WaylandEventHandler.cpp.trc.h"
#endif


namespace ScreenBroker {
// SCREENBROKER_LOG_SET_REALM(LogRealm::Wayland);

// ------------------------------------------------------------------------
const struct wl_registry_listener* WaylandEventHandler::GetRegistryEventHandler()
{
   static const struct wl_registry_listener cRegistryEventHandler =
   {
      WaylandEventHandler::RegistryEventGlobal,
      WaylandEventHandler::RegistryEventGlobalRemove
   };

   return &cRegistryEventHandler;
}


// ------------------------------------------------------------------------
const struct serverinfo_listener* GetServerInfoEventHandler()
{
   static const struct serverinfo_listener cServerInfoEventHandler =
   {
      WaylandEventHandler::ServerInfoEvent,
   };

   return &cServerInfoEventHandler;
}


// ------------------------------------------------------------------------
const struct wl_seat_listener* WaylandEventHandler::GetSeatEventHandler()
{
   static const struct wl_seat_listener cSeatEventHandler =
   {
      WaylandEventHandler::SeatEventCapabilities,
      WaylandEventHandler::SeatHandleName
   };

   return &cSeatEventHandler;
}


// ------------------------------------------------------------------------
const struct wl_keyboard_listener* WaylandEventHandler::GetKeyboardEventHandler()
{
   static const struct wl_keyboard_listener cKeyboardEventHandler =
   {
      WaylandEventHandler::KeyboardEventKeymap,
      WaylandEventHandler::KeyboardEventEnter,
      WaylandEventHandler::KeyboardEventLeave,
      WaylandEventHandler::KeyboardEventKey,
      WaylandEventHandler::KeyboardEventModifiers
#if (WAYLAND_VERSION_MAJOR >=1 && WAYLAND_VERSION_MINOR >= 6)
      , WaylandEventHandler::KeyboardHandleRepeatInfo
#endif
   };

   return &cKeyboardEventHandler;
}


// ------------------------------------------------------------------------
void WaylandEventHandler::RegistryEventGlobal(void* data,
      struct wl_registry* registry,
      uint32_t name,
      const char* interface,
      uint32_t version)
{
   // Retrieve current wayland context
   WaylandContext* lContext = static_cast<WaylandContext*>(data);
   if (0 == lContext)
   {
      ETG_TRACE_ERR(("%40s: Invalid wayland context", __FUNCTION__));
      return;
   }

   // Register compositor handler
   if (0 == strcmp(interface, "wl_compositor"))
   {
      ETG_TRACE_USR1(("%40s: data(%p), registry(%p), name(%u), interface(%40s), version(%u)",
                      __FUNCTION__, data, registry, name,
                      interface, version));

      struct wl_compositor* lCompositor =
         (struct wl_compositor*) wl_registry_bind(registry,
               name,
               &wl_compositor_interface,
               version);
      lContext->SetCompositor(lCompositor);
   }

   // Register serverinfo handler
   if (0 == strcmp(interface, "serverinfo"))
   {
      ETG_TRACE_USR1(("%40s: data(%p), registry(%p), name(%u), interface(%40s), version(%u)",
                      __FUNCTION__, data, registry, name,
                      interface, version));

      struct serverinfo* lServerInfo =
         (struct serverinfo*) wl_registry_bind(registry,
               name,
               &serverinfo_interface,
               1);
      serverinfo_add_listener(lServerInfo, GetServerInfoEventHandler(), data);
      serverinfo_get_connection_id(lServerInfo);
      lContext->SetServerInfo(lServerInfo);
   }

   // Register seat event handler
   if (0 == strcmp(interface, "wl_seat"))
   {
      ETG_TRACE_USR1(("%40s: data(%p), registry(%p), name(%u), interface(%40s), version(%u)",
                      __FUNCTION__, data, registry, name,
                      interface, version));

      struct wl_seat* lSeat = (wl_seat*) wl_registry_bind(
                                 registry, name, &wl_seat_interface, 1);
      wl_seat_add_listener(lSeat, GetSeatEventHandler(), data);
      lContext->SetSeat(lSeat);

      // Do another roundtrip to process requests issued by this seat listener
      wl_display_roundtrip(lContext->GetDisplay());
      ETG_TRACE_USR4(("Wayland roundtrip of seat event handler finished"));
   }
}


// ------------------------------------------------------------------------
void WaylandEventHandler::RegistryEventGlobalRemove(void* data,
      struct wl_registry* registry,
      uint32_t name)
{
   ETG_TRACE_USR1(("%40s: data(%p), registry(%p), name(%u)",
                   __FUNCTION__, data, registry, name));

   // Retrieve current wayland context
   WaylandContext* lContext = static_cast<WaylandContext*>(data);
   if (0 == lContext)
   {
      ETG_TRACE_ERR(("%40s: Invalid wayland context", __FUNCTION__));
      return;
   }

   ETG_TRACE_SYS(("Remove request for registry object '%d'", name));

   (void) registry;
}


// ------------------------------------------------------------------------
void WaylandEventHandler::ServerInfoEvent(void* data,
      struct serverinfo* serverInfo,
      uint32_t clientHandle)
{
   (void) serverInfo;
   ETG_TRACE_USR1(("%40s: data(%p), serverinfo(%p), clientHandle(%u)",
                   __FUNCTION__, data, serverInfo, clientHandle));

   // Retrieve current wayland context
   WaylandContext* lContext = static_cast<WaylandContext*>(data);
   if (0 == lContext)
   {
      ETG_TRACE_ERR(("%40s: Invalid wayland context", __FUNCTION__));
      return;
   }

   lContext->SetConnectionId(clientHandle);
}


// ------------------------------------------------------------------------
void WaylandEventHandler::SeatEventCapabilities(void* data,
      struct wl_seat* seat,
      uint32_t caps)
{
   ETG_TRACE_USR1(("%40s: data(%p), seat(%p), caps(%u)",
                   __FUNCTION__, data, seat, caps));

   // Retrieve current wayland context
   WaylandContext* lContext = static_cast<WaylandContext*>(data);
   if (0 == lContext)
   {
      ETG_TRACE_ERR(("%40s: Invalid wayland context", __FUNCTION__));
      return;
   }

   // Check if seat was already initialized
   struct wl_seat* lSeat = lContext->GetSeat();
   if (0 == lSeat)
   {
      ETG_TRACE_SYS(("Seat not initialized!"));
      return;
   }

   if (seat != lSeat)
   {
      ETG_TRACE_SYS(("Wrong seat instance (%p) requested! Initialization done with seat: %p",
                     seat,
                     lSeat));
      return;
   }

   // Register / de-register keyboard event handler
   struct wl_keyboard* lKeyboard = lContext->GetKeyboard();
   if ((caps & static_cast<uint32_t>(WL_SEAT_CAPABILITY_KEYBOARD)) && (0 == lKeyboard))
   {
      ETG_TRACE_USR1(("Enabling seat capability keyboard"));
      lKeyboard = wl_seat_get_keyboard(lSeat);
      wl_keyboard_set_user_data(lKeyboard, data);
      wl_keyboard_add_listener(lKeyboard, GetKeyboardEventHandler(), data);
   }
   else if (!(caps & static_cast<uint32_t>(WL_SEAT_CAPABILITY_KEYBOARD)) && (0 != lKeyboard))
   {
      wl_keyboard_destroy(lKeyboard);
      lKeyboard = 0;
   }
   lContext->SetKeyboard(lKeyboard);
}


void WaylandEventHandler::SeatHandleName(void* data, struct wl_seat* seat, const char* name)
{
   (void)seat;
   (void)data;
   ETG_TRACE_USR1(("SeatHandleGetName [%40s]", name));
}


// ------------------------------------------------------------------------
void WaylandEventHandler::KeyboardEventKeymap(void* data,
      struct wl_keyboard* keyboard,
      uint32_t format,
      int fd,
      uint32_t size)
{
   ETG_TRACE_USR4(("format(%u), fd(%d), size(%u)", format, fd, size));

   // Provide wayland context
   WaylandContext* lContext = static_cast<WaylandContext*>(data);
   if (0 == lContext)
   {
      ETG_TRACE_ERR(("%40s: Invalid wayland context", __FUNCTION__));
      return;
   }

   // Call event hook
   if (0 != lContext->GetEventHook())
   {
      lContext->GetEventHook()->OnKeyboardEventKeymap(lContext,
            keyboard,
            format,
            fd,
            size);
   }
   else
   {
      ETG_TRACE_SYS(("%40s: Event hook not set. Ignoring event!", __FUNCTION__));
   }
}


// ------------------------------------------------------------------------
void WaylandEventHandler::KeyboardEventEnter(void* data,
      struct wl_keyboard* keyboard,
      uint32_t serial,
      struct wl_surface* surface,
      struct wl_array* keys)
{
   ETG_TRACE_USR4(("serial(%u), surface(%p), keys(%p)",
                   serial, surface, keys));

   // Provide wayland context
   WaylandContext* lContext = static_cast<WaylandContext*>(data);
   if (0 == lContext)
   {
      ETG_TRACE_ERR(("%40s: Invalid wayland context", __FUNCTION__));
      return;
   }

   // Call event hook
   if (0 != lContext->GetEventHook())
   {
      lContext->GetEventHook()->OnKeyboardEventEnter(lContext,
            keyboard,
            serial,
            surface,
            keys);
   }
   else
   {
      ETG_TRACE_SYS(("%40s: Event hook not set. Ignoring event!", __FUNCTION__));
   }
}


// ------------------------------------------------------------------------
void WaylandEventHandler::KeyboardEventLeave(void* data,
      struct wl_keyboard* keyboard,
      uint32_t serial,
      struct wl_surface* surface)
{
   ETG_TRACE_USR4(("serial(%u), surface(%p)",
                   serial, surface));

   // Provide wayland context
   WaylandContext* lContext = static_cast<WaylandContext*>(data);
   if (0 == lContext)
   {
      ETG_TRACE_ERR(("%40s: Invalid wayland context", __FUNCTION__));
      return;
   }

   // Call event hook
   if (0 != lContext->GetEventHook())
   {
      lContext->GetEventHook()->OnKeyboardEventLeave(lContext,
            keyboard,
            serial,
            surface);
   }
   else
   {
      ETG_TRACE_SYS(("%40s: Event hook not set. Ignoring event!", __FUNCTION__));
   }
}


// ------------------------------------------------------------------------
void WaylandEventHandler::KeyboardEventKey(void* data,
      struct wl_keyboard* keyboard,
      uint32_t serial,
      uint32_t time,
      uint32_t key,
      uint32_t state)
{
   ETG_TRACE_USR4(("serial(%u), time(%u), key(%u), state(%u)",
                   serial, time, key, state));

   // Provide wayland context
   WaylandContext* lContext = static_cast<WaylandContext*>(data);
   if (0 == lContext)
   {
      ETG_TRACE_ERR(("%40s: Invalid wayland context", __FUNCTION__));
      return;
   }

   // Call event hook
   if (0 != lContext->GetEventHook())
   {
      lContext->GetEventHook()->OnKeyboardEventKey(lContext,
            keyboard,
            serial,
            time,
            key,
            state);
   }
   else
   {
      ETG_TRACE_SYS(("%40s: Event hook not set. Ignoring event!", __FUNCTION__));
   }
}


// ------------------------------------------------------------------------
void WaylandEventHandler::KeyboardEventModifiers(void* data,
      struct wl_keyboard* keyboard,
      uint32_t serial,
      uint32_t mods_depressed,
      uint32_t mods_latched,
      uint32_t mods_locked,
      uint32_t group)
{
   ETG_TRACE_USR4(("serial(%u), mods_depressed(%u), "
                   "mods_latched(%u), mods_locked(%u), group(%u)",
                   serial, mods_depressed,
                   mods_latched, mods_locked, group));

   // Provide wayland context
   WaylandContext* lContext = static_cast<WaylandContext*>(data);
   if (0 == lContext)
   {
      ETG_TRACE_ERR(("%40s: Invalid wayland context", __FUNCTION__));
      return;
   }

   // Call event hook
   if (0 != lContext->GetEventHook())
   {
      lContext->GetEventHook()->OnKeyboardEventModifiers(lContext,
            keyboard,
            serial,
            mods_depressed,
            mods_latched,
            mods_locked,
            group);
   }
   else
   {
      ETG_TRACE_SYS(("%40s: Event hook not set. Ignoring event!", __FUNCTION__));
   }
}


#if (WAYLAND_VERSION_MAJOR >=1 && WAYLAND_VERSION_MINOR >= 6)
void WaylandEventHandler::KeyboardHandleRepeatInfo(void* data, struct wl_keyboard* keyboard,
      int32_t rate, int32_t delay)
{
   (void) data;
   (void) keyboard;
   (void) rate;
   (void) delay;
}


#endif
}
