/* ***************************************************************************************
* FILE:          WaylandContext.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  WaylandContext.cpp is part of HMI-Base ScreenBrokerPlugins
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

// =============================================================================
//Pointer members are freed in Destroy function.
//lint -esym(1579, ScreenBroker::WaylandContext::*) "Pointer member might have been freed by a separate function but no function was seen"
//Pointer cast is required due to wayland interface.
//lint -efunc(740, ScreenBroker::WaylandContext::Init) " Unusual pointer cast (incompatible indirect types)
// =============================================================================

#include "WaylandContext.h"
#include "WaylandEventHandler.h"

#include <Shared/IlmAccessor.h>
#include <poll.h>
#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_PLUGINS
#include "trcGenProj/Header/WaylandContext.cpp.trc.h"
#endif


namespace ScreenBroker {
// SCREENBROKER_LOG_SET_REALM(LogRealm::Wayland);

// ------------------------------------------------------------------------
WaylandContext::WaylandContext()
   : mDisplay(0)
   , mRegistry(0)
   , mEventQueue(0)
   , mCompositor(0)
   , mServerInfo(0)
   , mConnectionId(0)
   , mSeat(0)
   , mKeyboard(0)
   , mSurface(0)
   , mData(0)
   , mEventHook(0)
{
}


// ------------------------------------------------------------------------
WaylandContext::~WaylandContext()
{
   Destroy();
}


// ------------------------------------------------------------------------
bool WaylandContext::Init(const char* displayId,
                          void* data,
                          WaylandEventHook* eventHook)
{
   ETG_TRACE_USR4(("Initializing wayland context"));
   mDisplay = wl_display_connect(displayId);
   bool lRc = (0 != mDisplay);
   if (lRc)
   {
      ETG_TRACE_USR1(("Wayland connection to display established: %p", mDisplay));
      mData = data;
      mEventHook = eventHook;

      mRegistry = wl_display_get_registry(mDisplay);
      wl_registry_add_listener(mRegistry, WaylandEventHandler::GetRegistryEventHandler(), this);
      wl_display_dispatch(mDisplay);
      wl_display_roundtrip(mDisplay);

      if (0 == mCompositor)
      {
         ETG_TRACE_ERR(("Failed to create wayland compositor listener for wl_display(%p)!",
                        mDisplay));
      }

      if (0 != mSeat)
      {
         mEventQueue = wl_display_create_queue(mDisplay);
         if (0 != mEventQueue)
         {
            wl_proxy_set_queue((struct wl_proxy*) mSeat, mEventQueue);
            ETG_TRACE_USR1(("Created wayland input event queue(%p) for wl_display(%p)",
                            mEventQueue,
                            mDisplay));
            if (0 != mKeyboard)
            {
               ETG_TRACE_USR1(("Enable keyboard events listening in wayland input event queue(%p)",
                               mEventQueue));
               wl_proxy_set_queue((struct wl_proxy*) mKeyboard, mEventQueue);
            }
         }
         else
         {
            ETG_TRACE_ERR(("Failed to create wayland input event queue for wl_display(%p)!",
                           mDisplay));
         }
      }
      else
      {
         ETG_TRACE_ERR(("Failed to create wayland seat event listener for wl_display(%p)!",
                        mDisplay));
      }
      lRc = (0 != mCompositor) && (0 != mSeat);
   }
   else
   {
      ETG_TRACE_ERR(("Failed to acquire wayland connection to display!"));
   }

   return lRc;
}


// ------------------------------------------------------------------------
struct wl_display* WaylandContext::GetDisplay()
{
   CriticalSectionLocker lLock(&mCs);
   return mDisplay;
}


// ------------------------------------------------------------------------
void WaylandContext::Destroy()
{
   CriticalSectionLocker lLock(&mCs);
   if (0 != mKeyboard)
   {
      wl_keyboard_destroy(mKeyboard);
      mKeyboard = 0;
   }
   if (0 != mSeat)
   {
      wl_seat_destroy(mSeat);
      mSeat = 0;
   }
   if (0 != mServerInfo)
   {
      serverinfo_destroy(mServerInfo);
      mServerInfo = 0;
   }
   if (0 != mCompositor)
   {
      wl_compositor_destroy(mCompositor);
      mCompositor = 0;
   }
   if (0 != mRegistry)
   {
      wl_registry_destroy(mRegistry);
      mRegistry = 0;
   }

   mConnectionId = 0;
   mData = 0;
   mEventHook = 0;

   if (0 != mEventQueue)
   {
      wl_event_queue_destroy(mEventQueue);
      mEventQueue = 0;
   }
   if (0 != mDisplay)
   {
      wl_display_disconnect(mDisplay);
      mDisplay = 0;
   }
}


// ------------------------------------------------------------------------
void WaylandContext::Listen()
{
   struct pollfd lPollFd[1];
   // Dispatch global queue first
   (void) wl_display_dispatch_pending(mDisplay);
   {
      CriticalSectionLocker lLock(&mCs);
      // Creating poll wayland display file descriptor for capturing incoming events
      lPollFd[0].fd = wl_display_get_fd(mDisplay);
      lPollFd[0].events = POLLIN;

#if ((WAYLAND_VERSION_MAJOR >= 1) && (WAYLAND_VERSION_MINOR >= 2))
      // Empty queue before getting exclusive access
      while (-1 == wl_display_prepare_read_queue(mDisplay, mEventQueue))
      {
         (void) wl_display_dispatch_queue_pending(mDisplay, mEventQueue);
      }
#else
      // Empty queue without getting exclusive access (old version)
      (void) wl_display_dispatch_queue_pending(mDisplay, mEventQueue);
#endif
      // Send buffered requests to server
      (void) wl_display_flush(mDisplay);
   }

   // Listen on wayland display file descriptor using poll
   switch (poll(lPollFd, 1, -1))
   {
      case -1:    // error
#if ((WAYLAND_VERSION_MAJOR >= 1) && (WAYLAND_VERSION_MINOR >= 2))
         wl_display_cancel_read(mDisplay);
#endif
         break;
      case 0:     // timed out
         (void) wl_display_dispatch_queue_pending(mDisplay, mEventQueue);
         break;
      default:    // success
         if (lPollFd[0].revents & POLLIN)
         {
#if ((WAYLAND_VERSION_MAJOR >= 1) && (WAYLAND_VERSION_MINOR >= 2))
            (void) wl_display_read_events(mDisplay);
            (void) wl_display_dispatch_queue_pending(mDisplay, mEventQueue);
#else
            (void) wl_display_dispatch_queue(mDisplay, mEventQueue);
#endif
         }
         break;
   }
}


}
