/* ***************************************************************************************
* FILE:          PopupFilter.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  PopupFilter.cpp is part of HMI-Base ScreenBrokerPlugins
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#include "PopupFilter.h"
//#include "IlmAccessor.h"

#include <ScreenBrokerSettings/Settings.h>
#include <algorithm>
#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_PLUGINS
#include "trcGenProj/Header/PopupFilter.cpp.trc.h"
#endif


namespace ScreenBroker {
// SCREENBROKER_LOG_SET_REALM(LogRealm::States);

// ------------------------------------------------------------------------
// (Plugin) global handled PopupFilter object has to be instantiated once for all plugins
// This is normally done in ScreenBrokerActivator.cpp.
extern PopupFilter gPopupFilterInstance;

// ------------------------------------------------------------------------
PopupFilter& PopupFilter::GetInstance()
{
   return gPopupFilterInstance;
}


// ------------------------------------------------------------------------
PopupFilter::PopupFilter()
{
}


// ------------------------------------------------------------------------
void PopupFilter::Insert(UInt32 displayId,
                         UInt32 appId,
                         const PopupPresentationArg& filterArg,
                         bool disableAll)
{
   DisplayAppFilterMap::const_iterator lItDisplay = mDisplayAppFilterMap.find(displayId);
   if (mDisplayAppFilterMap.end() != lItDisplay)
   {
      AppFilterMap lAppFilterMap = (*lItDisplay).second;
      lAppFilterMap[appId] = Properties(filterArg, disableAll);
      mDisplayAppFilterMap[displayId] = lAppFilterMap;
   }
   else
   {
      AppFilterMap lAppFilterMap;
      lAppFilterMap[appId] = Properties(filterArg, disableAll);
      mDisplayAppFilterMap[displayId] = lAppFilterMap;
   }
}


// ------------------------------------------------------------------------
bool PopupFilter::Get(UInt32 displayId,
                      UInt32 appId,
                      PopupPresentationArg& popupPresentationArg) const
{
   DisplayAppFilterMap::const_iterator lItDisplay = mDisplayAppFilterMap.find(displayId);
   bool lRc = (mDisplayAppFilterMap.end() != lItDisplay);
   if (lRc)
   {
      AppFilterMap lAppFilterMap = (*lItDisplay).second;

      AppFilterMap::const_iterator lItApp = lAppFilterMap.find(appId);
      lRc = (lAppFilterMap.end() != lItApp);
      if (lRc)
      {
         popupPresentationArg = (*lItApp).second.GetPopupPresentationArg();
      }
   }
   return lRc;
}


// ------------------------------------------------------------------------
bool PopupFilter::IsDisableAll(UInt32 displayId,
                               UInt32 appId) const
{
   DisplayAppFilterMap::const_iterator lItDisplay = mDisplayAppFilterMap.find(displayId);
   bool lRc = (mDisplayAppFilterMap.end() != lItDisplay);
   if (lRc)
   {
      AppFilterMap lAppFilterMap = (*lItDisplay).second;

      AppFilterMap::const_iterator lItApp = lAppFilterMap.find(appId);
      lRc = (lAppFilterMap.end() != lItApp);
      lRc = lRc && (*lItApp).second.GetDisableAll();
   }
   return lRc;
}


// ------------------------------------------------------------------------
void PopupFilter::Delete(UInt32 displayId,
                         UInt32 appId)
{
   DisplayAppFilterMap::const_iterator lItDisplay = mDisplayAppFilterMap.find(displayId);
   if (mDisplayAppFilterMap.end() != lItDisplay)
   {
      AppFilterMap lAppFilterMap = (*lItDisplay).second;
      lAppFilterMap.erase(appId);
      mDisplayAppFilterMap[displayId] = lAppFilterMap;
   }
}


// ------------------------------------------------------------------------
bool PopupFilter::Exists(UInt32 displayId,
                         UInt32 appId) const
{
   DisplayAppFilterMap::const_iterator lItDisplay = mDisplayAppFilterMap.find(displayId);
   bool lRc = (mDisplayAppFilterMap.end() != lItDisplay);
   if (lRc)
   {
      AppFilterMap lAppFilterMap = (*lItDisplay).second;
      AppFilterMap::const_iterator lItApp = lAppFilterMap.find(appId);
      lRc = (lAppFilterMap.end() != lItApp);
   }
   return lRc;
}


// ------------------------------------------------------------------------
void PopupFilter::Dump()
{
   ETG_TRACE_USR1(("$$$ Dump popup filters:"));

   Int i = 0;
   for (DisplayAppFilterMap::const_iterator lItDisplay = mDisplayAppFilterMap.begin(); mDisplayAppFilterMap.end() != lItDisplay; ++lItDisplay)
   {
      UInt32 lDisplayId = (*lItDisplay).first;
      AppFilterMap lAppFilterMap = (*lItDisplay).second;

      for (AppFilterMap::const_iterator lItApp = lAppFilterMap.begin(); lAppFilterMap.end() != lItApp; ++lItApp)
      {
         ++i;
         UInt32 lAppId = (*lItApp).first;
         const PopupPresentationArg& lPopupPresentationArg = (*lItApp).second.GetPopupPresentationArg();
         bool lDisableAll = (*lItApp).second.GetDisableAll();
         ETG_TRACE_USR1(("$   %d. display: %d application: %d (disableAll: %d, Priority: %u)",
                         i,
                         lDisplayId,
                         lAppId,
                         lDisableAll,
                         lPopupPresentationArg.Priority()));
      }
   }
}


// ------------------------------------------------------------------------
void PopupFilter::Reset()
{
   mDisplayAppFilterMap.clear();
}


}
