/* ***************************************************************************************
* FILE:          PluginActions.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  PluginActions.h is part of HMI-Base ScreenBrokerPlugins
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#if !defined(ScreenBrokerPlugins_Shared_PluginActions_h)
#define ScreenBrokerPlugins_Shared_PluginActions_h

#include <ScreenBroker/ScreenBroker.h>
#include <ScreenBroker/SurfaceState.h>
#include <ScreenBroker/LayerState.h>
#include <ScreenBroker/Service/PopupManager/PopupState.h>
#include <Shared/Animation/AnimationType.h>
#include <Shared/Animation/AnimationUserData.h>
#include <string>
#include <list>

namespace ScreenBroker {

///
class PluginActions
{
   public:
      typedef std::set<UInt32> ApplicationList;

      /** Sets the visibility for a surface and sends a notification (visible -> Mapped, invisible -> Unmapped)
       *  to the client application holding this very surface.
       *  @param surfaceId The surface for which the visibility shall be manipulated.
       *  @param surfaceState Needed for visibility determination.
       *  @param userData The user data, which shall be sent back in the notification (NotifyOnSurfaceStateChange()).
       *  @param updateSurfaceVisibility True, if surface visibility on ILM shall be update, false if no ILM update shall happen.
       *  @param sendNotification True, if surface state changed notification shall be sent.
       *  @return True if successful, false otherwise.
       */
      static bool NotifyVisibility(UInt32 surfaceId,
                                   ScreenBroker::SurfaceState::Enum surfaceState,
                                   UInt32 userData,
                                   UInt32 customAnimationType,
                                   bool updateSurfaceVisibility = true,
                                   bool sendNotification = true);

      /** Sets the visibility for a popup and sends a notification (visible -> Mapped, invisible -> Unmapped)
       *  to the client application holding this very surface.
       *  @param popupState The popup for which the visibility shall be manipulated.
       *  @param surfaceState State of the surface.
       *  @param updateSurfaceVisibility True, if surface visibility on ILM shall be update, false if no ILM update shall happen.
       *  @param sendNotification True, if clients shall be notified of surface state changes.
       *  @return True if successful, false otherwise.
       */
      static bool NotifyPopupVisibility(PopupState* popupState,
                                        ScreenBroker::SurfaceState::Enum surfaceState,
                                        bool updateSurfaceVisibility = true,
                                        bool sendNotification = true,
                                        bool notifyOnPopupStatus = false);

      /**
       * @brief Informs about the activation state of a popup.
       * A popup is active, either it has system modality, or in case of application modality, the popup
       * belongs to an active application.
       * @param popupState Instance of the popup to be checked.
       * @return True, if the given popup is active in context of the given application, false otherwise.
       */
      static bool IsPopupActive(const PopupState& popupState);

      /**
       * @brief Informs if the given popup state is to be filtered out by the currently set popup filter.
       * @param popupState Instance of the popup to be checked.
       * @return True, if for the given popup based on given application the currently set popup filter applies,
       * false otherwise.
       */
      static bool IsPopupFiltered(const PopupState& popupState);

      /**
       * @brief Enables/disables (depending on surfaceState) popup filter of given application and notifies the applications,
       * if any of their popups are filtered/unfiltered.
       *
       * Note: The applications need to react of this respective surface state notifications (Filtered resp. Unfiltered), in order
       * to hide/show the popup views.
       */
      static void NotifyPopupFilter();

      /**
       * @brief GetTopVisiblePopup
       * @return
       */
      static PopupState* GetTopVisiblePopup();

      static PopupState* GetTopVisiblePopupForDisplay(UInt32 surfaceId);

      /**
       * @brief Re-organizes the application modal popups attached to the closing application.
       * Useful in case of an application switch.
       * @param surfaceId of Application, which is closed.
       */
      static void ProcessPopupsOnAppHide(UInt32 surfaceId);

      /**
       * @brief Maps the top popup of the active application, if queued.
       * Useful in case of an application switch.
       * @param appId Application for which the top popup has to be mapped.
       */
      static void ProcessPopupsOnAppShow(UInt32 appId);

      /**
       * @brief Cleans up expired popups from the list of active popups.
       * @return Current not expired top popup state, which is not filtered.
       */
      static PopupState* RemoveExpiredTopPopups(UInt32 screenArea = 0);

      /**
       * @brief Checks if the popup timing of given popup is still valid.
       * @param popupState Popup to check the timing.
       * @return True, timing still valid, false otherwise.
       */
      static bool IsPopupTimingValid(const PopupState& popupState);

      /**
       * @brief Provides the string representation of the given surface state.
       * @param surfaceState Surface state to provide the string representation.
       * @return String representation of the given surface state.
       */
      static std::string SurfaceStateString(SurfaceState::Enum surfaceState);

      /**
       * @brief Provides the string representation of the given animation type.
       * @param animationType Animation type to provide the string representation.
       * @return String representation of the given animation type.
       */
      static std::string AnimationTypeString(AnimationType::Enum animationType);

      /**
       * @brief Search the top most surface which has the XML event tag
       * (see ScreenLayouts.xml) for event "Key" set.
       * @param displayId Display for which the top most active surface with enabled key events shall be returned.
       * @param ignorePermanentSurfaces Shall permanent surfaces be ignored?
       * @param includePopupSurfaces Shall popup surfaces be included?
       * @return Top most active surface, which has key events enabled.
       */
      static UInt32 GetTopKeyEventSurface(UInt32 displayId,
                                          bool ignorePermanentSurfaces = false,
                                          bool includePopupSurfaces = false);

      ///
      static void SendNotification(UInt32 surfaceId,
                                   const AnimationUserData& animationUserData);

      ///
      static void SendLayerAnimationState(UInt32 userData,
                                          UInt32 surfaceId,
                                          ScreenBroker::LayerState::Enum animationState);

      ///
      static void SendAnimationState(UInt32 userData,
                                     UInt32 surfaceId,
                                     ScreenBroker::SurfaceState::Enum animationState);

      ///
      static bool CalculateVisibility(SurfaceState::Enum surfaceState);

      ///
      static void CheckExplicitCommit(UInt32 surfaceId);

      /**
       * @brief GetDisplayIdOfSurface Get the DisplayId of a surface.
       * @param surfaceId The surface for which the DisplayId should be returned.
       * @return DisplayId of the given Surface
       */
      static UInt32 GetDisplayIdOfSurface(UInt32 surfaceId);

      /**
       * @brief GetDisplayIdOfScreenArea Get the DisplayId of a screenArea.
       * @param screenArea The screenArea for which the DisplayId should be returned.
       * @return DisplayId of the given ScreenArea
       */
      static UInt32 GetDisplayIdOfScreenArea(UInt32 screenArea);

      /**
       * @brief GetActiveApplications Get active applications.
       * @return Ids of active applications
       */
      static ApplicationList GetActiveApplications(UInt32 screenArea);

      ///
      static bool IsNotifyOnPopStatusForScreenAreaId(UInt32 screenAreaId);
};


}

#endif
