/* ***************************************************************************************
* FILE:          PluginActions.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  PluginActions.cpp is part of HMI-Base ScreenBrokerPlugins
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

// =============================================================================
// This header file is out of scope of sbplugins.
//lint -esym(451, errno.h) "Header file '/usr/include/errno.h' repeatedly included but does not have a standard include guard"
// =============================================================================

#include <Base.h>
#include "PluginActions.h"

#include "PopupFilter.h"
#include "IlmAccessor.h"
#include "ActivePopups.h"
#include "ActiveSurfaces.h"
#include <ScreenBroker/Service/ServiceApi.h>
#include <ScreenBroker/Service/PopupManager/PopupManager.h>
#include <ScreenBroker/Util/Time.h>
#include <Shared/FocusHandler.h>
#include <Shared/AnimationHandler.h>
#include <Shared/Animation/AnimationUserData.h>
#include <Shared/Animation/AnimationBase.h>
#include <Shared/Animation/AnimationFactory.h>
#include <ScreenBrokerSettings/Settings.h>
#include <errno.h>
#include <ScreenLayouter/ScreenLayouter.h>

// ----------------------------------------------------------------------------
#define CASE_ENUM2STRING(ns, val) case ns::val: return #val
#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_PLUGINS
#include "trcGenProj/Header/PluginActions.cpp.trc.h"
#endif


namespace ScreenBroker {
// SCREENBROKER_LOG_SET_REALM(LogRealm::Actions);

// ------------------------------------------------------------------------
bool PluginActions::NotifyVisibility(UInt32 surfaceId,
                                     SurfaceState::Enum surfaceState,
                                     UInt32 userData,
                                     UInt32 customAnimationType,
                                     bool updateSurfaceVisibility,
                                     bool sendNotification)
{
   bool lRc = true;
   AnimationDirection::Enum lAnimationDirection = AnimationDirection::Undefined;

   // Calculate and set visiblility
   bool lVisibility = CalculateVisibility(surfaceState);
   AnimationUserData lAnimationUserData(userData, surfaceState, sendNotification, lVisibility);

   bool lAnimationEstablished = false;
   if (updateSurfaceVisibility)
   {
      // Retrieve instance of screen layouter
      ScreenLayouter* lScreenLayouter = PLUGIN(ScreenLayouter);
      if (0 != lScreenLayouter)
      {
         lAnimationDirection = lScreenLayouter->GetAnimationDirection(surfaceId, (AnimationType::Enum)customAnimationType);
      }
      if ((lAnimationDirection == AnimationDirection::Undefined) ||
            (lAnimationDirection == AnimationDirection::Custom))
      {
         lAnimationDirection = lVisibility ? AnimationDirection::Show : AnimationDirection::Hide;
      }
      else if (lAnimationDirection != AnimationDirection::Custom)
      {
         AnimationDirection::Enum animationDirection = lVisibility ? AnimationDirection::Show : AnimationDirection::Hide;
         if (lAnimationDirection != animationDirection)
         {
            lAnimationDirection = animationDirection;
         }
         ETG_TRACE_ERR(("Notify: Wrong Custom animation type is requested, due to this it's tough to calculate the animation direction!!!"));
      }

      // Establish animation, dependant on XML configuration
      lAnimationEstablished = AnimationHandler::EstablishAnimation(lAnimationDirection,//lVisibility ? AnimationDirection::Show : AnimationDirection::Hide,
                              surfaceId,
                              lAnimationUserData,
                              (AnimationType::Enum)customAnimationType);
      // Set surface visibility immediately, if no animation was established
      if (!lAnimationEstablished)
      {
         ilmErrorTypes lIlmError = ILM_SUCCESS;
         lRc = IlmAccessor::SetSurfaceVisibility(surfaceId, (lVisibility ? ILM_TRUE : ILM_FALSE), false, lIlmError);
      }
   }

   // Notify surface state changes to clients immediately, if no animation was established
   if (!lAnimationEstablished)
   {
      SendNotification(surfaceId, lAnimationUserData);
   }
   return lRc;
}


// ------------------------------------------------------------------------
bool PluginActions::NotifyPopupVisibility(PopupState* popupState,
      ScreenBroker::SurfaceState::Enum surfaceState,
      bool updateSurfaceVisibility,
      bool sendNotification,
      bool notifyOnPopupStatus)
{
   bool lRc = (0 != popupState);
   if (lRc)
   {
      //notify popup current status to all clients
      if (notifyOnPopupStatus)
      {
         ActiveSurfaces lActiveSurfaces = ActiveSurfaces::GetInstance();
         UInt32 lActiveSurfaceId = lActiveSurfaces.GetTop(PluginActions::GetDisplayIdOfSurface(popupState->GetSurfaceId()));

         ServiceApi::NotifyCurrentStatus((surfaceState == SurfaceState::Mapped) ?
                                         ScreenBrokerProtocol::CurrentStatusRequestId::PopupInForeground :
                                         ScreenBrokerProtocol::CurrentStatusRequestId::PopupToBackground, //popup status
                                         lActiveSurfaceId, //active application surface id
                                         popupState->GetPopupScreenAreaId(), // popup layer id
                                         popupState->GetUserData());
      }

      // Check close on external touch
      ScreenBrokerProtocol::RequestId<ScreenBrokerProtocol::ShowPopupMode::Enum> lRequestId(popupState->GetRequestId());
      bool lCloseOnExternalTouch = lRequestId.Test(ScreenBrokerProtocol::ShowPopupMode::CloseOnExternalTouch);

      // Disable close on external touch for unmapped/queued popup
      if (lCloseOnExternalTouch && (
               (surfaceState == SurfaceState::Unmapped) || (surfaceState == SurfaceState::Queued)))
      {
         ETG_TRACE_USR1(("Notify: close on external touch disabled."));
         ServiceApi::NotifyCloseOnExternalTouch(false);
      }

      // Set & notify visiblity to application
      lRc = NotifyVisibility(popupState->GetSurfaceId(),
                             surfaceState,
                             popupState->GetUserData(),
                             AnimationType::None, //not pointing to any of the custom animation
                             updateSurfaceVisibility,
                             sendNotification);

      // Update visibility in popup state
      bool lVisibility = CalculateVisibility(surfaceState);
      if (lRc)
      {
         popupState->SetVisible(lVisibility);
      }

      // Removed queued popup state from focus list
      if (surfaceState == SurfaceState::Queued || surfaceState == SurfaceState::Unmapped)
      {
         FocusHandler::GetInstance().RemoveSurface(popupState->GetSurfaceId());
      }

      // Enable close on external touch for mapped popup
      if (lCloseOnExternalTouch && (surfaceState == SurfaceState::Mapped))
      {
         ETG_TRACE_USR1(("Notify: close on external touch enabled."));
         ServiceApi::NotifyCloseOnExternalTouch(true);
      }
   }
   return lRc;
}


// ------------------------------------------------------------------------
bool PluginActions::IsPopupActive(const PopupState& popupState)
{
   // popup is active if it is a system popup
   bool lRc = Modality::System == popupState.GetModality();

   // application popup is active if it belongs to an active application
   if ((!lRc) && (Modality::Application == popupState.GetModality()))
   {
      ActiveSurfaces lActiveSurfaces = ActiveSurfaces::GetInstance();
      UInt32 lSurfaceId = lActiveSurfaces.GetTop(PluginActions::GetDisplayIdOfSurface(popupState.GetSurfaceId()));
      UInt32 lActiveAppId = lActiveSurfaces.GetAppId(lSurfaceId);
      lRc = (popupState.GetAppId() == lActiveAppId);
   }

   return lRc;
}


// ------------------------------------------------------------------------
bool PluginActions::IsPopupFiltered(const PopupState& popupState)
{
   ActiveSurfaces& lActiveSurfaces = ActiveSurfaces::GetInstance();

   UInt32 lDisplayId = GetDisplayIdOfSurface(popupState.GetSurfaceId());
   UInt32 lSurfaceId = lActiveSurfaces.GetTop(lDisplayId);
   UInt32 lAppId = lActiveSurfaces.GetAppId(lSurfaceId);

   // Get instance of popup filter container and check if filter for an active application exists
   PopupFilter& lPopupFilter = PopupFilter::GetInstance();
   PopupPresentationArg lFilterArg;
   bool lRc = lPopupFilter.Get(lDisplayId, lAppId, lFilterArg);
   lRc = lRc &&  IsPopupActive(popupState);
   lRc = lRc && (lPopupFilter.IsDisableAll(lDisplayId, lAppId) || (popupState.GetPriority() > lFilterArg.Priority()));

   return lRc;
}


// ------------------------------------------------------------------------
void PluginActions::NotifyPopupFilter()
{
   ETG_TRACE_USR1(("%40s",
                   __FUNCTION__));

   ActivePopups& lActivePopups = ActivePopups::GetInstance();
   ActiveSurfaces& lActiveSurfaces = ActiveSurfaces::GetInstance();

   ScreenLayouter* lScreenLayouter = PLUGIN(ScreenLayouter);
   if (0 != lScreenLayouter)
   {
      IScreenLayouter::ScreenAreaList& lScreenAreaList = lScreenLayouter->GetScreenAreaList();
      for (IScreenLayouter::ScreenAreaList::const_iterator it = lScreenAreaList.begin();
            lScreenAreaList.end() != it;
            ++it)
      {
         // notify Blur popup status while application switch
         bool notifyOnPopupStatus = false;
         notifyOnPopupStatus = PluginActions::IsNotifyOnPopStatusForScreenAreaId(it->GetId());

         const ScreenArea& lScreenArea = (*it);
         if (lScreenArea.IsPopupArea())
         {
            UInt32 lScreenAreaId = lScreenArea.GetId();
            UInt32 lDisplayId = lScreenArea.GetDisplayId();
            UInt32 lAppId = lActiveSurfaces.GetAppId(lActiveSurfaces.GetTop(lDisplayId));

            // Get (unfiltered) top popup
            PopupState* lTopPopupState = lActivePopups.GetOverallTop(false, lScreenAreaId);

            // Check for filter in given application
            if (PopupFilter::GetInstance().Exists(lDisplayId, lAppId))
            {
               // Overwrite top popup with the filtered one
               lTopPopupState = lActivePopups.GetOverallTop(true, lScreenAreaId);
               // Notify filtered popups via traversing through active popups list
               PopupState* lPopupState = lActivePopups.GetOverallLast(lScreenAreaId);
               while (0 != lPopupState)
               {
                  // Process only relevant (active) popups
                  if (IsPopupFiltered(*lPopupState))
                  {
                     if (lPopupState->IsVisible())
                     {
                        ETG_TRACE_USR1(("%40s: Applying filter (appId: %u) to %u/0x%08x (prio: %u)",
                                        __FUNCTION__,
                                        lAppId,
                                        lPopupState->GetSurfaceId(),
                                        lPopupState->GetUserData(),
                                        lPopupState->GetPriority()));
                        // Unblur if Popup filter is set
                        // Blur if Popup filter is cleared
                        (void) NotifyPopupVisibility(lPopupState,
                                                     SurfaceState::Queued,
                                                     true,/* Default argument*/
                                                     true,/* Default argument*/
                                                     notifyOnPopupStatus);
                     }
                     else
                     {
                        FocusHandler::GetInstance().RemoveSurface(lPopupState->GetSurfaceId());
                     }
                  }
                  lPopupState = lActivePopups.GetOverallPrevious(lScreenAreaId);
               }
            }

            // Re-activate new top popup, if not already visible
            if ((0 != lTopPopupState) && (!lTopPopupState->IsVisible()))
            {
               (void) NotifyPopupVisibility(lTopPopupState,
                                            SurfaceState::Mapped,
                                            true,/* Default argument*/
                                            true,/* Default argument*/
                                            notifyOnPopupStatus);

               ETG_TRACE_USR1(("%40s: New top popup (%u/0x%08x) after filter manipulation",
                               __FUNCTION__,
                               lTopPopupState->GetSurfaceId(),
                               lTopPopupState->GetUserData()));

               FocusHandler& lFocusHandler = FocusHandler::GetInstance();
               // Set focus priority of popup
               lFocusHandler.SetKeyboardPriority(lTopPopupState->GetSurfaceId(), lTopPopupState->GetFocusPriority());
               lFocusHandler.SetPointerPriority(lTopPopupState->GetSurfaceId(), lTopPopupState->GetFocusPriority());

               // Add popup surface to focus handler
               lFocusHandler.PushBackSurface(lTopPopupState->GetSurfaceId());
            }
         }
      }
   }
}


// ------------------------------------------------------------------------
PopupState* PluginActions::GetTopVisiblePopup()
{
   ActivePopups& lActivePopups = ActivePopups::GetInstance();
   PopupState* lPopupState = lActivePopups.GetOverallLast();
   while ((0 != lPopupState) && (!lPopupState->IsVisible()))
   {
      lPopupState = lActivePopups.GetOverallPrevious();
   }
   return lPopupState;
}


// ------------------------------------------------------------------------
PopupState* PluginActions::GetTopVisiblePopupForDisplay(UInt32 surfaceId)
{
   ScreenLayouter* lScreenLayouter = PLUGIN(ScreenLayouter);
   PopupState* lPopupState = 0;
   ActivePopups& lActivePopups = ActivePopups::GetInstance();
   if (0 != lScreenLayouter)
   {
      UInt32 requestedDisplayId = lScreenLayouter->GetDisplayIdOfSurface(surfaceId);
      do
      {
         if (0 != lPopupState)
         {
            lPopupState = lActivePopups.GetOverallPrevious();
         }
         else
         {
            lPopupState = lActivePopups.GetOverallLast();
         }

         if ((0 != lPopupState) &&
               (requestedDisplayId == lScreenLayouter->GetDisplayIdOfSurface(lPopupState->GetSurfaceId())) &&
               (lPopupState->IsVisible()))
         {
            break;
         }
      }
      while (0 != lPopupState);
   }
   return lPopupState;
}


// ------------------------------------------------------------------------
void PluginActions::ProcessPopupsOnAppHide(UInt32 surfaceId)
{
   bool IsRefreshApplicationListIt = false;
   ETG_TRACE_USR1(("%40s: surfaceId=%u", __FUNCTION__, surfaceId));

   ScreenLayouter* lScreenLayouter = PLUGIN(ScreenLayouter);
   if (0 != lScreenLayouter)
   {
      IScreenLayouter::ScreenAreaList& lScreenAreaList = lScreenLayouter->GetScreenAreaList();
      for (IScreenLayouter::ScreenAreaList::const_iterator it = lScreenAreaList.begin();
            lScreenAreaList.end() != it;
            ++it)
      {
         // notify Blur popup status while application switch
         bool notifyOnPopupStatus = false;
         notifyOnPopupStatus = PluginActions::IsNotifyOnPopStatusForScreenAreaId(it->GetId());

         UInt32 lDisplayId = lScreenLayouter->GetDisplayIdOfScreenArea(it->GetId());
         if ((lDisplayId == GetDisplayIdOfSurface(surfaceId)) && (it->IsPopupArea()))
         {
            ActivePopups& lActivePopups = ActivePopups::GetInstance();

            PopupState* lAppPopupState = lActivePopups.GetAppFirst(it->GetId());
            while (0 != lAppPopupState)
            {
               IsRefreshApplicationListIt = false;
               // Check if current popup has requestId flag 'CloseAppOnLeave' set
               ScreenBrokerProtocol::RequestId<ScreenBrokerProtocol::ShowPopupMode::Enum> lRequestId(lAppPopupState->GetRequestId());
               if (lRequestId.Test(ScreenBrokerProtocol::ShowPopupMode::CloseOnAppLeave))
               {
                  ETG_TRACE_USR1(("Un%40s popup (%u/0x%08X)",
                                  lAppPopupState->IsVisible() ? "map" : "queue",
                                  lAppPopupState->GetSurfaceId(),
                                  lAppPopupState->GetUserData()));
                  //Use case : During app switch with center system popup visible and center app popup get closed (CloseOnAppLeave),
                  //the Blur/UnBlur notification not required.
                  //happened, in this case Unblur for center popup need not be sent due to
                  bool isPopupVisible = lAppPopupState->IsVisible() ? notifyOnPopupStatus : false;
                  (void) PluginActions::NotifyPopupVisibility(lAppPopupState,
                        (lAppPopupState->IsVisible() ?
                         SurfaceState::Unmapped :
                         SurfaceState::Unqueued),
                        true,/* Default argument*/
                        true,/* Default argument*/
                        isPopupVisible);
                  // Remove popup from popup manager
                  PopupManager::GetInstance().RemovePopup(*lAppPopupState);
                  // Remove popup state instance
                  (void) lActivePopups.Remove(lAppPopupState);
                  IsRefreshApplicationListIt = true;
               }
               // Retrieve next application popup
               lAppPopupState = lActivePopups.GetAppNext(it->GetId(), IsRefreshApplicationListIt);
            }

            // Queue application's visible popups
            PopupState* lTopPopupState = lActivePopups.GetAppFirst(it->GetId());

            while (0 != lTopPopupState)
            {
               if (lTopPopupState->IsVisible())
               {
                  ETG_TRACE_USR1(("Queue application popup (%u/0x%08X)",
                                  lTopPopupState->GetSurfaceId(),
                                  lTopPopupState->GetUserData()));
                  (void) PluginActions::NotifyPopupVisibility(lTopPopupState,
                        SurfaceState::Queued,
                        true,/* Default argument*/
                        true,/* Default argument*/
                        notifyOnPopupStatus);
               }
               lTopPopupState = lActivePopups.GetAppNext(it->GetId());
            }
         }
      }
   }
   else
   {
      ETG_TRACE_ERR(("ProcessPopupsOnAppHide() failed! ScreenLayouter instance not accessible!"));
   }
}


// ------------------------------------------------------------------------
void PluginActions::ProcessPopupsOnAppShow(UInt32 surfaceId)
{
   ETG_TRACE_USR1(("%40s: surfaceId=%u", __FUNCTION__, surfaceId));
   PluginActions::NotifyPopupFilter();
}


// ------------------------------------------------------------------------
PopupState* PluginActions::RemoveExpiredTopPopups(UInt32 screenArea)
{
   ActivePopups& lActivePopups = ActivePopups::GetInstance();
   // Retrieve top most popup of current active application not considering filter & visibility
   PopupState* lTopPopupState = lActivePopups.GetOverallTop(false, screenArea);
   // Remove all top popups, where timing values are not valid any more
   while ((0 != lTopPopupState) && !PluginActions::IsPopupTimingValid(*lTopPopupState))
   {
      PopupState* lExpiredTopPopupState = lTopPopupState;
      PopupManager::GetInstance().RemovePopup(*lExpiredTopPopupState);
      ETG_TRACE_USR1(("Removing popup %u/0x%08X due to invalid timing",
                      lExpiredTopPopupState->GetSurfaceId(),
                      lExpiredTopPopupState->GetUserData()));
      (void) PluginActions::NotifyPopupVisibility(lExpiredTopPopupState,
            lExpiredTopPopupState->IsVisible() ?
            SurfaceState::Unmapped :
            SurfaceState::Unqueued);
      // Do ActivePopups::Remove at last as this also removes the instance.
      lTopPopupState = lActivePopups.Remove(lExpiredTopPopupState);
   }
   return lActivePopups.GetOverallTop(true, screenArea);
}


// ------------------------------------------------------------------------
bool PluginActions::IsPopupTimingValid(const PopupState& popupState)
{
   // If popups presentation time is 0, no time-out handling is established
   if (0 == popupState.GetPresentationTime())
   {
      return true;
   }

   // First check general timing values
   bool lPopupTimingValid = (popupState.GetPresentationTime() >= popupState.GetMinimumPresentationTime());

   if (lPopupTimingValid)
   {
      // Retrieve time paased since popup states update time base
      // ------------------------------------------------------------------------
      UInt32 lPassed = Time::Passed(popupState.GetNextUpdateTimeBase());

      // Check validity period
      lPopupTimingValid = ((0 == popupState.GetValidityPeriod()) ||
                           (popupState.GetValidityPeriod() > lPassed));
      if (lPopupTimingValid)
      {
         // Check current presentation time (which is equal to next update period)
         lPopupTimingValid = (popupState.GetNextUpdatePeriod() > lPassed);
         if (lPopupTimingValid)
         {
            // Check minimum presentation time
            UInt32 lRemainingPresentationTime = (popupState.GetNextUpdatePeriod() - lPassed);
            lPopupTimingValid = (lRemainingPresentationTime >= popupState.GetMinimumPresentationTime());
            if (!lPopupTimingValid)
            {
               ETG_TRACE_USR1(("Popup (%u/0x%08X) minimum presentation time cannot be met (remainingPT: %ums, minPT: %ums)",
                               popupState.GetSurfaceId(),
                               popupState.GetUserData(),
                               lRemainingPresentationTime,
                               popupState.GetMinimumPresentationTime()));
            }
            else
            {
               ETG_TRACE_USR4(("Popup (%u/0x%08X) still valid (timePassed: %ums, remainingPT: %ums, pT: %ums, minPT: %ums, vP: %ums)",
                               popupState.GetSurfaceId(),
                               popupState.GetUserData(),
                               lPassed,
                               lRemainingPresentationTime,
                               popupState.GetPresentationTime(),
                               popupState.GetMinimumPresentationTime(),
                               popupState.GetValidityPeriod()));
            }
         }
         else
         {
            ETG_TRACE_USR1(("Popup (%u/0x%08X) current presentation time exceeded (timePassed: %ums, currentPT: %ums)",
                            popupState.GetSurfaceId(),
                            popupState.GetUserData(),
                            lPassed,
                            popupState.GetNextUpdatePeriod()));
         }
      }
      else
      {
         ETG_TRACE_USR1(("Popup (%u/0x%08X) validity period exceeded (timePassed: %ums, vP: %ums)",
                         popupState.GetSurfaceId(),
                         popupState.GetUserData(),
                         lPassed,
                         popupState.GetValidityPeriod()));
      }
   }
   else
   {
      ETG_TRACE_SYS(("Popup (%u/0x%08X) general timing not valid for presentation (pT: %ums, minPT: %ums, vP: %ums)",
                     popupState.GetSurfaceId(),
                     popupState.GetUserData(),
                     popupState.GetPresentationTime(),
                     popupState.GetMinimumPresentationTime(),
                     popupState.GetValidityPeriod()));
   }

   return lPopupTimingValid;
}


// ------------------------------------------------------------------------
std::string PluginActions::SurfaceStateString(SurfaceState::Enum surfaceState)
{
   switch (surfaceState)
   {
         CASE_ENUM2STRING(SurfaceState, Registered);
         CASE_ENUM2STRING(SurfaceState, Unmapped);
         CASE_ENUM2STRING(SurfaceState, Unqueued);
         CASE_ENUM2STRING(SurfaceState, Queued);
         CASE_ENUM2STRING(SurfaceState, Mapped);
      default:
         break;
   }
   return "InvalidEnumValue";
}


// ------------------------------------------------------------------------
std::string PluginActions::AnimationTypeString(AnimationType::Enum animationType)
{
   switch (animationType)
   {
         CASE_ENUM2STRING(AnimationType, None);
         CASE_ENUM2STRING(AnimationType, FadeIn);
         CASE_ENUM2STRING(AnimationType, FadeOut);
         CASE_ENUM2STRING(AnimationType, ZoomInFromTopLeft);
         CASE_ENUM2STRING(AnimationType, ZoomOutToTopLeft);
         CASE_ENUM2STRING(AnimationType, SlideInFromLeft);
         CASE_ENUM2STRING(AnimationType, SlideOutToLeft);
         CASE_ENUM2STRING(AnimationType, SlideInFromTop);
         CASE_ENUM2STRING(AnimationType, SlideOutToTop);
         CASE_ENUM2STRING(AnimationType, SlideInFromRight);
         CASE_ENUM2STRING(AnimationType, SlideOutToRight);
         CASE_ENUM2STRING(AnimationType, SlideInFromBottom);
         CASE_ENUM2STRING(AnimationType, SlideOutToBottom);
         CASE_ENUM2STRING(AnimationType, ZoomInFromTopRight);
         CASE_ENUM2STRING(AnimationType, ZoomOutToTopRight);
         CASE_ENUM2STRING(AnimationType, ZoomInFromBottomLeft);
         CASE_ENUM2STRING(AnimationType, ZoomOutToBottomLeft);
         CASE_ENUM2STRING(AnimationType, ZoomInFromBottomRight);
         CASE_ENUM2STRING(AnimationType, ZoomOutToBottomRight);
         CASE_ENUM2STRING(AnimationType, ZoomInFromCenter);
         CASE_ENUM2STRING(AnimationType, ZoomOutToCenter);
         CASE_ENUM2STRING(AnimationType, ZoomInFadeIn);
         CASE_ENUM2STRING(AnimationType, ZoomInFadeOut);
         CASE_ENUM2STRING(AnimationType, ZoomOutFadeIn);
         CASE_ENUM2STRING(AnimationType, ZoomOutFadeOut);
         CASE_ENUM2STRING(AnimationType, ShowImmediately);
         CASE_ENUM2STRING(AnimationType, HideImmediately);
      default:
         break;
   }
   return "InvalidEnumValue";
}


// ------------------------------------------------------------------------
bool PluginActions::CalculateVisibility(SurfaceState::Enum surfaceState)
{
   bool lVisibility = (SurfaceState::Mapped == surfaceState);
   // Note: If queued popups shall be made visible then SurfaceState::Queued has to be added
   // for calculating visibility. Additionally (at least) changes in ProcessPopupsOnAppShow() resp.
   // ProcessPopupsOnAppHide() has to be done.
   return lVisibility;
}


// ------------------------------------------------------------------------
UInt32 PluginActions::GetTopKeyEventSurface(UInt32 displayId,
      bool ignorePermanentSurfaces,
      bool includePopupSurfaces)
{
   UInt32 lTopKeySurfaceId = 0;

   ScreenLayouter* lScreenLayouter = PLUGIN(ScreenLayouter);
   if (0 != lScreenLayouter)
   {
      // First find top most surface
      ActiveSurfaces& lActiveSurfaces = ActiveSurfaces::GetInstance();
      UInt32 lSurfaceId = lActiveSurfaces.GetFirst(displayId, ignorePermanentSurfaces);
      while (0 != lSurfaceId)
      {
         if (0 != (lScreenLayouter->GetEvents(lSurfaceId) & static_cast<Int32>(Event::Key)))
         {
            lTopKeySurfaceId = lSurfaceId;
         }
         lSurfaceId = lActiveSurfaces.GetNext(displayId, ignorePermanentSurfaces);
      }

      if (includePopupSurfaces)
      {
         // Find top most popup based on top application
         ActivePopups& lActivePopups = ActivePopups::GetInstance();
         PopupState* lPopupState = lActivePopups.GetOverallFirst();
         while (0 != lPopupState)
         {
            if (0 != (lScreenLayouter->GetEvents(lPopupState->GetSurfaceId()) &
                      static_cast<Int32>(Event::Key)))
            {
               lTopKeySurfaceId = lPopupState->GetSurfaceId();
            }
            lPopupState = lActivePopups.GetOverallNext();
         }
      }
   }
   else
   {
      ETG_TRACE_ERR(("GetTopKeyEventSurface() failed! ScreenLayouter instance not accessible!"));
   }

   return lTopKeySurfaceId;
}


// ------------------------------------------------------------------------
void PluginActions::SendNotification(UInt32 surfaceId,
                                     const AnimationUserData& animationUserData)
{
   // Notify surface state changes to clients
   if (animationUserData.sendNotification)
   {
      ETG_TRACE_USR1(("%40s surface %u/0x%08X (surface state: %40s)",
                      (animationUserData.visibility ? "Show" : "Hide"),
                      surfaceId,
                      animationUserData.userData,
                      SurfaceStateString(animationUserData.surfaceState).c_str()));
      ServiceApi::NotifySurfaceStateChange(animationUserData.userData,
                                           surfaceId,
                                           animationUserData.surfaceState);
   }
}


// ------------------------------------------------------------------------
void PluginActions::SendLayerAnimationState(UInt32 userData,
      UInt32 layerId,
      ScreenBroker::LayerState::Enum animationState)
{
   // Notify animation state changes to clients
   ETG_TRACE_USR1(("%40s animation of Layer %u/0x%08X",
                   (ScreenBroker::LayerState::AnimationStarted == animationState ? "Start" : "Stop"),
                   layerId,
                   userData));
   ServiceApi::NotifyLayerStateChanged(userData,
                                       layerId,
                                       static_cast<LayerState::Enum>(animationState));
}


// ------------------------------------------------------------------------
void PluginActions::SendAnimationState(UInt32 userData,
                                       UInt32 surfaceId,
                                       ScreenBroker::SurfaceState::Enum animationState)
{
   // Notify animation state changes to clients
   ETG_TRACE_USR1(("%40s animation of surface %u/0x%08X",
                   (ScreenBroker::SurfaceState::AnimationStarted == animationState ? "Start" : "Stop"),
                   surfaceId,
                   userData));
   ServiceApi::NotifySurfaceStateChange(userData,
                                        surfaceId,
                                        static_cast<SurfaceState::Enum>(animationState));
}


// ------------------------------------------------------------------------
void PluginActions::CheckExplicitCommit(UInt32 surfaceId)
{
   ScreenLayouter* lScreenLayouter = PLUGIN(ScreenLayouter);
   if (0 != lScreenLayouter)
   {
      if (!(lScreenLayouter->GetExplicitCommit(surfaceId)))
      {
         ilmErrorTypes lIlmError = ILM_SUCCESS;
         (void) IlmAccessor::Commit(lIlmError);
      }
      else
      {
         ETG_TRACE_USR1(("Explicit commit of surface %u required", surfaceId));
      }
   }
   else
   {
      ETG_TRACE_ERR(("CheckExplicitCommit() failed! ScreenLayouter instance not accessible!"));
   }
}


// ------------------------------------------------------------------------
UInt32 PluginActions::GetDisplayIdOfSurface(UInt32 surfaceId)
{
   UInt32 lDisplayId = 0;
   ScreenLayouter* lScreenLayouter = PLUGIN(ScreenLayouter);
   if (0 != lScreenLayouter)
   {
      lDisplayId = lScreenLayouter->GetDisplayIdOfSurface(surfaceId);
   }
   else
   {
      ETG_TRACE_ERR(("GetDisplayIdOfSurface() failed! ScreenLayouter instance not accessible!"));
   }
   return lDisplayId;
}


// ------------------------------------------------------------------------
UInt32 PluginActions::GetDisplayIdOfScreenArea(UInt32 screenArea)
{
   UInt32 lDisplayId = 0;
   ScreenLayouter* lScreenLayouter = PLUGIN(ScreenLayouter);
   if (0 != lScreenLayouter)
   {
      lDisplayId = lScreenLayouter->GetDisplayIdOfScreenArea(screenArea);
   }
   else
   {
      ETG_TRACE_ERR(("GetDisplayIdOfScreenArea() failed! ScreenLayouter instance not accessible!"));
   }
   return lDisplayId;
}


// ------------------------------------------------------------------------
PluginActions::ApplicationList PluginActions::GetActiveApplications(UInt32 screenArea)
{
   ApplicationList lApplicationList;
   //Extract display id using ScreenLayouter
   IScreenLayouter* lScreenLayouter = ServiceApi::ScreenLayouter();
   if (0 != lScreenLayouter)
   {
      IScreenLayouter::DisplayIdList lDisplayIdList = lScreenLayouter->GetDisplayIdList();
      for (IScreenLayouter::DisplayIdList::iterator lIt = lDisplayIdList.begin(); lIt != lDisplayIdList.end(); ++lIt)
      {
         if ((screenArea == 0) || (*lIt == GetDisplayIdOfScreenArea(screenArea)))
         {
            ActiveSurfaces lActiveSurfaces = ActiveSurfaces::GetInstance();
            UInt32 lSurfaceId = lActiveSurfaces.GetTop(*lIt);
            lApplicationList.insert(lActiveSurfaces.GetAppId(lSurfaceId));
         }
      }
   }
   return lApplicationList;
}


// ------------------------------------------------------------------------
bool PluginActions::IsNotifyOnPopStatusForScreenAreaId(UInt32 screenAreaId)
{
   bool isNotifyOnPopupStatus = false;

   // Retrieve instance of screen layouter
   IScreenLayouter* lScreenLayouter = PLUGIN(ScreenLayouter);

   if (0 != lScreenLayouter)
   {
      const ScreenArea* lScreenArea = lScreenLayouter->GetScreenArea(screenAreaId);

      if ((NULL != lScreenArea) && (lScreenArea->IsNotifyOnPopupStatus() == true))
      {
         isNotifyOnPopupStatus = true;
      }
   }
   else
   {
      ETG_TRACE_ERR(("IsNotifyOnPopStatusForScreenAreaId() failed! ScreenLayouter instance not accessible!"));
   }
   return isNotifyOnPopupStatus;
}


}
