/* ***************************************************************************************
* FILE:          IlmAccessor.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  IlmAccessor.cpp is part of HMI-Base ScreenBrokerPlugins
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

// =============================================================================
//lint -emacro(506, TEST_ACCEPT_INPUT_EVENTS)  "Constant value Boolean"
//Pointer cast is required due to wayland interface.
//lint -efunc(740, ScreenBroker::IlmAccessor::CreateSurface) " Unusual pointer cast (incompatible indirect types)
// Using the "do { ... } while(0) construct within macros is a common practice in order
// to make the use of macros safe within conditions.
//lint -emacro({717}, SAFE_FREE)    "do ... while(0);"
//lint -emacro({717}, CHECK_ILM_ACCESS)    "do ... while(0);"
//lint -emacro({717}, TEST_ACCEPT_INPUT_EVENTS)    "do ... while(0);"
// =============================================================================

#include "IlmAccessor.h"
#include "WaylandContext.h"
#include <ScreenBroker/Service/ServiceApi.h>

#include <stdlib.h>
#include <map>
#include <stdio.h>

#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL
#include <ilm/ilm_input.h>
#endif

///
#define SAFE_FREE(array)    \
    do {                    \
        if (0 != array) {   \
            free(array);    \
            array = 0;      \
        }                   \
    } while(0)

#define CHECK_ILM_ACCESS(rc)                \
    do {                                    \
        if (!ServiceApi::HasIlmAccess()) {  \
            return rc;                      \
        }                                   \
    } while(0)
#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_PLUGINS
#include "trcGenProj/Header/IlmAccessor.cpp.trc.h"
#endif


namespace ScreenBroker {
// SCREENBROKER_LOG_SET_REALM(LogRealm::Actions);

// ------------------------------------------------------------------------
typedef std::map<UInt32, ilmInputDevice> SurfaceInputAcceptanceMap;
SurfaceInputAcceptanceMap gSurfaceInputAcceptanceMap;

// ------------------------------------------------------------------------
bool IlmAccessor::Commit(ilmErrorTypes& ilmError)
{
   CHECK_ILM_ACCESS(true);
   // Commit changes to ILM
   ilmError = ilm_commitChanges();
   bool lRc = (ILM_SUCCESS == ilmError);
   // Log transaction info
   if (lRc)
   {
      ETG_TRACE_USR4(("Commit to layer manager: '%40s'", ILM_ERROR_STRING(ilmError)));
   }
   else
   {
      ETG_TRACE_ERR(("Commit to layer manager: '%40s'", ILM_ERROR_STRING(ilmError)));
   }
   return lRc;
}


// ------------------------------------------------------------------------
bool IlmAccessor::SetLayerRenderOrder(UInt32 displayId,
                                      UInt32* layerArray,
                                      UInt32 layerCount,
                                      bool commit,
                                      ilmErrorTypes& ilmError)
{
   CHECK_ILM_ACCESS(true);
   if ((0 == layerArray) || (0 == layerCount))
   {
      ETG_TRACE_SYS(("Display %u has no layers to set render order!", displayId));
      return false;
   }

   // Set render order of layers for given display
   ilmError = ilm_displaySetRenderOrder(displayId, layerArray, layerCount);
   bool lRc = (ILM_SUCCESS == ilmError);

   // Commit changes to ILM
   if (commit)
   {
      lRc = lRc && Commit(ilmError);
   }

   // Log transaction info
   if (lRc)
   {
      ETG_TRACE_USR1(("Set of layer render order for display %u %40s",
                      displayId,
                      (commit ? "succeeded" : "requested")));
   }
   else
   {
      ETG_TRACE_ERR(("Set of layer render order for display %u failed!",
                     displayId));
   }

   return lRc;
}


// ------------------------------------------------------------------------
bool IlmAccessor::CreateSurface(wl_surface*& surface,
                                wl_egl_window* const& eglWindow,
                                WaylandContext& context,
                                UInt32 surfaceId,
                                t_ilm_int width,
                                t_ilm_int height,
                                bool commit,
                                ilmErrorTypes& ilmError)
{
#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL
   (void)surface;
   (void)eglWindow;
   (void)context;
   (void)surfaceId;
   (void)width;
   (void)height;
   (void)commit;
   (void)ilmError;
   ETG_TRACE_SYS(("Creation of surface should be handled in Client for IVI-SHELL!"));
   return false;
#else

   CHECK_ILM_ACCESS(true);
   // Create wayland surface
   ETG_TRACE_USR1(("Creating wayland surface"));
   surface = wl_compositor_create_surface(context.GetCompositor());
   bool lRc = (0 != surface);
   if (lRc)
   {
      // Egl window not needed for not
//            ETG_TRACE_USR1(("Creating egl surface (%dx%d) via wayland egl", width, height));
//            eglWindow = wl_egl_window_create(surface, width, height);
      (void) eglWindow;

      // Register surface to layermanager
      uint32_t lNativeHandle;
#if ((WAYLAND_VERSION_MAJOR >= 1) && (WAYLAND_VERSION_MINOR >= 11))
      struct wl_proxy* lProxy = (struct wl_proxy*) surface;
      lNativeHandle = (context.GetConnectionId() << 16) | (uint32_t) wl_proxy_get_id(lProxy);
#else
      struct wl_object* lObject = (struct wl_object*) surface;
      lNativeHandle = (context.GetConnectionId() << 16) | (uint32_t) lObject->id;
#endif


      ilmError = ilm_surfaceCreate(static_cast<t_ilm_nativehandle>(lNativeHandle),
                                   width,
                                   height,
                                   ILM_PIXELFORMAT_RGBA_8888,
                                   &surfaceId);

      lRc = (ILM_SUCCESS == ilmError);

      // Commit changes to ILM
      if (commit)
      {
         lRc = lRc && Commit(ilmError);
      }
      // Log transaction info
      if (lRc)
      {
         ETG_TRACE_USR1(("Creation of surface %u with dimension (%dx%d) %40s",
                         surfaceId,
                         width,
                         height,
                         (commit ? "succeeded" : "requested")));
      }
      else
      {
         ETG_TRACE_ERR(("Creation of surface %u failed!",
                        surfaceId));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Could not acquire wayland surface: Failed to create wl_surface!"));
   }

   return lRc;
#endif
}


// ------------------------------------------------------------------------
// Adds given surface to given layer, if not already done
bool IlmAccessor::AddSurfaceToLayer(UInt32 layerId,
                                    UInt32 surfaceId,
                                    bool commit,
                                    ilmErrorTypes& ilmError)
{
   CHECK_ILM_ACCESS(true);
   t_ilm_int lNoOfSurfaceIds = 0;
   t_ilm_surface* lSurfaceIds = 0;
   ilmError = ilm_getSurfaceIDsOnLayer(layerId, &lNoOfSurfaceIds, &lSurfaceIds);
   bool lRc = (ILM_SUCCESS == ilmError);

   // Search if surface is already mapped to layer
   Int i = 0;
   while (lRc && (i < lNoOfSurfaceIds) && (lSurfaceIds[i] != surfaceId))
   {
      ++i;
   }

   // Map surface to layer (if layer not found)
   if (i == lNoOfSurfaceIds)
   {
      if (lRc)
      {
         ilmError = ilm_layerAddSurface(layerId, surfaceId);
         lRc = (ILM_SUCCESS == ilmError);
      }
      // Commit changes to ILM
      if (commit)
      {
         lRc = lRc && Commit(ilmError);
      }
      // Log transaction info
      if (lRc)
      {
         ETG_TRACE_USR1(("Add Surface %u to layer %u %40s",
                         surfaceId,
                         layerId,
                         (commit ? "succeeded" : "requested")));
      }
      else
      {
         ETG_TRACE_ERR(("Add Surface %u to layer %u failed!",
                        surfaceId,
                        layerId));
      }
   }

   SAFE_FREE(lSurfaceIds);
   return lRc;
}


// ------------------------------------------------------------------------
bool IlmAccessor::SetSurfaceRenderOrder(UInt32 layerId,
                                        t_ilm_layer renderOrder[],
                                        t_ilm_int count,
                                        bool commit,
                                        ilmErrorTypes& ilmError)
{
   CHECK_ILM_ACCESS(true);
   // Set surface source rectangle
   ilmError = ilm_layerSetRenderOrder(layerId, renderOrder, count);
   bool lRc = (ILM_SUCCESS == ilmError);
   // Commit changes to ILM
   if (commit)
   {
      lRc = lRc && Commit(ilmError);
   }
   // Log transaction info
   if (lRc)
   {
      ETG_TRACE_USR4(("Update surface render order for layer surface %u %40s",
                      layerId,
                      (commit ? "succeeded" : "requested")));
   }
   else
   {
      ETG_TRACE_ERR(("Update surface render order for layer surface %u failed!",
                     layerId));
   }
   return lRc;
}


// ------------------------------------------------------------------------
bool IlmAccessor::SetSurfaceSourceRectangle(UInt32 surfaceId,
      t_ilm_int posX,
      t_ilm_int posY,
      t_ilm_int width,
      t_ilm_int height,
      bool commit,
      ilmErrorTypes& ilmError)
{
   CHECK_ILM_ACCESS(true);
   // Set surface source rectangle
   ilmError = ilm_surfaceSetSourceRectangle(surfaceId, posX, posY, width, height);
   bool lRc = (ILM_SUCCESS == ilmError);
   // Commit changes to ILM
   if (commit)
   {
      lRc = lRc && Commit(ilmError);
   }
   // Log transaction info
   if (lRc)
   {
      ETG_TRACE_USR4(("Set source rectangle for surface %u to position (%u, %u) with %ux%u %40s",
                      surfaceId,
                      posX,
                      posY,
                      width,
                      height,
                      (commit ? "succeeded" : "requested")));
   }
   else
   {
      ETG_TRACE_ERR(("Set source rectangle request for surface %u to position (%u, %u) with %ux%u failed!",
                     surfaceId,
                     posX,
                     posY,
                     width,
                     height));
   }
   return lRc;
}


// ------------------------------------------------------------------------
bool IlmAccessor::GetSurfaceSourceRectangle(UInt32 surfaceId,
      t_ilm_int& posX,
      t_ilm_int& posY,
      t_ilm_int& width,
      t_ilm_int& height,
      ilmErrorTypes& ilmError)
{
   CHECK_ILM_ACCESS(true);
   ilmSurfaceProperties properties;
   ilmError = ilm_getPropertiesOfSurface(surfaceId, &properties);
   if (ilmError != ILM_SUCCESS)
   {
      return false;
   }

   posX = static_cast<t_ilm_int>(properties.sourceX);
   posY = static_cast<t_ilm_int>(properties.sourceY);
   width = static_cast<t_ilm_int>(properties.sourceWidth);
   height = static_cast<t_ilm_int>(properties.sourceHeight);

   return true;
}


// ------------------------------------------------------------------------
bool IlmAccessor::SetSurfaceDestinationRectangle(UInt32 surfaceId,
      t_ilm_int posX,
      t_ilm_int posY,
      t_ilm_int width,
      t_ilm_int height,
      bool commit,
      ilmErrorTypes& ilmError)
{
   CHECK_ILM_ACCESS(true);
   // Set surface destination rectangle
   ilmError = ilm_surfaceSetDestinationRectangle(surfaceId, posX, posY, width, height);
   bool lRc = (ILM_SUCCESS == ilmError);
   // Commit changes to ILM
   if (commit)
   {
      lRc = lRc && Commit(ilmError);
   }
   // Log transaction info
   if (lRc)
   {
      ETG_TRACE_USR4(("Set destination rectangle for surface %u to position (%d, %d) with %dx%d %40s",
                      surfaceId,
                      posX,
                      posY,
                      width,
                      height,
                      (commit ? "succeeded" : "requested")));
   }
   else
   {
      ETG_TRACE_ERR(("Set destination rectangle request for surface %u to position (%d, %d) with %dx%d failed!",
                     surfaceId,
                     posX,
                     posY,
                     width,
                     height));
   }
   return lRc;
}


// ------------------------------------------------------------------------
bool IlmAccessor::GetSurfaceDestinationRectangle(UInt32 surfaceId,
      t_ilm_int& posX,
      t_ilm_int& posY,
      t_ilm_int& width,
      t_ilm_int& height,
      ilmErrorTypes& ilmError)
{
   CHECK_ILM_ACCESS(true);
   ilmSurfaceProperties properties;
   ilmError = ilm_getPropertiesOfSurface(surfaceId, &properties);
   if (ilmError != ILM_SUCCESS)
   {
      return false;
   }

   posX = static_cast<t_ilm_int>(properties.destX);
   posY = static_cast<t_ilm_int>(properties.destY);
   width = static_cast<t_ilm_int>(properties.destWidth);
   height = static_cast<t_ilm_int>(properties.destHeight);

   return true;
}


// ------------------------------------------------------------------------
bool IlmAccessor::SetSurfaceRectangle(UInt32 surfaceId,
                                      t_ilm_int posX,
                                      t_ilm_int posY,
                                      t_ilm_int width,
                                      t_ilm_int height,
                                      bool commit,
                                      ilmErrorTypes& ilmError)
{
   CHECK_ILM_ACCESS(true);
   t_ilm_int sourceX;
   t_ilm_int sourceY;
   t_ilm_int sourceWidth;
   t_ilm_int sourceHeight;
   t_ilm_int destX;
   t_ilm_int destY;
   t_ilm_int destWidth;
   t_ilm_int destHeight;

   if (posX < 0)
   {
      sourceX = -posX;
      destX = 0;
      sourceWidth = width - sourceX;
      destWidth = width - sourceX;
   }
   else
   {
      sourceX = 0;
      destX = posX;
      sourceWidth = width - destX;
      destWidth = width - destX;
   }

   if (posY < 0)
   {
      sourceY = -posY;
      destY = 0;
      sourceHeight = height - sourceY;
      destHeight = height - sourceY;
   }
   else
   {
      sourceY = 0;
      destY = posY;
      sourceHeight = height - destY;
      destHeight = height - destY;
   }

   ETG_TRACE_USR4(("SetSurfaceRectangle ID:%u source (%d,%d)/%dx%d dest (%d,%d)/%dx%d",
                   surfaceId,
                   sourceX, sourceY,
                   sourceWidth, sourceHeight,
                   destX, destY,
                   destWidth, destHeight));

   bool rc = IlmAccessor::SetSurfaceSourceRectangle(surfaceId,
             sourceX, sourceY,
             sourceWidth, sourceHeight,
             false, ilmError);
   rc = rc && IlmAccessor::SetSurfaceDestinationRectangle(surfaceId,
         destX, destY,
         destWidth, destHeight,
         commit, ilmError);
   return rc;
}


// ------------------------------------------------------------------------
bool IlmAccessor::SetSurfacePosition(UInt32 surfaceId,
                                     t_ilm_uint posX,
                                     t_ilm_uint posY,
                                     bool commit,
                                     ilmErrorTypes& ilmError)
{
   CHECK_ILM_ACCESS(true);
   t_ilm_uint lPosition[2] = { posX, posY };
#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL
   t_ilm_int x, y, width, height;
   GetSurfaceDestinationRectangle(surfaceId, x, y, width, height, ilmError);
   if (ilmError == ILM_SUCCESS)
   {
      ilmError = ilm_surfaceSetDestinationRectangle(surfaceId, posX, posY, width, height);
   }
#else
   // Set surface position
   ilmError = ilm_surfaceSetPosition(surfaceId, lPosition);
#endif
   bool lRc = (ILM_SUCCESS == ilmError);
   // Commit changes to ILM
   if (commit)
   {
      lRc = lRc && Commit(ilmError);
   }
   // Log transaction info
   if (lRc)
   {
      ETG_TRACE_USR4(("Move surface %u to position (%u, %u) %40s",
                      surfaceId,
                      lPosition[0],
                      lPosition[1],
                      (commit ? "succeeded" : "requested")));
   }
   else
   {
      ETG_TRACE_ERR(("Move request for surface %u to position (%u, %u) failed!",
                     surfaceId,
                     lPosition[0],
                     lPosition[1]));
   }
   return lRc;
}


// ------------------------------------------------------------------------
bool IlmAccessor::GetSurfacePosition(UInt32 surfaceId,
                                     t_ilm_uint& posX,
                                     t_ilm_uint& posY,
                                     ilmErrorTypes& ilmError)
{
   CHECK_ILM_ACCESS(true);
#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL
   t_ilm_int x, y, w, h;
   GetSurfaceDestinationRectangle(surfaceId, x, y, w, h, ilmError);

   if (ILM_SUCCESS == ilmError)
   {
      posX = x;
      posY = y;
   }
   return (ILM_SUCCESS == ilmError);
#else
   t_ilm_uint position[2];
   ilmError = ilm_surfaceGetPosition(surfaceId, position);
   bool rc = (ILM_SUCCESS == ilmError);
   if (rc)
   {
      posX = position[0];
      posY = position[1];
   }
   return rc;
#endif
}


// ------------------------------------------------------------------------
bool IlmAccessor::SetSurfaceDimension(UInt32 surfaceId,
                                      t_ilm_uint width,
                                      t_ilm_uint height,
                                      bool commit,
                                      ilmErrorTypes& ilmError)
{
   CHECK_ILM_ACCESS(true);
   t_ilm_uint lDimension[2] = { width, height };
#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL
   t_ilm_int x, y, w, h;
   GetSurfaceDestinationRectangle(surfaceId, x, y, w, h, ilmError);
   if (ilmError == ILM_SUCCESS)
   {
      ilmError = ilm_surfaceSetDestinationRectangle(surfaceId, x, y, width, height);
   }
#else
   // Set surface dimension
   ilmError = ilm_surfaceSetDimension(surfaceId, lDimension);
#endif
   bool lRc = (ILM_SUCCESS == ilmError);
   // Commit changes to ILM
   if (commit)
   {
      lRc = lRc && Commit(ilmError);
   }
   // Log transaction info
   if (lRc)
   {
      ETG_TRACE_USR4(("Set dimension of surface %u to %ux%u %40s",
                      surfaceId,
                      lDimension[0],
                      lDimension[1],
                      (commit ? "succeeded" : "requested")));
   }
   else
   {
      ETG_TRACE_ERR(("Set dimension request for surface %u to %ux%u failed!",
                     surfaceId,
                     lDimension[0],
                     lDimension[1]));
   }
   return lRc;
}


// ------------------------------------------------------------------------
bool IlmAccessor::GetSurfaceDimension(UInt32 surfaceId,
                                      t_ilm_uint& width,
                                      t_ilm_uint& height,
                                      ilmErrorTypes& ilmError)
{
   CHECK_ILM_ACCESS(true);
#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL
   t_ilm_int x, y, w, h;
   GetSurfaceDestinationRectangle(surfaceId, x, y, w, h, ilmError);
   if (ilmError == ILM_SUCCESS)
   {
      width = w;
      height = h;
   }
   return (ILM_SUCCESS == ilmError);
#else
   t_ilm_uint dimension[2];
   ilmError = ilm_surfaceGetDimension(surfaceId, dimension);
   bool rc = (ILM_SUCCESS == ilmError);
   if (rc)
   {
      width  = dimension[0];
      height = dimension[1];
   }
   return rc;
#endif
}


// ------------------------------------------------------------------------
bool IlmAccessor::GetSurfaceOpacity(UInt32 surfaceId,
                                    t_ilm_float& opacity,
                                    ilmErrorTypes& ilmError)
{
   CHECK_ILM_ACCESS(true);
   ilmError = ilm_surfaceGetOpacity(surfaceId, &opacity);
   bool rc = (ilmError == ILM_SUCCESS);

   if (!rc)
   {
      ETG_TRACE_ERR(("Unable to receive opacity for surface %u",
                     surfaceId));
   }

   return rc;
}


// ------------------------------------------------------------------------
bool IlmAccessor::SetSurfaceOpacity(UInt32 surfaceId,
                                    t_ilm_float opacity,
                                    bool commit,
                                    ilmErrorTypes& ilmError)
{
   CHECK_ILM_ACCESS(true);
   // Set surface opacity
   ilmError = ilm_surfaceSetOpacity(surfaceId, opacity);
   bool lRc = (ILM_SUCCESS == ilmError);
   // Commit changes to ILM
   if (commit)
   {
      lRc = lRc && Commit(ilmError);
   }
   // Log transaction info
   if (lRc)
   {
      ETG_TRACE_USR4(("Opacity %40s for surface %u (%.3f %%)",
                      (commit ? "set" : "requested"),
                      surfaceId,
                      (float)opacity));
   }
   else
   {
      ETG_TRACE_ERR(("Opacity request for surface %u (%.3f %%) failed!",
                     surfaceId,
                     (float)opacity));
   }
   return lRc;
}


// ------------------------------------------------------------------------
bool IlmAccessor::SetSurfaceVisibility(UInt32 surfaceId,
                                       t_ilm_bool visible,
                                       bool commit,
                                       ilmErrorTypes& ilmError)
{
   CHECK_ILM_ACCESS(true);
   // Set surface visibility
   ilmError = ilm_surfaceSetVisibility(surfaceId, visible);
   bool lRc = (ILM_SUCCESS == ilmError);
   // Commit changes to ILM
   if (commit)
   {
      lRc = lRc && Commit(ilmError);
   }
   // Log transaction info
   if (lRc)
   {
      ETG_TRACE_USR1(("Visibility %40s for surface %u (%40s)",
                      (commit ? "set" : "requested"),
                      surfaceId,
                      (visible ? "true" : "false")));
   }
   else
   {
      ETG_TRACE_ERR(("Visibility request for surface %u (%40s) failed!",
                     surfaceId,
                     (visible ? "true" : "false")));
   }
   return lRc;
}


// ------------------------------------------------------------------------
bool IlmAccessor::GetSurfaceVisibility(UInt32 surfaceId,
                                       t_ilm_bool& visible,
                                       ilmErrorTypes& ilmError)
{
   CHECK_ILM_ACCESS(true);

   ilmError = ilm_surfaceGetVisibility(surfaceId, &visible);
   bool lRc = (ILM_SUCCESS == ilmError);

   return lRc;
}


// ------------------------------------------------------------------------
bool IlmAccessor::SwitchSurfaceVisibility(UInt32 oldSurfaceId,
      t_ilm_bool oldVisible,
      UInt32 newSurfaceId,
      t_ilm_bool newVisible,
      bool commit,
      ilmErrorTypes& ilmError)
{
   CHECK_ILM_ACCESS(true);
   // Set surface visibility of oldSurfaceId
   ilmError = ilm_surfaceSetVisibility(oldSurfaceId, oldVisible);
   bool lRc = (ILM_SUCCESS == ilmError);
   // Set surface visibility of NewSurfaceId
   if (lRc)
   {
      ilmError = ilm_surfaceSetVisibility(newSurfaceId, newVisible);
      lRc = (ILM_SUCCESS == ilmError);
   }
   // Commit changes to ILM
   if (commit)
   {
      lRc = lRc && Commit(ilmError);
   }
   // Log transaction info
   if (lRc)
   {
      ETG_TRACE_USR1(("Visibility switch for surface %u (%40s) and surface %u (%40s) %40s",
                      oldSurfaceId, (oldVisible ? "true" : "false"),
                      newSurfaceId, (newVisible ? "true" : "false"),
                      (commit ? "succeeded" : "requested")));
   }
   else
   {
      ETG_TRACE_ERR(("Visibility switch request for surface %u (%40s) and surface %u (%40s) failed!",
                     oldSurfaceId, (oldVisible ? "true" : "false"),
                     newSurfaceId, (newVisible ? "true" : "false")));
   }
   return lRc;
}


// ------------------------------------------------------------------------
bool IlmAccessor::ScaleSurfaceToLayer(UInt32 layerId,
                                      UInt32 surfaceId,
                                      bool commit,
                                      ilmErrorTypes& ilmError)
{
   CHECK_ILM_ACCESS(true);
   // Get layer dimension
   t_ilm_uint lDimension[2] = { 0, 0 };
#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL
   ilmLayerProperties properties;
   ilmError = ilm_getPropertiesOfLayer(layerId, &properties);
   if (ilmError == ILM_SUCCESS)
   {
      lDimension[0] = properties.destWidth;
      lDimension[1] = properties.destHeight;
   }
#else
   ilmError = ilm_layerGetDimension(layerId, lDimension);
#endif
   bool lRc = (ILM_SUCCESS == ilmError);

   // Update surface dimension
   if (lRc)
   {
#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL
      SetSurfaceDimension(surfaceId, lDimension[0], lDimension[1], false, ilmError);
#else
      ilmError = ilm_surfaceSetDimension(surfaceId, lDimension);
#endif
      lRc = (ILM_SUCCESS == ilmError);
   }

   // Commit changes to ILM
   if (commit)
   {
      lRc = lRc && Commit(ilmError);
   }
   // Log transaction info
   if (lRc)
   {
      ETG_TRACE_USR4(("Scale dimension of surface %u to %ux%u (layer %u) %40s",
                      surfaceId,
                      lDimension[0],
                      lDimension[1],
                      layerId,
                      (commit ? "succeeded" : "requested")));
   }
   else
   {
      ETG_TRACE_ERR(("Scale dimension request for surface %u to %ux%u (layer %u) failed!",
                     surfaceId,
                     lDimension[0],
                     lDimension[1],
                     layerId));
   }
   return lRc;
}


// ------------------------------------------------------------------------
bool IlmAccessor::SetKeyboardFocus(UInt32 surfaceId,
                                   bool commit,
                                   ilmErrorTypes& ilmError)
{
   CHECK_ILM_ACCESS(true);
   // Set keyboard focus to given surface

   t_ilm_surface lSurfaceIDs[1] = {surfaceId};
   ilmError = ilm_setInputFocus(lSurfaceIDs, 1, ILM_INPUT_DEVICE_KEYBOARD, ILM_TRUE);

   bool lRc = (ILM_SUCCESS == ilmError);
   // Commit changes to ILM
   if (commit)
   {
      lRc = lRc && Commit(ilmError);
   }
   // Log transaction info
   if (lRc)
   {
      ETG_TRACE_USR4(("Keyboard focus %40s to surface %u",
                      (commit ? "set" : "requested"),
                      surfaceId));
   }
   else
   {
      ETG_TRACE_ERR(("Keyboard focus request for surface %u failed!", surfaceId));
   }
   return lRc;
}


// ------------------------------------------------------------------------
UInt32 IlmAccessor::GetKeyboardFocus()
{
   CHECK_ILM_ACCESS(0);
   t_ilm_surface lFocusedSurface = 0;
#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL
   t_ilm_surface* surfaces;
   ilmInputDevice* bitmasks;
   t_ilm_uint num_ids = 0;
   ilmErrorTypes lIlmError = ilm_getInputFocus(&surfaces, &bitmasks, &num_ids);
   if (ILM_SUCCESS == lIlmError)
   {
      for (uint i = 0; i < num_ids; ++i)
      {
         if (bitmasks[i] & ILM_INPUT_DEVICE_KEYBOARD)
         {
            lFocusedSurface = surfaces[i];
            break;
         }
      }
      free(bitmasks);
      free(surfaces);
   }
#else
   ilm_GetKeyboardFocusSurfaceId(&lFocusedSurface);
#endif
   return UInt32(lFocusedSurface);
}


// ------------------------------------------------------------------------
bool IlmAccessor::HasKeyboardFocus(UInt32 surfaceId)
{
   CHECK_ILM_ACCESS(true);
   bool lRc;

   ilmInputDevice* lIlmInputDevice;
   t_ilm_surface* lIlmSurfaces;
   t_ilm_uint lNumIds = 0;

   ilmErrorTypes lIlmError = ilm_getInputFocus(&lIlmSurfaces, &lIlmInputDevice, &lNumIds);

   lRc = false;
   if (ILM_SUCCESS == lIlmError)
   {
      for (uint i = 0; i < lNumIds; ++i)
      {
         if ((lIlmInputDevice[i] & ILM_INPUT_DEVICE_KEYBOARD) &&
               (lIlmSurfaces[i] == surfaceId))
         {
            lRc = true;
         }
      }
      free(lIlmInputDevice);
      free(lIlmSurfaces);
   }
   return lRc;
}


bool IlmAccessor::SetPointerFocus(UInt32 surfaceId,
                                  bool commit,
                                  ilmErrorTypes& ilmError)
{
   CHECK_ILM_ACCESS(true);
   // Set pointer focus to given surface
   t_ilm_surface lSurfaceIDs[1] = {surfaceId};
   ilmError = ilm_setInputFocus(lSurfaceIDs, 1, ILM_INPUT_DEVICE_POINTER, ILM_TRUE);

   bool lRc = (ILM_SUCCESS == ilmError);
   // Commit changes to ILM
   if (commit)
   {
      lRc = lRc && Commit(ilmError);
   }
   // Log transaction info
   if (lRc)
   {
      ETG_TRACE_USR4(("Pointer focus %40s to surface %u",
                      (commit ? "set" : "requested"),
                      surfaceId));
   }
   else
   {
      ETG_TRACE_ERR(("Pointer focus request for surface %u failed!", surfaceId));
   }
   return lRc;
}


// ------------------------------------------------------------------------
UInt32 IlmAccessor::GetPointerFocus()
{
   CHECK_ILM_ACCESS(0);
   t_ilm_surface lFocusedSurface = 0;
#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL
   t_ilm_surface* surfaces;
   ilmInputDevice* bitmasks;
   t_ilm_uint num_ids = 0;
   ilmErrorTypes lIlmError = ilm_getInputFocus(&surfaces, &bitmasks, &num_ids);
   if (ILM_SUCCESS == lIlmError)
   {
      for (uint i = 0; i < num_ids; ++i)
      {
         if (bitmasks[i] & ILM_INPUT_DEVICE_POINTER)
         {
            lFocusedSurface = surfaces[i];
            break;
         }
      }
      free(bitmasks);
      free(surfaces);
   }
#else
   ilm_GetPointerFocusSurfaceId(&lFocusedSurface);
#endif
   return UInt32(lFocusedSurface);
}


// ------------------------------------------------------------------------
bool IlmAccessor::HasPointerFocus(UInt32 surfaceId)
{
   CHECK_ILM_ACCESS(true);
   bool lRc;
   ilmInputDevice* lIlmInputDevice;
   t_ilm_surface* lIlmSurfaces;
   t_ilm_uint lNumIds = 0;
   ilmErrorTypes lIlmError = ilm_getInputFocus(&lIlmSurfaces, &lIlmInputDevice, &lNumIds);

   lRc = false;
   if (ILM_SUCCESS == lIlmError)
   {
      for (uint i = 0; i < lNumIds; ++i)
      {
         if ((lIlmInputDevice[i] & ILM_INPUT_DEVICE_POINTER) &&
               (lIlmSurfaces[i] == surfaceId))
         {
            lRc = true;
         }
      }
      free(lIlmInputDevice);
      free(lIlmSurfaces);
   }
   return lRc;
}


// ------------------------------------------------------------------------
bool IlmAccessor::AcceptInputEvents(UInt32 surfaceId,
                                    ilmInputDevice inputDevices,
                                    t_ilm_bool activate,
                                    bool commit,
                                    ilmErrorTypes& ilmError)
{
   CHECK_ILM_ACCESS(true);
   bool lRc = true;
#ifndef VARIANT_S_FTR_ENABLE_IVI_SHELL
   //this function is redundant for pointer and keyboard but is required for touch.
   // Enable all input events for entered surface
   ilmError = ilm_UpdateInputEventAcceptanceOn(surfaceId,
              inputDevices,
              activate);
   lRc = (ILM_SUCCESS == ilmError);
   // Commit changes to ILM
   if (commit)
   {
      lRc = lRc && Commit(ilmError);
   }
#else
   (void) ilmError;
#endif
   // Log transaction info
   if (lRc)
   {
      if (activate)
      {
         gSurfaceInputAcceptanceMap[surfaceId] |= inputDevices;
      }
      else
      {
         gSurfaceInputAcceptanceMap[surfaceId] &= ~inputDevices;
      }
      ETG_TRACE_USR4(("Input event acceptance %40s for surface %u is set for devices %u",
                      (commit ? "set" : "requested"),
                      surfaceId,
                      inputDevices));
   }
   else
   {
      ETG_TRACE_ERR(("Input event acceptance request for surface %u for devices %u failed!",
                     surfaceId,
                     inputDevices));
   }
   return lRc;
}


// ------------------------------------------------------------------------
ilmInputDevice IlmAccessor::GetInputAcceptance(UInt32 surfaceId)
{
   return gSurfaceInputAcceptanceMap[surfaceId];
}


// ------------------------------------------------------------------------
bool IlmAccessor::SetIntputAcceptanceOn(UInt32 surfaceId, t_ilm_uint NumberSeats,
                                        t_ilm_string* seats)
{
   CHECK_ILM_ACCESS(true);
   ilmErrorTypes lIlmError = ilm_setInputAcceptanceOn(surfaceId, NumberSeats, seats);
   bool lRc = (ILM_SUCCESS == lIlmError);
   if (lRc)
   {
      ETG_TRACE_USR4(("Set input acceptance for surface %u on following seats:", surfaceId));
      for (uint i = 0; i < NumberSeats; ++i)
      {
         ETG_TRACE_USR4(("\t%40s", seats[i]));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Failed to set input acceptance for surface %u on following seats:", surfaceId));
      for (uint i = 0; i < NumberSeats; ++i)
      {
         ETG_TRACE_ERR(("\t%40s", seats[i]));
      }
   }
   return lRc;
}


// ------------------------------------------------------------------------
bool IlmAccessor::UpdateAppearance(UInt32 layerId,
                                   UInt32 surfaceId,
                                   Scaling::Enum scaling,
                                   HorizontalAlignment::Enum horizontalAlignment,
                                   VerticalAlignment::Enum verticalAlignment,
                                   bool commit,
                                   ilmErrorTypes& ilmError)
{
   CHECK_ILM_ACCESS(true);
   bool lRc;

   // Retrieve layer dimension
   t_ilm_uint lLayerDimension[2] = { 0, 0 };
#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL
   ilmLayerProperties properties;
   ilmError = ilm_getPropertiesOfLayer(layerId, &properties);
   if (ilmError == ILM_SUCCESS)
   {
      lLayerDimension[0] = properties.destWidth;
      lLayerDimension[1] = properties.destHeight;
   }
#else
   ilmError = ilm_layerGetDimension(layerId, lLayerDimension);
#endif
   lRc = (ILM_SUCCESS == ilmError);

   t_ilm_uint lLayerWidth = lLayerDimension[0];
   t_ilm_uint lLayerHeight = lLayerDimension[1];

   // Retrieve surface position
   t_ilm_uint lSurfacePosition[2] = { 0, 0 };
   if (lRc)
   {
#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL
      GetSurfacePosition(surfaceId, lSurfacePosition[0], lSurfacePosition[1], ilmError);
#else
      ilmError = ilm_surfaceGetPosition(surfaceId, lSurfacePosition);
#endif
      lRc = (ILM_SUCCESS == ilmError);
   }

   t_ilm_uint lPosX = lSurfacePosition[0];
   t_ilm_uint lPosY = lSurfacePosition[1];

   // Retrieve surface dimension
   t_ilm_uint lSurfaceDimension[2] = { 0, 0 };
   if (lRc)
   {
#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL
      GetSurfaceDimension(surfaceId, lSurfaceDimension[0], lSurfaceDimension[1], ilmError);
#else
      ilmError = ilm_surfaceGetDimension(surfaceId, lSurfaceDimension);
#endif
      lRc = (ILM_SUCCESS == ilmError);
   }

   t_ilm_uint lWidth = lSurfaceDimension[0];
   t_ilm_uint lHeight = lSurfaceDimension[1];

   if (lRc && (0 != lWidth) && (0 != lHeight))
   {
      // calculate dimension (scaling)
      t_ilm_float lAspectRatio = t_ilm_float(lWidth) / t_ilm_float(lHeight);
      switch (scaling)
      {
         case Scaling::Cut:
            lWidth = t_ilm_uint(lLayerDimension[1] * lAspectRatio);
            lHeight = lLayerDimension[1];
            break;
         case Scaling::Fit:
            lWidth = lLayerDimension[0];
            lHeight = t_ilm_uint(lLayerDimension[0] / lAspectRatio);
            break;
         case Scaling::Stretch:
            lWidth = lLayerDimension[0];
            lHeight = lLayerDimension[1];
            break;
         case Scaling::None:
         default:
            break;
      }

      // calculate x position (horizontal alignment)
      switch (horizontalAlignment)
      {
         case HorizontalAlignment::Left:
            break;
         case HorizontalAlignment::Right:
            if (lLayerWidth > lWidth)
            {
               lPosX = static_cast<t_ilm_uint>(lLayerWidth - lWidth);
            }
            break;
         case HorizontalAlignment::Center:
         default:
            if (lLayerWidth > lWidth)
            {
               lPosX = static_cast<t_ilm_uint>((lLayerWidth - lWidth) / 2);
            }
            break;
      }

      // calculate y position (vertical alignment)
      switch (verticalAlignment)
      {
         case VerticalAlignment::Top:
            break;
         case VerticalAlignment::Bottom:
            if (lLayerHeight > lHeight)
            {
               lPosY = static_cast<t_ilm_uint>(lLayerHeight - lHeight);
            }
            break;
         case VerticalAlignment::Center:
         default:
            if (lLayerHeight > lHeight)
            {
               lPosY = static_cast<t_ilm_uint>((lLayerHeight - lHeight) / 2);
            }
            break;
      }

      // set surface position and dimension
      lRc = IlmAccessor::SetSurfacePosition(surfaceId,
                                            lPosX,
                                            lPosY,
                                            false,
                                            ilmError);

      lRc = lRc && IlmAccessor::SetSurfaceDimension(surfaceId,
            lWidth,
            lHeight,
            false,
            ilmError);
   }

   // Commit changes to ILM
   if (commit)
   {
      lRc = lRc && IlmAccessor::Commit(ilmError);
   }

   // Log transaction info
   if (lRc)
   {
      ETG_TRACE_USR4(("Update alignment and scaling of surface %u %40s",
                      surfaceId,
                      (commit ? "succeeded" : "requested")));
   }
   else
   {
      ETG_TRACE_ERR(("Update alignment and scaling request for surface %u failed!",
                     surfaceId));
   }
   return lRc;
}


void IlmAccessor::GetScreenIds(std::vector<UInt32>& ids)
{
   t_ilm_uint numberOfScreens = 0;
   t_ilm_uint* screenIds = 0;
   ilmErrorTypes err = ilm_getScreenIDs(&numberOfScreens, &screenIds);

   if (err != ILM_SUCCESS)
   {
      ETG_TRACE_ERR(("ilm_getScreenIDs returned with error"));
      return;
   }

   for (uint32_t i = 0; i < numberOfScreens; ++i)
   {
      ids.push_back(screenIds[i]);
   }
}


bool IlmAccessor::GetScreenDimensions(UInt32 screenId,
                                      t_ilm_uint& width,
                                      t_ilm_uint& height,
                                      ilmErrorTypes& ilmError)
{
   ilmError = ilm_getScreenResolution(screenId, &width, &height);

   return (ilmError == ILM_SUCCESS);
}


#define TEST_ACCEPT_INPUT_EVENTS(surfaceId, inputDevices, activate)             \
    do {                                                                        \
        ilmErrorTypes ilmError = ILM_SUCCESS;                                   \
        AcceptInputEvents(surfaceId, inputDevices, activate, false, ilmError);  \
        sprintf(str, "%sable input acceptance %u -> resulting in %u",  \
                              activate ? "En" : "Dis",                          \
                              inputDevices,                                     \
                              GetInputAcceptance(surfaceId));                   \
    } while(0)

// ------------------------------------------------------------------------
void IlmAccessor::TestInputAcceptance(UInt32 surfaceId)
{
   ETG_TRACE_USR1((">>> BEGIN %40s(%u) <<<", __FUNCTION__, surfaceId));

   {
      char str[230];
      TEST_ACCEPT_INPUT_EVENTS(surfaceId, ILM_INPUT_DEVICE_KEYBOARD, ILM_TRUE);
      ETG_TRACE_USR4(("%40s", str));
   }
   {
      char str[230];
      TEST_ACCEPT_INPUT_EVENTS(surfaceId, ILM_INPUT_DEVICE_KEYBOARD, ILM_FALSE);
      ETG_TRACE_USR4(("%40s", str));
   }
   {
      char str[230];
      TEST_ACCEPT_INPUT_EVENTS(surfaceId, ILM_INPUT_DEVICE_POINTER , ILM_TRUE);
      ETG_TRACE_USR4(("%40s", str));
   }
   {
      char str[230];
      TEST_ACCEPT_INPUT_EVENTS(surfaceId, ILM_INPUT_DEVICE_KEYBOARD, ILM_TRUE);
      ETG_TRACE_USR4(("%40s", str));
   }
   {
      char str[230];
      TEST_ACCEPT_INPUT_EVENTS(surfaceId, ILM_INPUT_DEVICE_ALL     , ILM_FALSE);
      ETG_TRACE_USR4(("%40s", str));
   }
   {
      char str[230];
      TEST_ACCEPT_INPUT_EVENTS(surfaceId, ILM_INPUT_DEVICE_TOUCH   , ILM_TRUE);
      ETG_TRACE_USR4(("%40s", str));
   }
   {
      char str[230];
      TEST_ACCEPT_INPUT_EVENTS(surfaceId, ILM_INPUT_DEVICE_ALL     , ILM_TRUE);
      ETG_TRACE_USR4(("%40s", str));
   }
   {
      char str[230];
      TEST_ACCEPT_INPUT_EVENTS(surfaceId, ILM_INPUT_DEVICE_POINTER , ILM_FALSE);
      ETG_TRACE_USR4(("%40s", str));
   }
   {
      char str[230];
      TEST_ACCEPT_INPUT_EVENTS(surfaceId, ILM_INPUT_DEVICE_KEYBOARD, ILM_FALSE);
      ETG_TRACE_USR4(("%40s", str));
   }
   {
      char str[230];
      TEST_ACCEPT_INPUT_EVENTS(surfaceId, ILM_INPUT_DEVICE_TOUCH   , ILM_FALSE);
      ETG_TRACE_USR4(("%40s", str));
   }

   ETG_TRACE_USR1((">>> END %40s(%u) <<<", __FUNCTION__, surfaceId));
}


// ------------------------------------------------------------------------
bool IlmAccessor::GetLayerProperties(UInt32 layerId,
                                     t_ilm_int& posX,
                                     t_ilm_int& posY,
                                     t_ilm_int& width,
                                     t_ilm_int& height,
                                     t_ilm_bool& visibility,
                                     t_ilm_float& opacity,
                                     ilmErrorTypes& ilmError)
{
   CHECK_ILM_ACCESS(true);
   // Get layer dimension
   ilmLayerProperties properties;
   ilmError = ilm_getPropertiesOfLayer(layerId, &properties);
   if (ilmError == ILM_SUCCESS)
   {
      width = properties.destWidth;
      height = properties.destHeight;
      posX = properties.destX;
      posY = properties.destY;
      visibility = properties.visibility;
      opacity = properties.opacity;
   }
   return (ilmError == ILM_SUCCESS);
}


// ------------------------------------------------------------------------
bool IlmAccessor::SetLayerRectangle(UInt32 surfaceId,
                                    t_ilm_int posX,
                                    t_ilm_int posY,
                                    t_ilm_int width,
                                    t_ilm_int height,
                                    bool commit,
                                    ilmErrorTypes& ilmError)
{
   CHECK_ILM_ACCESS(true);
   t_ilm_int sourceX;
   t_ilm_int sourceY;
   t_ilm_int sourceWidth;
   t_ilm_int sourceHeight;
   t_ilm_int destX;
   t_ilm_int destY;
   t_ilm_int destWidth;
   t_ilm_int destHeight;

   if (posX < 0)
   {
      sourceX = -posX;
      destX = 0;
      sourceWidth = width - sourceX;
      destWidth = width - sourceX;
   }
   else
   {
      sourceX = 0;
      destX = posX;
      sourceWidth = width - destX;
      destWidth = width - destX;
   }

   if (posY < 0)
   {
      sourceY = -posY;
      destY = 0;
      sourceHeight = height - sourceY;
      destHeight = height - sourceY;
   }
   else
   {
      sourceY = 0;
      destY = posY;
      sourceHeight = height - destY;
      destHeight = height - destY;
   }

   ETG_TRACE_USR4(("SetLayerRectangle ID:%u source (%d,%d)/%dx%d dest (%d,%d)/%dx%d",
                   surfaceId,
                   sourceX, sourceY,
                   sourceWidth, sourceHeight,
                   destX, destY,
                   destWidth, destHeight));

   bool rc = IlmAccessor::SetLayerSourceRectangle(surfaceId,
             sourceX, sourceY,
             sourceWidth, sourceHeight,
             false, ilmError);
   rc = rc && IlmAccessor::SetLayerDestinationRectangle(surfaceId,
         destX, destY,
         destWidth, destHeight,
         commit, ilmError);
   return rc;
}


// ------------------------------------------------------------------------
bool IlmAccessor::SetLayerSourceRectangle(UInt32 layerId,
      t_ilm_int posX,
      t_ilm_int posY,
      t_ilm_int width,
      t_ilm_int height,
      bool commit,
      ilmErrorTypes& ilmError)
{
   CHECK_ILM_ACCESS(true);
   // Get layer dimension
   ilmError = ilm_layerSetSourceRectangle(layerId, posX, posY, width, height);
   bool lRc = (ILM_SUCCESS == ilmError);
   if ((ilmError == ILM_SUCCESS) && (commit))
   {
      lRc = Commit(ilmError);
   }
   return lRc;
}


// ------------------------------------------------------------------------
bool IlmAccessor::GetLayerRectangle(UInt32 layerId,
                                    t_ilm_int& posX,
                                    t_ilm_int& posY,
                                    t_ilm_int& width,
                                    t_ilm_int& height,
                                    ilmErrorTypes& ilmError)
{
   CHECK_ILM_ACCESS(true);
   // Get layer dimension
   ilmLayerProperties properties;
   ilmError = ilm_getPropertiesOfLayer(layerId, &properties);
   if (ilmError == ILM_SUCCESS)
   {
      width = properties.destWidth;
      height = properties.destHeight;
      posX = properties.destX;
      posY = properties.destY;
   }
   return (ilmError == ILM_SUCCESS);
}


// ------------------------------------------------------------------------
bool IlmAccessor::SetLayerDestinationRectangle(UInt32 layerId,
      t_ilm_int posX,
      t_ilm_int posY,
      t_ilm_int width,
      t_ilm_int height,
      bool commit,
      ilmErrorTypes& ilmError)
{
   CHECK_ILM_ACCESS(true);
   // Get layer dimension
   ilmError = ilm_layerSetDestinationRectangle(layerId, posX, posY, width, height);
   bool lRc = (ILM_SUCCESS == ilmError);
   if ((ilmError == ILM_SUCCESS) && (commit))
   {
      lRc = Commit(ilmError);
   }
   return lRc;
}


// ------------------------------------------------------------------------
bool IlmAccessor::SetLayerOpacity(UInt32 layerId,
                                  t_ilm_float opacity,
                                  bool commit,
                                  ilmErrorTypes& ilmError)
{
   CHECK_ILM_ACCESS(true);
   // Get layer dimension
   ilmError = ilm_layerSetOpacity(layerId, opacity);
   bool lRc = (ILM_SUCCESS == ilmError);
   if ((ilmError == ILM_SUCCESS) && (commit))
   {
      lRc = Commit(ilmError);
   }
   return lRc;
}


///
bool IlmAccessor::SetLayerVisibility(UInt32 layerId,
                                     t_ilm_bool visibility,
                                     bool commit,
                                     ilmErrorTypes& ilmError)
{
   CHECK_ILM_ACCESS(true);
   // Get layer dimension
   ilmError = ilm_layerSetVisibility(layerId, visibility);
   bool lRc = (ILM_SUCCESS == ilmError);
   if ((ilmError == ILM_SUCCESS) && (commit))
   {
      lRc = Commit(ilmError);
   }
   return lRc;
}


// ------------------------------------------------------------------------
bool IlmAccessor::isLayerAvailable(UInt32 layerId)
{
   CHECK_ILM_ACCESS(true);
   // Read all available layers
   t_ilm_int pALLength;
   t_ilm_layer* ppALArray;
   if (ILM_SUCCESS == ilm_getLayerIDs(&pALLength, &ppALArray))
   {
      ETG_TRACE_USR4(("ILM: %u layers found", pALLength));
      for (Int i = 0; i < pALLength; ++i)
      {
         if (layerId == ppALArray[i])
         {
            ETG_TRACE_USR4(("  ILM layer ID:%u found", ppALArray[i]));
            SAFE_FREE(ppALArray);
            return true;
         }
      }
      SAFE_FREE(ppALArray);
   }
   return false;
}


// ------------------------------------------------------------------------
UInt32 IlmAccessor::getAttachedSurfaceCount(UInt32 layerId)
{
   t_ilm_int lNoOfSurfaceIds = 0;
   t_ilm_surface* lSurfaceIds = 0;
   CHECK_ILM_ACCESS(true);
   if (ILM_SUCCESS == ilm_getSurfaceIDsOnLayer(layerId, &lNoOfSurfaceIds, &lSurfaceIds))
   {
      ETG_TRACE_USR4((" %u Surfaces attached to %u layer", lNoOfSurfaceIds, layerId));
   }
   SAFE_FREE(lSurfaceIds);
   return static_cast<UInt32>(lNoOfSurfaceIds);
}


// ------------------------------------------------------------------------
bool IlmAccessor::removeAllAttachedSurfaces(UInt32 layerId)
{
   bool lRc = true;
   t_ilm_int lNoOfSurfaceIds = 0;
   t_ilm_surface* lSurfaceIds = 0;
   CHECK_ILM_ACCESS(true);
   if (ILM_SUCCESS == ilm_getSurfaceIDsOnLayer(layerId, &lNoOfSurfaceIds, &lSurfaceIds))
   {
      ETG_TRACE_USR4((" %u Surfaces attached to %u layer", lNoOfSurfaceIds, layerId));
      Int i = 0;
      while (i < lNoOfSurfaceIds)
      {
         lRc = lRc && (ILM_SUCCESS == ilm_layerRemoveSurface(layerId, lSurfaceIds[i]));
         ETG_TRACE_USR4((" %u Surface Removed from %u layer", lSurfaceIds[i], layerId));
         ++i;
      }
   }
   SAFE_FREE(lSurfaceIds);
   if (!lRc)
   {
      ETG_TRACE_ERR((" All Attached Surfaces are not removed from %u layer", layerId));
   }
   return lRc;
}


}
