/* ***************************************************************************************
* FILE:          SlidingAnimation.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  SlidingAnimation.cpp is part of HMI-Base ScreenBrokerPlugins
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#include "SlidingAnimation.h"


#include <Shared/IlmAccessor.h>
#include <Shared/Animation/InterpolationFactory.h>
#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_PLUGINS
#include "trcGenProj/Header/SlidingAnimation.cpp.trc.h"
#endif


namespace ScreenBroker {
// SCREENBROKER_LOG_SET_REALM(LogRealm::Animation);

// ------------------------------------------------------------------------
bool SlidingAnimation::InitializeSurfaceState(SlidingAnimationSurfaceProperties& surfaceProperties, bool commit)
{
   ilmErrorTypes lIlmError = ILM_SUCCESS;

   AccelerationType::Enum lAcceleration = GetAcceleration();
   bool rc = false;
   InterpolationStrategy* lInterpolation = InterpolationFactory::Create(lAcceleration);
   if (0 != lInterpolation)
   {
      Float lAmplitude = GetAmplitude();
      lInterpolation->SetAmplitude(lAmplitude);
      SetInterpolation(lInterpolation);

      Position& lInitialPosition = surfaceProperties.originalPropertyValue.position;
      Position& lFinalPosition = surfaceProperties.finalPropertyValue.position;

      ETG_TRACE_USR1(("Slide: Established for %40s ID:%u from (%d,%d) to (%d,%d) in %ums with %u updates",
                      (isLayerContent ? "Layer" : "Surface"),
                      (isLayerContent ? surfaceProperties.layerId : surfaceProperties.surfaceId),
                      lInitialPosition.x, lInitialPosition.y,
                      lFinalPosition.x, lFinalPosition.y,
                      GetDuration(),
                      GetUpdateCount()));

      ETG_TRACE_USR1(("Slide: Initialize %40s ID:%u to x=%d y=%d",
                      (isLayerContent ? "Layer" : "Surface"),
                      (isLayerContent ? surfaceProperties.layerId : surfaceProperties.surfaceId),
                      lInitialPosition.x, lInitialPosition.y));
      if (isLayerContent)
      {
         rc = IlmAccessor::SetLayerRectangle(
                 surfaceProperties.layerId,
                 lInitialPosition.x,
                 lInitialPosition.y,
                 surfaceProperties.originalPropertyValue.width,
                 surfaceProperties.originalPropertyValue.height,
                 commit,
                 lIlmError);
      }
      else
      {
         rc = IlmAccessor::SetSurfaceRectangle(
                 surfaceProperties.surfaceId,
                 lInitialPosition.x,
                 lInitialPosition.y,
                 surfaceProperties.originalPropertyValue.width,
                 surfaceProperties.originalPropertyValue.height,
                 commit,
                 lIlmError);
      }
   }
   return rc;
}


// ------------------------------------------------------------------------
bool SlidingAnimation::UpdateSurfaceState(const SlidingAnimationSurfaceProperties& surfaceProperties,
      UInt32 count,
      bool commit)
{
   ilmErrorTypes lIlmError = ILM_SUCCESS;
   bool rc = false;

   InterpolationStrategy* lInterpolation = GetInterpolation();
   if (0 != lInterpolation)
   {
      const Position& lOriginalPosition = surfaceProperties.originalPropertyValue.position;
      const Position& lFinalPosition = surfaceProperties.finalPropertyValue.position;

      t_ilm_int lDeltaPosX = static_cast<t_ilm_int>(lInterpolation->Interpolate(GetUpdateCount(),
                             static_cast<Float>(lOriginalPosition.x),
                             static_cast<Float>(lFinalPosition.x),
                             count));
      t_ilm_int lDeltaPosY = static_cast<t_ilm_int>(lInterpolation->Interpolate(GetUpdateCount(),
                             static_cast<Float>(lOriginalPosition.y),
                             static_cast<Float>(lFinalPosition.y),
                             count));
      //Add original offset values to interpolated delta positions
      t_ilm_int lCurrentPosX = lOriginalPosition.x + lDeltaPosX;
      t_ilm_int lCurrentPosY = lOriginalPosition.y + lDeltaPosY;

      ETG_TRACE_USR4(("Slide: Update %40s ID:%u to (%d,%d) (count: %u)",
                      (isLayerContent ? "Layer" : "Surface"),
                      (isLayerContent ? surfaceProperties.layerId : surfaceProperties.surfaceId),
                      lCurrentPosX, lCurrentPosY,
                      count));
      if (isLayerContent)
      {
         rc = IlmAccessor::SetLayerRectangle(
                 surfaceProperties.layerId,
                 lCurrentPosX,
                 lCurrentPosY,
                 surfaceProperties.originalPropertyValue.width,
                 surfaceProperties.originalPropertyValue.height,
                 commit,
                 lIlmError);
      }
      else
      {
         rc = IlmAccessor::SetSurfaceRectangle(
                 surfaceProperties.surfaceId,
                 lCurrentPosX,
                 lCurrentPosY,
                 surfaceProperties.originalPropertyValue.width,
                 surfaceProperties.originalPropertyValue.height,
                 commit,
                 lIlmError);
      }
   }

   return rc;
}


// ------------------------------------------------------------------------
bool SlidingAnimation::SetFinalSurfaceState(const SlidingAnimationSurfaceProperties& surfaceProperties,
      bool commit)
{
   ilmErrorTypes lIlmError = ILM_SUCCESS;
   const Position& lFinalPosition =  surfaceProperties.finalPropertyValue.position;
   const Rectangle& originalBounds = surfaceProperties.originalPropertyValue;

   ETG_TRACE_USR1(("Slide: Finalize %40s ID:%u to (%d,%d)",
                   (isLayerContent ? "Layer" : "Surface"),
                   (isLayerContent ? surfaceProperties.layerId : surfaceProperties.surfaceId),
                   lFinalPosition.x, lFinalPosition.y));
   bool rc = false;
   if (isLayerContent)
   {
      rc = IlmAccessor::SetLayerRectangle(
              surfaceProperties.layerId,
              lFinalPosition.x,
              lFinalPosition.y,
              originalBounds.width,
              originalBounds.height, commit, lIlmError);
   }
   else
   {
      rc = IlmAccessor::SetSurfaceRectangle(
              surfaceProperties.surfaceId,
              lFinalPosition.x,
              lFinalPosition.y,
              originalBounds.width,
              originalBounds.height, commit, lIlmError);
   }

   InterpolationStrategy* lInterpolation = GetInterpolation();
   if (0 != lInterpolation)
   {
      lInterpolation->Release();
      SetInterpolation(0);
   }

   return rc;
}


}
