/* ***************************************************************************************
* FILE:          InterpolationFactory.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  InterpolationFactory.cpp is part of HMI-Base ScreenBrokerPlugins
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "InterpolationFactory.h"


#include <Shared/Animation/BounceEaseInterpolationStrategy.h>
#include <Shared/Animation/EaseInterpolationStrategy.h>
#include <Shared/Animation/LinearInterpolationStrategy.h>
#include <Shared/Animation/QuadEaseInterpolationStrategy.h>
#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_PLUGINS
#include "trcGenProj/Header/InterpolationFactory.cpp.trc.h"
#endif


namespace ScreenBroker {
// SCREENBROKER_LOG_SET_REALM(LogRealm::Animation);

typedef InterpolationPool<LinearInterpolationStrategy, InterpolationFactory::cMaxInterpolations> LinearInterpolationPool;
typedef InterpolationPool<BounceEaseInterpolationStrategy, InterpolationFactory::cMaxInterpolations> BounceEaseInterpolationPool;
typedef InterpolationPool<QuadEaseInterpolationStrategy, InterpolationFactory::cMaxInterpolations> QuadEaseInterpolationPool;

// ------------------------------------------------------------------------
InterpolationStrategy* InterpolationFactory::Create(AccelerationType::Enum acceleration)
{
   InterpolationStrategy* lInterpolationStrategy = 0;

   switch (acceleration)
   {
      case AccelerationType::Linear:
         lInterpolationStrategy = CreateLinearInterpolationStrategy();
         break;
      case AccelerationType::InQuad:
         lInterpolationStrategy = CreateInQuadInterpolationStrategy();
         break;
      case AccelerationType::OutQuad:
         lInterpolationStrategy = CreateOutQuadInterpolationStrategy();
         break;
      case AccelerationType::OutBounce:
         lInterpolationStrategy = CreateOutBounceInterpolationStrategy();
         break;
      default:
         break;
         // nothing to do
   }

   return lInterpolationStrategy;
}


// ------------------------------------------------------------------------
InterpolationStrategy* InterpolationFactory::CreateLinearInterpolationStrategy()
{
   LinearInterpolationStrategy* lInterpolationStrategy =
      dynamic_cast<LinearInterpolationStrategy*>(LinearInterpolationPool::GetInstance().GetInterpolation());

   if (0 == lInterpolationStrategy)
   {
      ETG_TRACE_ERR(("Failed to get instance of LinearInterpolationStrategy"));
   }

   return lInterpolationStrategy;
}


// ------------------------------------------------------------------------
InterpolationStrategy* InterpolationFactory::CreateInQuadInterpolationStrategy()
{
   QuadEaseInterpolationStrategy* lInterpolationStrategy =
      dynamic_cast<QuadEaseInterpolationStrategy*>(QuadEaseInterpolationPool::GetInstance().GetInterpolation());

   if (0 != lInterpolationStrategy)
   {
      lInterpolationStrategy->SetEaseDirection(EaseDirection::EaseIn);
   }
   else
   {
      ETG_TRACE_ERR(("Failed to get instance of QuadEaseInterpolationStrategy"));
   }

   return lInterpolationStrategy;
}


// ------------------------------------------------------------------------
InterpolationStrategy* InterpolationFactory::CreateOutQuadInterpolationStrategy()
{
   QuadEaseInterpolationStrategy* lInterpolationStrategy =
      dynamic_cast<QuadEaseInterpolationStrategy*>(QuadEaseInterpolationPool::GetInstance().GetInterpolation());

   if (0 != lInterpolationStrategy)
   {
      lInterpolationStrategy->SetEaseDirection(EaseDirection::EaseOut);
   }
   else
   {
      ETG_TRACE_ERR(("Failed to get instance of QuadEaseInterpolationStrategy"));
   }

   return lInterpolationStrategy;
}


// ------------------------------------------------------------------------
InterpolationStrategy* InterpolationFactory::CreateOutBounceInterpolationStrategy()
{
   BounceEaseInterpolationStrategy* lInterpolationStrategy =
      dynamic_cast<BounceEaseInterpolationStrategy*>(BounceEaseInterpolationPool::GetInstance().GetInterpolation());

   if (0 != lInterpolationStrategy)
   {
      lInterpolationStrategy->SetEaseDirection(EaseDirection::EaseOut);
   }
   else
   {
      ETG_TRACE_ERR(("Failed to get instance of BounceEaseInterpolationStrategy"));
   }

   return lInterpolationStrategy;
}


}
