/* ***************************************************************************************
* FILE:          FadingAnimation.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FadingAnimation.cpp is part of HMI-Base ScreenBrokerPlugins
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "FadingAnimation.h"

#include <Shared/Animation/AccelerationType.h>
#include <Shared/Animation/InterpolationFactory.h>

#include <Shared/IlmAccessor.h>
#include <stdio.h>

#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_PLUGINS
#include "trcGenProj/Header/FadingAnimation.cpp.trc.h"
#endif


namespace ScreenBroker {
// SCREENBROKER_LOG_SET_REALM(LogRealm::Animation);

// ------------------------------------------------------------------------
bool FadingAnimation::InitializeSurfaceState(FadingAnimationSurfaceProperties& surfaceProperties, bool commit)
{
   AccelerationType::Enum lAcceleration = GetAcceleration();
   bool rc = false;
   InterpolationStrategy* lInterpolation = InterpolationFactory::Create(lAcceleration);
   if (0 != lInterpolation)
   {
      Float lAmplitude = GetAmplitude();
      lInterpolation->SetAmplitude(lAmplitude);
      SetInterpolation(lInterpolation);

      ilmErrorTypes lIlmError = ILM_SUCCESS;

      /*    ETG_TRACE_USR1(("Fade: Established for surface ID:%u from %.2f %% to %.2f %% opacity",
                                surfaceProperties.surfaceId,
                                surfaceProperties.originalPropertyValue * 100.,
                                surfaceProperties.finalPropertyValue * 100.));
          ETG_TRACE_USR1(("Fade: Established for surface ID:%u with opacity changing in %ums with %u updates",
                                        surfaceProperties.surfaceId,
                                        GetDuration(),
                                        GetUpdateCount())); */
      /*   ETG_TRACE_USR1(("Fade: Established for surface ID:%u from %.2f %% to %.2f %% opacity in %ums with %u updates",
                                       surfaceProperties.surfaceId,
                                       surfaceProperties.originalPropertyValue * 100.,
                                       surfaceProperties.finalPropertyValue * 100.,
      							  GetDuration(),
      							  GetUpdateCount()));*/

      {
         char str[230];
         sprintf(str, "Fade: Established for %40s ID:%u from %.2f %% to %.2f %% opacity in %ums with %u updates",
                 (isLayerContent ? "Layer" : "Surface"),
                 (isLayerContent ? surfaceProperties.layerId : surfaceProperties.surfaceId),
                 surfaceProperties.originalPropertyValue * 100.,
                 surfaceProperties.finalPropertyValue * 100.,
                 GetDuration(),
                 GetUpdateCount());
         ETG_TRACE_USR4(("%s", str));
      }

      ETG_TRACE_USR1(("Fade: Initialize %40s ID:%u to %.2f %% opacity",
                      (isLayerContent ? "Layer" : "Surface"),
                      (isLayerContent ? surfaceProperties.layerId : surfaceProperties.surfaceId),
                      (float)(surfaceProperties.originalPropertyValue * 100.)));
      if (isLayerContent)
      {
         rc = IlmAccessor::SetLayerOpacity(surfaceProperties.layerId,
                                           static_cast<t_ilm_float>(surfaceProperties.originalPropertyValue),
                                           commit,
                                           lIlmError);
      }
      else
      {
         rc = IlmAccessor::SetSurfaceOpacity(surfaceProperties.surfaceId,
                                             static_cast<t_ilm_float>(surfaceProperties.originalPropertyValue),
                                             commit,
                                             lIlmError);
      }
   }
   return rc;
}


// ------------------------------------------------------------------------
bool FadingAnimation::UpdateSurfaceState(const FadingAnimationSurfaceProperties& surfaceProperties,
      UInt32 count,
      bool commit)
{
   ilmErrorTypes lIlmError = ILM_SUCCESS;
   bool rc = false;

   InterpolationStrategy* lInterpolation = GetInterpolation();
   if (0 != lInterpolation)
   {
      t_ilm_float lDeltaOpacity = lInterpolation->Interpolate(GetUpdateCount(),
                                  surfaceProperties.originalPropertyValue,
                                  surfaceProperties.finalPropertyValue,
                                  count);
      //Add delta opacity values to original Opacity values to correct the animation direction
      t_ilm_float lCurrentOpacity = surfaceProperties.originalPropertyValue + lDeltaOpacity;
      ETG_TRACE_USR4(("Fade: Update %40s ID:%u to %.2f %% opacity (count: %u)",
                      (isLayerContent ? "Layer" : "Surface"),
                      (isLayerContent ? surfaceProperties.layerId : surfaceProperties.surfaceId),
                      (float)(lCurrentOpacity * 100.),
                      count));
      if (isLayerContent)
      {
         rc = IlmAccessor::SetLayerOpacity(surfaceProperties.layerId,
                                           lCurrentOpacity,
                                           commit,
                                           lIlmError);
      }
      else
      {
         rc = IlmAccessor::SetSurfaceOpacity(surfaceProperties.surfaceId,
                                             lCurrentOpacity,
                                             commit,
                                             lIlmError);
      }
   }
   return rc;
}


// ------------------------------------------------------------------------
bool FadingAnimation::SetFinalSurfaceState(const FadingAnimationSurfaceProperties& surfaceProperties, bool commit)
{
   ilmErrorTypes lIlmError = ILM_SUCCESS;
   bool rc = false;
   ETG_TRACE_USR1(("Fading: Finalize %40s ID:%u to %.2f %% opacity",
                   (isLayerContent ? "Layer" : "Surface"),
                   surfaceProperties.surfaceId,
                   (float)(surfaceProperties.finalPropertyValue * 100.)));
   if (isLayerContent)
   {
      rc = IlmAccessor::SetLayerOpacity(surfaceProperties.layerId,
                                        static_cast<t_ilm_float>(surfaceProperties.finalPropertyValue),
                                        commit,
                                        lIlmError);
   }
   else
   {
      rc = IlmAccessor::SetSurfaceOpacity(surfaceProperties.surfaceId,
                                          static_cast<t_ilm_float>(surfaceProperties.finalPropertyValue),
                                          commit,
                                          lIlmError);
   }

   InterpolationStrategy* lInterpolation = GetInterpolation();
   if (0 != lInterpolation)
   {
      lInterpolation->Release();
      SetInterpolation(0);
   }

   return rc;
}


}
