/* ***************************************************************************************
* FILE:          AnimatorTimer.h
* SW-COMPONENT:  HMI-BASE
* DESCRIPTION:   AnimatorTimer.h is part of HMI-Base ScreenBrokerPlugins
* COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#if !defined(ScreenBrokerPlugins_Shared_Animation_AnimatorTimer_h)
#define ScreenBrokerPlugins_Shared_Animation_AnimatorTimer_h

#include <iostream>
#include <map>
#include <pthread.h>
#include <ScreenBroker/Util/CriticalSection.h>

namespace ScreenBroker {
/** Forward declaration of Animator Class */
class Animator;

/** Defininf event Listener Queue Size for Polling */
const UInt32 eventListenerQueueSize = 25;
/** TimerFDMap is shorthand to use Map of File Descriptor ID and Expiration Timer Value */
typedef std::map<Int32, UInt32> TimerFDMap;

/**
 *AnimatorTimer is a Singleton class used to execute timed updates(minimum of 1ms)
 *requested for animation.Timer is accomplished by using timerfd
 *which is monitored by epoll_wait
 */
class AnimatorTimer
{
   public:

      /**
       *<Destructor for restting data>
       */
      ~AnimatorTimer();

      /**
       *<To Create a single instance of Timer>
       */
      static AnimatorTimer* getInstance()
      {
         if (0 == _instanceSelf)
         {
            _instanceSelf = new AnimatorTimer();
         }
         return _instanceSelf;
      }

      /**
       *<To Destroy the single instance of Timer>
       */
      static void destroyInstance()
      {
         if (0 != _instanceSelf)
         {
            delete _instanceSelf;
            _instanceSelf = 0;
         }
      }

      /**
       *<Method to Initiate Exit of Timer thread>
       *<should be invoked when no valid timer FD>
       */
      void terminateThread();

      /**
       *<Method To Initialize Timer Epolling in a seprate thread>
       *@pre <_timerFDMap should not be empty and atleast one time should be running>
       */
      bool initializeThread();

      /**
       *<Method to create the timer and map file descriptor id with expiratin period>
       *@param[in] <expirationPeriod> <Expiration period to update map _timerFDMap>
       *@return <Timer FD value of the created value>
       *@throws <-1=Failed to Create>
       */
      Int32 createTimer(UInt32 expirationPeriod);
      /**
       *<Method to start the timer with requested data>
       *@param[in] <idTFD> <TimerFD id to identify the timer>
       *@param[in] <expirationPeriod> <duration on which timer shall be updated>
       *@param[in] <isOneTimeShot> <decribes repeat status(default = false)>
       *@return <true = started/false = failed>
       *@pre <Timer should be created else starting will always fail>
       */
      bool startTimer(Int32 idTFD, UInt32 expirationPeriod, bool isOneTimeShot = false);

      /**
       *<Method to Stop the timer with requested data/to stop all timer with default data>
       *@param[in] <idTFD> <TimerFD id to identify the timer(default=-1)>
       */
      void stopTimer(Int32 idTFD = -1) ;

      /**
       *<Method to destroy the timer with requested data/to stop all timer with default data>
       *@param[in] <idTFD> <TimerFD id to identify the timer(default=-1)>
       */
      void destroyTimer(Int32 idTFD = -1) ;

      /**
       *<Method to get the list of created timers with thread saftey>
       *@return <_timerFDMap list of created timers>
       */
      bool isInTimerFDMap(Int32 idTFD)
      {
         CriticalSectionLocker lockData(&_criticalSection);
         return (_timerFDMap.find(idTFD) != _timerFDMap.end());
      }

      /**
       *<Method to Check if TimerFDMap is empty>
       *@return <_timerFDMap.empty()>
       */
      bool isTimerFDMapEmpty()
      {
         CriticalSectionLocker lockData(&_criticalSection);
         return _timerFDMap.empty();
      }

      /**
       *<Method to get Event Listener ID with thread saftey>
       *@return <Epoll Listener FD ID>
       */
      UInt32 getEventListenerFDID()
      {
         CriticalSectionLocker lockData(&_criticalSection);
         return _eventPollFDID;
      }

      /**
       *<Method to provide Exipred interval update from polling thread>
       *@param[in] <idTFD> <TimerFD id to identify the timer>
       *@param[in] <missed> <no of intervals expired but missed to read>
       */
      void expiredIntervalUpdate(Int32 idTFD, UInt64 missed);

      /**
        *<Method to update call back instance for timer expiry update>
        *@param[in] <IAnimatorTimer*> <IAnimatorTimer class instance object to update>
        *@pre <IAnimatorTimer class has to be initialized before setting call back>
        */
      void setCallBackInstance(Animator* pointerAnimator)
      {
         _instanceAnimator = pointerAnimator;
      }

      /** Static object of CriticalSection for thread saftey read/write */
      static CriticalSection _criticalSection;

   private:

      /** File descriptor Event Polling ID */
      Int32 _eventPollFDID;

      /** Static object of AnimatorTimer */
      static AnimatorTimer* _instanceSelf;

      /** Pointer of Animator to Store Animator Class Instance */
      Animator* _instanceAnimator;

      /** Signed Integer of 32 bit to Store Pipe Read FD ID */
      Int32 _readPipeFD;

      /** Signed Integer of 32 bit  to Store Pipe Wite FD ID */
      Int32 _writePipeFD;

      /** Signed Integer of 32 bit  to Store Thread ID */
      Int32 _threadID;

      /** pthread_t object to store the created thread instance via pass by reference */
      pthread_t _instanceThread;

      /** TimerFDMap to Store created timerfd with expiration time */
      TimerFDMap _timerFDMap;

      /**
       *<Constructor for initializing member attributes>
       */
      AnimatorTimer();

      /**
       *<Copy Constructor without definition to avoid copy>
       */
      AnimatorTimer(const AnimatorTimer& obj);

      /**
       *<Assignment Constructor without definition to avoid assignment>
       */
      AnimatorTimer& operator =(const AnimatorTimer& obj);

      /**
       *<Method to get validity of timer in the list>
       *@param[in] <idTFD> <TimerFD id to identify the timer>
       *@return <true=valid/false=not valid>
       */
      bool isValid(Int32 idTFD) ;

      /**
       *<Method to get validity of timer in the list>
       *@param[in] <idTFD> <TimerFD id to identify the timer>
       *@return <true=Running/false=Stopped>
       */
      bool isRunning(Int32 idTFD);

      /**
       *<Method to wake up epoll event by writing a character to pipe>
       */
      void wakeupEventListener();
      /**
       *<Method to Create wake up mechanism for epoll event using pipe>
       */
      Int32 initializeWakeUP();
      /**
       *<Method to initilize eventlistener by creating epoll>
       */
      Int32 initializeEventListener();

      /**
       *<Method to destory eventlistener by close(epoll)>
       */
      void destroyEPollEventListener();

      /**
       *<Method to attache timer fd to eventlistener>
       *@param[in] <idTFD> <TimerFD id to identify the timer>
       *@return <-1=failed/idTFD=Success>
       */
      Int32 attachEventListener(Int32 idTFD);

      /**
       *<Method to attache timer fd to eventlistener>
       *@param[in] <idTFD> <TimerFD id to identify the timer>
       */
      void detachEventListener(Int32 idTFD);
};


} //namespace ScreenBroker
#endif /* ScreenBrokerPlugins_Shared_Animation_AnimatorTimer_h */
