/* ***************************************************************************************
* FILE:          AnimatorTimer.cpp
* SW-COMPONENT:  HMI-BASE
* DESCRIPTION:   AnimatorTimer.cpp is part of HMI-Base ScreenBrokerPlugins
* COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include <sys/time.h>
#include <sys/timerfd.h>
#include <sys/epoll.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <climits>
#include <ScreenBroker/Util/Time.h>
#include <Shared/Animation/AnimatorTimer.h>
#include "Animator.h"

#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_PLUGINS
#include "trcGenProj/Header/AnimatorTimer.cpp.trc.h"
#endif

namespace ScreenBroker {

CriticalSection AnimatorTimer::_criticalSection;
AnimatorTimer* AnimatorTimer::_instanceSelf = 0;
// ------------------------------------------------------------------------
static void* loopedTimerPolling(void* data)
{
   AnimatorTimer* ptrTimer = static_cast<AnimatorTimer*>(data);
   while (!(ptrTimer->isTimerFDMapEmpty()))
   {
      epoll_event queuedEventsList[eventListenerQueueSize];
      // and wait for events
      //-1 as timeout for epoll to wait until wake up call
      Int32 queueSize = epoll_wait(ptrTimer->getEventListenerFDID(), queuedEventsList, eventListenerQueueSize, -1);
      if (queueSize < 0)
      {
         char buffer[256];
         char* errorMsg = strerror_r(errno, buffer, 256);
         ETG_TRACE_USR4_THR(("Epoll_Wait Error: %s", errorMsg));
         if ((errno == EINVAL) || (errno == EBADF))
         {
            ETG_TRACE_FATAL_THR(("Polling Empty Exiting Thread"));
            pthread_exit(NULL);
            return 0;
         }
      }
      else
      {
         Int32 queueIndex = 0;
         while (queueIndex < queueSize)
         {
            if (ptrTimer->isInTimerFDMap(queuedEventsList[queueIndex].data.fd))
            {
               uint64_t missed;
               if (-1 != read((queuedEventsList[queueIndex].data.fd), &missed, sizeof(missed)))
               {
                  ptrTimer->expiredIntervalUpdate(queuedEventsList[queueIndex].data.fd, missed);
               }
            }
            ++queueIndex;
         }
      }
   }
   ETG_TRACE_USR4_THR(("Exiting Thread"));
   pthread_exit(NULL);
   return 0;
}


// ------------------------------------------------------------------------
AnimatorTimer::AnimatorTimer():
   _eventPollFDID(-1),
   _threadID(-1),
   _instanceThread(0)
{
   _timerFDMap.clear();
   _instanceAnimator = 0;
   _readPipeFD = -1;
   _writePipeFD = -1;
   initializeEventListener();
}


// ------------------------------------------------------------------------
AnimatorTimer::~AnimatorTimer()
{
   terminateThread();
   destroyEPollEventListener();
   _timerFDMap.clear();
   _instanceAnimator = 0;
}


// ------------------------------------------------------------------------
bool AnimatorTimer::initializeThread()
{
   CriticalSectionLocker lockData(&_criticalSection);
   if ((-1 == _threadID) && (!(_timerFDMap.empty())))
   {
      // Starting timer thread waiting for expiration
      _threadID = pthread_create(&_instanceThread, NULL, loopedTimerPolling, reinterpret_cast<void*>(_instanceSelf));
      if (0 == _threadID)
      {
         ETG_TRACE_USR4_THR(("Starting thread ID : %d", _threadID));
         return true;
      }
      else
      {
         _threadID = -1;
         ETG_TRACE_FATAL_THR(("Initializing thread Failed"));
      }
   }
   else if ((0 == _threadID) && (!(_timerFDMap.empty())))
   {
      return true;
   }
   return false;
}


// ------------------------------------------------------------------------
void AnimatorTimer::terminateThread()
{
   if (0 == _threadID)
   {
      destroyTimer(-1);
      destroyEPollEventListener();
      _threadID = -1;
      ETG_TRACE_USR4_THR(("Thread Termination Initiated"));
   }
   else
   {
      ETG_TRACE_USR4_THR(("Thread not available termination"));
   }
}


// ------------------------------------------------------------------------
void AnimatorTimer::expiredIntervalUpdate(Int32 idTFD, UInt64 missed)
{
   CriticalSectionLocker lockData(&_criticalSection);
   UInt32 lCount = ((missed <= UINT_MAX) ? static_cast<UInt32>(missed) : static_cast<UInt32>(UINT_MAX));
   if (0 != _instanceAnimator)
   {
      _instanceAnimator->expiredTimerUpdate(idTFD, lCount);
   }
   else
   {
      ETG_TRACE_USR4_THR(("Invalid Update"));
   }
}


// ------------------------------------------------------------------------
Int32 AnimatorTimer::initializeWakeUP()
{
   //Adding Wake Up Mechanism
   Int32 pipefds[2];
   epoll_event ev;
   if (0 == pipe(pipefds))
   {
      _readPipeFD = pipefds[0];
      _writePipeFD = pipefds[1];
      // make read-end non-blocking
      Int32 flagread = fcntl(_readPipeFD, F_GETFL, 0);
      Int32 flagwrite = fcntl(_writePipeFD, F_SETFL, flagread | O_NONBLOCK);
      if ((-1 == flagread) || (-1 == flagwrite))
      {
         ETG_TRACE_USR4_THR(("Pipe Read/Write Control Failed"));
         _readPipeFD = close(_readPipeFD);
         _writePipeFD = close(_writePipeFD);
         if ((_readPipeFD == 0) && (0 == _writePipeFD))
         {
            ETG_TRACE_FATAL_THR(("Read&Write File Descriptors Closed"));
         }
         _writePipeFD = -1;
         _readPipeFD = -1;
         return -1;
      }
      else
      {
         // add the read end to the epoll
         ev.events = EPOLLIN;
         if (-1 == epoll_ctl(_eventPollFDID, EPOLL_CTL_ADD, _readPipeFD, &ev))
         {
            ETG_TRACE_FATAL_THR(("Pipe not Attached to Event Listener"));
            _readPipeFD = close(_readPipeFD);
            _writePipeFD = close(_writePipeFD);
            if ((_readPipeFD == 0) && (0 == _writePipeFD))
            {
               ETG_TRACE_FATAL_THR(("Read&Write File Descriptors Closed"));
            }
            _writePipeFD = -1;
            _readPipeFD = -1;
            return -1;
         }
      }
   }
   else
   {
      ETG_TRACE_FATAL_THR(("Pipe Creation Failed"));
      return -1;
   }
   return 0;
}


// ------------------------------------------------------------------------
Int32 AnimatorTimer::initializeEventListener()
{
   CriticalSectionLocker lockData(&_criticalSection);
   if (_eventPollFDID == -1)
   {
      _eventPollFDID = epoll_create(eventListenerQueueSize);
      if (_eventPollFDID != -1)
      {
         if (-1 == initializeWakeUP())
         {
            if (-1 == close(_eventPollFDID))
            {
               ETG_TRACE_FATAL_THR(("Close Event Listener Failed"));
            }
            _eventPollFDID = -1;
         }
         else
         {
            ETG_TRACE_USR4_THR(("Poll List Created"));
         }
      }
      else
      {
         ETG_TRACE_FATAL_THR(("Poll List Creation Failed"));
      }
   }
   return _eventPollFDID;
}


// ------------------------------------------------------------------------
void AnimatorTimer::wakeupEventListener()
{
   if (_eventPollFDID != -1)
   {
      char lOneChar = '1';
      if (-1 == write(_writePipeFD, &lOneChar, 1))
      {
         ETG_TRACE_FATAL_THR(("Pipe Write Failed"));
      }
   }
}


// ------------------------------------------------------------------------
void AnimatorTimer::destroyEPollEventListener()
{
   CriticalSectionLocker lockData(&_criticalSection);
   if (_eventPollFDID != -1)
   {
      wakeupEventListener();
      if (0 == close(_eventPollFDID))
      {
         _readPipeFD = close(_readPipeFD);
         _writePipeFD = close(_writePipeFD);
         if ((_readPipeFD == 0) && (0 == _writePipeFD))
         {
            ETG_TRACE_FATAL_THR(("Read&Write File Descriptors Closed"));
         }
         _writePipeFD = -1;
         _readPipeFD = -1;
         _eventPollFDID = -1;
      }
   }
}


// ------------------------------------------------------------------------
Int32 AnimatorTimer::attachEventListener(Int32 idTFD)
{
   CriticalSectionLocker lockData(&_criticalSection);
   if ((-1 != idTFD) && (-1 != initializeEventListener()))
   {
      epoll_event eventData;
      // add timer to reactor
      // notification is a read event
      eventData.events = EPOLLIN;
      eventData.data.fd = idTFD;
      if (-1 == epoll_ctl(_eventPollFDID, EPOLL_CTL_ADD, idTFD, &eventData))
      {
         ETG_TRACE_FATAL_THR(("Timer FD not Attached to Event Listener"));
         return -1;
      }
      else
      {
         ETG_TRACE_USR4_THR(("Timer FD Attached to Event Listener"));
         return idTFD;
      }
   }
   return -1;
}


// ------------------------------------------------------------------------
void AnimatorTimer::detachEventListener(Int32 idTFD)
{
   CriticalSectionLocker lockData(&_criticalSection);
   if (-1 != idTFD)
   {
      epoll_event eventData;
      // add timer to reactor
      // notification is a read event
      eventData.events = EPOLLIN;
      eventData.data.fd = idTFD;
      if (-1 == epoll_ctl(_eventPollFDID, EPOLL_CTL_DEL, idTFD, &eventData))
      {
         ETG_TRACE_FATAL_THR(("Timer FD Detach from Event Listener failed"));
      }
      ETG_TRACE_USR4_THR(("Timer FD Detached from Event Listener"));
   }
}


// ------------------------------------------------------------------------
Int32 AnimatorTimer::createTimer(UInt32 expirationPeriod)
{
   CriticalSectionLocker lockData(&_criticalSection);
   Int32 idTFD = timerfd_create(Time::ClockType(), TFD_NONBLOCK);
   bool lRc = (-1 != idTFD);
   if (lRc)
   {
      ETG_TRACE_USR4_THR(("Initializing animation timer for Interval : %d", expirationPeriod));
      if (-1 == attachEventListener(idTFD))
      {
         return -1;
      }
      _timerFDMap[idTFD] = expirationPeriod;
      ETG_TRACE_USR4_THR(("Created Timer of ID: %d", idTFD));
      return idTFD;
   }
   else
   {
      ETG_TRACE_FATAL_THR(("Creating animation timer failed for Interval :%d ", expirationPeriod));
   }
   return -1;
}


// ------------------------------------------------------------------------
void AnimatorTimer::destroyTimer(Int32 idTFD)
{
   CriticalSectionLocker lockData(&_criticalSection);
   stopTimer(idTFD);
   if (-1 == idTFD)
   {
      for (TimerFDMap::iterator it = _timerFDMap.begin(); it != _timerFDMap.end(); ++it)
      {
         if (0 == close(it->first))
         {
            ETG_TRACE_USR4_THR(("Destroyed Timer for Interval : %d", it->first));
         }
      }
      _timerFDMap.clear();
   }
   else if (_timerFDMap.find(idTFD) != _timerFDMap.end())
   {
      if (0 == close(idTFD))
      {
         ETG_TRACE_USR4_THR(("Destroyed Timer for Interval :%d", idTFD));
      }
      _timerFDMap.erase(idTFD);
   }
}


// ------------------------------------------------------------------------
bool AnimatorTimer::startTimer(Int32 idTFD, UInt32 expirationPeriod, bool isOneTimeShot)
{
   CriticalSectionLocker lockData(&_criticalSection);
   if ((_timerFDMap.find(idTFD) != _timerFDMap.end()))
   {
      ETG_TRACE_USR4_THR(("Starting %40stimer of animation %d with expiration in %ums",
                          (isOneTimeShot ? "t" : "periodic t"),
                          idTFD,
                          expirationPeriod));
      // Calculate timer expiration value
      UInt32 lSec = expirationPeriod / 1000;
      UInt32 lNs = (expirationPeriod - (lSec * 1000)) * 1000000;
      struct itimerspec iterValue;
      if (isOneTimeShot)
      {
         iterValue.it_interval.tv_sec = 0;
         iterValue.it_interval.tv_nsec = 0;
      }
      else
      {
         iterValue.it_interval.tv_sec = lSec;
         iterValue.it_interval.tv_nsec = lNs;
      }
      iterValue.it_value.tv_sec = lSec;
      iterValue.it_value.tv_nsec = lNs;

      if (0 == timerfd_settime(idTFD, 0, &iterValue, NULL))
      {
         ETG_TRACE_USR4_THR(("%40stimer of ID : %d with expiration in %ums started",
                             (isOneTimeShot ? "T" : "Periodic t"),
                             idTFD,
                             expirationPeriod));
         return true;
      }
      else
      {
         ETG_TRACE_FATAL_THR(("Starting animation timer failed for : %d", idTFD));
      }
   }
   else
   {
      ETG_TRACE_USR4_THR(("Invalid Timer ID"));
   }
   return false;
}


// ------------------------------------------------------------------------
void AnimatorTimer::stopTimer(Int32 idTFD)
{
   CriticalSectionLocker lockData(&_criticalSection);
   if (-1 == idTFD)
   {
      ETG_TRACE_USR4_THR(("Stopping All active Timers"));

      // Calculate reset timer expiration value
      struct itimerspec iterValue;
      iterValue.it_interval.tv_sec = 0;
      iterValue.it_interval.tv_nsec = 0;
      iterValue.it_value.tv_sec = 0;
      iterValue.it_value.tv_nsec = 1;
      for (TimerFDMap::iterator it = _timerFDMap.begin(); it != _timerFDMap.end(); ++it)
      {
         // Set expiration time to zero
         if (0 == timerfd_settime(it->first, 0, &iterValue, NULL))
         {
            ETG_TRACE_USR4_THR(("Timer Stopped for Interval: %d", it->first));
         }
      }
   }
   else if ((_timerFDMap.find(idTFD) != _timerFDMap.end()))
   {
      // Calculate reset timer expiration value
      struct itimerspec iterValue;
      iterValue.it_interval.tv_sec = 0;
      iterValue.it_interval.tv_nsec = 0;
      iterValue.it_value.tv_sec = 0;
      iterValue.it_value.tv_nsec = 1;
      // Set expiration time to zero
      if (0 == timerfd_settime(idTFD, 0, &iterValue, NULL))
      {
         ETG_TRACE_USR4_THR(("Timer Stopped for Interval :%d ", (_timerFDMap.find(idTFD)->second)));
      }
   }
}


// ------------------------------------------------------------------------
bool AnimatorTimer::isRunning(const Int32 idTFD)
{
   CriticalSectionLocker lockData(&_criticalSection);
   bool isActive = false;
   if ((!_timerFDMap.empty()) && (_timerFDMap.find(idTFD) != _timerFDMap.end()))
   {
      struct itimerspec iterValue;
      // Check if timer is still active
      if (0 == timerfd_gettime(idTFD, &iterValue))
      {
         isActive = (iterValue.it_value.tv_sec != 0) || (iterValue.it_value.tv_nsec != 0);
      }
   }
   return isActive;
}


// ------------------------------------------------------------------------
bool AnimatorTimer::isValid(const Int32 idTFD)
{
   CriticalSectionLocker lockData(&_criticalSection);
   if (!(_timerFDMap.empty()) && (_timerFDMap.find(idTFD) != _timerFDMap.end()))
   {
      ETG_TRACE_USR4_THR(("Valid Timer :%d", _timerFDMap.find(idTFD)->second));
      return true;
   }
   ETG_TRACE_USR4_THR(("InValid for Timer"));
   return false;
}


}
